/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.util.List;
import java.util.Locale;

import com.google.common.collect.Lists;

import net.minecraft.block.Block;
import net.minecraft.client.renderer.texture.TextureMap;
import net.minecraft.entity.item.*;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.launchwrapper.Launch;
import net.minecraft.stats.Achievement;
import net.minecraft.util.BlockPos;
import net.minecraft.util.ResourceLocation;
import net.minecraft.world.World;

import net.minecraftforge.client.event.TextureStitchEvent;
import net.minecraftforge.common.DimensionManager;
import net.minecraftforge.common.ForgeChunkManager;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.config.Property;
import net.minecraftforge.fml.client.event.ConfigChangedEvent;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.fml.common.event.*;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.network.NetworkRegistry;
import net.minecraftforge.fml.common.registry.EntityRegistry;
import net.minecraftforge.fml.common.registry.GameRegistry;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import buildcraft.api.blueprints.*;
import buildcraft.api.core.JavaTools;
import buildcraft.api.library.LibraryAPI;
import buildcraft.api.statements.StatementManager;
import buildcraft.builders.*;
import buildcraft.builders.blueprints.RealBlueprintDeployer;
import buildcraft.builders.schematics.*;
import buildcraft.builders.statements.BuildersActionProvider;
import buildcraft.core.*;
import buildcraft.core.blueprints.SchematicRegistry;
import buildcraft.core.builders.schematics.*;
import buildcraft.core.builders.schematics.SchematicBlockCreative;
import buildcraft.core.config.ConfigManager;
import buildcraft.core.network.EntityIds;

@Mod(name = "BuildCraft Builders", version = DefaultProps.VERSION, useMetadata = false, modid = "BuildCraft|Builders",
        dependencies = DefaultProps.DEPENDENCY_CORE)
public class BuildCraftBuilders extends BuildCraftMod {

    @Mod.Instance("BuildCraft|Builders")
    public static BuildCraftBuilders instance;

    public static BlockConstructionMarker constructionMarkerBlock;
    public static BlockFiller fillerBlock;
    public static BlockBuilder builderBlock;
    public static BlockArchitect architectBlock;
    public static BlockBlueprintLibrary libraryBlock;
    public static BlockQuarry quarryBlock;
    public static BlockFrame frameBlock;
    public static ItemBlueprintTemplate templateItem;
    public static ItemBlueprintStandard blueprintItem;

    public static Achievement architectAchievement;
    public static Achievement libraryAchievement;
    public static Achievement blueprintAchievement;
    public static Achievement builderAchievement;
    public static Achievement templateAchievement;
    public static Achievement chunkDestroyerAchievement;

    public static BlueprintServerDatabase serverDB;
    public static LibraryDatabase clientDB;

    public static boolean debugPrintSchematicList = false;
    public static boolean dropBrokenBlocks = false;

    public static boolean quarryLoadsChunks = true;
    public static boolean quarryOneTimeUse = false;

    private String oldBlueprintServerDir, blueprintClientDir;

    public class QuarryChunkloadCallback implements ForgeChunkManager.OrderedLoadingCallback {
        @Override
        public void ticketsLoaded(List<ForgeChunkManager.Ticket> tickets, World world) {
            for (ForgeChunkManager.Ticket ticket : tickets) {
                int quarryX = ticket.getModData().func_74762_e("quarryX");
                int quarryY = ticket.getModData().func_74762_e("quarryY");
                int quarryZ = ticket.getModData().func_74762_e("quarryZ");
                BlockPos pos = new BlockPos(quarryX, quarryY, quarryZ);

                Block block = world.func_180495_p(pos).func_177230_c();
                if (block == quarryBlock) {
                    TileQuarry tq = (TileQuarry) world.func_175625_s(pos);
                    tq.forceChunkLoading(ticket);
                }
            }
        }

        @Override
        public List<ForgeChunkManager.Ticket> ticketsLoaded(List<ForgeChunkManager.Ticket> tickets, World world, int maxTicketCount) {
            List<ForgeChunkManager.Ticket> validTickets = Lists.newArrayList();
            for (ForgeChunkManager.Ticket ticket : tickets) {
                int quarryX = ticket.getModData().func_74762_e("quarryX");
                int quarryY = ticket.getModData().func_74762_e("quarryY");
                int quarryZ = ticket.getModData().func_74762_e("quarryZ");
                BlockPos pos = new BlockPos(quarryX, quarryY, quarryZ);

                Block block = world.func_180495_p(pos).func_177230_c();
                if (block == quarryBlock) {
                    validTickets.add(ticket);
                }
            }
            return validTickets;
        }
    }

    @Mod.EventHandler
    public void loadConfiguration(FMLPreInitializationEvent evt) {
        BuildCraftCore.mainConfigManager.register("blueprints.serverDatabaseDirectory", "\"$MINECRAFT" + File.separator + "config" + File.separator
            + "buildcraft" + File.separator + "blueprints" + File.separator + "server\"", "DEPRECATED - USED ONLY FOR COMPATIBILITY",
                ConfigManager.RestartRequirement.GAME);
        BuildCraftCore.mainConfigManager.register("blueprints.clientDatabaseDirectory", "\"$MINECRAFT" + File.separator + "blueprints\"",
                "Location for the client blueprint database (used by the Electronic Library).", ConfigManager.RestartRequirement.NONE);

        BuildCraftCore.mainConfigManager.register("general.markerRange", 64, "Set the maximum marker range.", ConfigManager.RestartRequirement.NONE);
        BuildCraftCore.mainConfigManager.register("general.quarry.oneTimeUse", false, "Should the quarry only be usable once after placing?",
                ConfigManager.RestartRequirement.NONE);
        BuildCraftCore.mainConfigManager.register("general.quarry.doChunkLoading", true, "Should the quarry keep the chunks it is working on loaded?",
                ConfigManager.RestartRequirement.NONE);

        BuildCraftCore.mainConfigManager.register("builders.dropBrokenBlocks", false, "Should the builder and filler drop the cleared blocks?",
                ConfigManager.RestartRequirement.NONE);

        BuildCraftCore.mainConfigManager.get("blueprints.serverDatabaseDirectory").setShowInGui(false);
        BuildCraftCore.mainConfigManager.get("general.markerRange").setMinValue(8).setMaxValue(64);

        serverDB = new BlueprintServerDatabase();
        clientDB = new LibraryDatabase();

        reloadConfig(ConfigManager.RestartRequirement.GAME);

        Property printSchematicList = BuildCraftCore.mainConfiguration.get("debug", "printBlueprintSchematicList", false);
        debugPrintSchematicList = printSchematicList.getBoolean();
    }

    public void reloadConfig(ConfigManager.RestartRequirement restartType) {
        if (restartType == ConfigManager.RestartRequirement.GAME) {
            reloadConfig(ConfigManager.RestartRequirement.WORLD);
        } else if (restartType == ConfigManager.RestartRequirement.WORLD) {
            oldBlueprintServerDir = BuildCraftCore.mainConfigManager.get("blueprints.serverDatabaseDirectory").getString();
            oldBlueprintServerDir = JavaTools.stripSurroundingQuotes(replacePathVariables(oldBlueprintServerDir));

            reloadConfig(ConfigManager.RestartRequirement.NONE);
        } else {
            quarryOneTimeUse = BuildCraftCore.mainConfigManager.get("general.quarry.oneTimeUse").getBoolean();
            quarryLoadsChunks = BuildCraftCore.mainConfigManager.get("general.quarry.doChunkLoading").getBoolean();

            blueprintClientDir = BuildCraftCore.mainConfigManager.get("blueprints.clientDatabaseDirectory").getString();
            blueprintClientDir = JavaTools.stripSurroundingQuotes(replacePathVariables(blueprintClientDir));
            clientDB.init(new String[] { blueprintClientDir, getDownloadsDir() }, blueprintClientDir);

            DefaultProps.MARKER_RANGE = BuildCraftCore.mainConfigManager.get("general.markerRange").getInt();

            dropBrokenBlocks = BuildCraftCore.mainConfigManager.get("builders.dropBrokenBlocks").getBoolean();

            if (BuildCraftCore.mainConfiguration.hasChanged()) {
                BuildCraftCore.mainConfiguration.save();
            }
        }
    }

    @SubscribeEvent
    public void onConfigChanged(ConfigChangedEvent.PostConfigChangedEvent event) {
        if ("BuildCraftCore".equals(event.modID)) {
            reloadConfig(event.isWorldRunning ? ConfigManager.RestartRequirement.NONE : ConfigManager.RestartRequirement.WORLD);
        }
    }

    private static String getDownloadsDir() {
        final String os = System.getProperty("os.name").toLowerCase();

        if (os.contains("nix") || os.contains("lin") || os.contains("mac")) {
            // Linux, Mac or other UNIX
            // According XDG specification every user-specified folder can be localized
            // or even moved to any destination, so we obtain real path with xdg-user-dir
            try {
                Process process = Runtime.getRuntime().exec(new String[] { "xdg-user-dir", "DOWNLOAD" });
                BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream(), "utf-8"));
                process.waitFor();
                String line = reader.readLine().trim();
                reader.close();

                if (line.length() > 0) {
                    return line;
                }
            } catch (Exception ignored) {
                // Very bad, we have a error while obtaining xdg dir :(
                // Just ignore, uses default dir
            }
        }
        // Windows or unknown system
        return "$HOME" + File.separator + "Downloads";
    }

    private String replacePathVariables(String path) {
        String result = path.replace("$DOWNLOADS", getDownloadsDir());
        result = result.replace("$HOME", System.getProperty("user.home"));

        if (Launch.minecraftHome == null) {
            result = result.replace("$MINECRAFT", new File(".").getAbsolutePath());
        } else {
            result = result.replace("$MINECRAFT", Launch.minecraftHome.getAbsolutePath());
        }

        if ("/".equals(File.separator)) {
            result = result.replaceAll("\\\\", "/");
        } else {
            result = result.replaceAll("/", "\\\\");
        }

        return result;
    }

    @Mod.EventHandler
    public void preInit(FMLPreInitializationEvent evt) {
        templateItem = new ItemBlueprintTemplate();
        templateItem.func_77655_b("templateItem");
        BCRegistry.INSTANCE.registerItem(templateItem, false);

        blueprintItem = new ItemBlueprintStandard();
        blueprintItem.func_77655_b("blueprintItem");
        BCRegistry.INSTANCE.registerItem(blueprintItem, false);

        quarryBlock = (BlockQuarry) CompatHooks.INSTANCE.getBlock(BlockQuarry.class);
        BCRegistry.INSTANCE.registerBlock(quarryBlock.func_149663_c("quarryBlock"), false);

        fillerBlock = (BlockFiller) CompatHooks.INSTANCE.getBlock(BlockFiller.class);
        BCRegistry.INSTANCE.registerBlock(fillerBlock.func_149663_c("fillerBlock"), false);

        frameBlock = new BlockFrame();
        BCRegistry.INSTANCE.registerBlock(frameBlock.func_149663_c("frameBlock"), true);

        builderBlock = (BlockBuilder) CompatHooks.INSTANCE.getBlock(BlockBuilder.class);
        BCRegistry.INSTANCE.registerBlock(builderBlock.func_149663_c("builderBlock"), false);

        architectBlock = (BlockArchitect) CompatHooks.INSTANCE.getBlock(BlockArchitect.class);
        BCRegistry.INSTANCE.registerBlock(architectBlock.func_149663_c("architectBlock"), false);

        libraryBlock = (BlockBlueprintLibrary) CompatHooks.INSTANCE.getBlock(BlockBlueprintLibrary.class);
        BCRegistry.INSTANCE.registerBlock(libraryBlock.func_149663_c("libraryBlock"), false);

        BCRegistry.INSTANCE.registerTileEntity(TileQuarry.class, "buildcraft.builders.Quarry", "Machine");
        BCRegistry.INSTANCE.registerTileEntity(TileMarker.class, "buildcraft.builders.Marker", "Marker");
        BCRegistry.INSTANCE.registerTileEntity(TileFiller.class, "buildcraft.builders.Filler", "Filler");
        BCRegistry.INSTANCE.registerTileEntity(TileBuilder.class, "buildcraft.builders.Builder", "net.minecraft.src.builders.TileBuilder");
        BCRegistry.INSTANCE.registerTileEntity(TileArchitect.class, "buildcraft.builders.Architect", "net.minecraft.src.builders.TileTemplate");
        BCRegistry.INSTANCE.registerTileEntity(TilePathMarker.class, "buildcraft.builders.PathMarker", "net.minecraft.src.builders.TilePathMarker");
        BCRegistry.INSTANCE.registerTileEntity(TileBlueprintLibrary.class, "buildcraft.builders.BlueprintLibrary",
                "net.minecraft.src.builders.TileBlueprintLibrary");

        constructionMarkerBlock = (BlockConstructionMarker) CompatHooks.INSTANCE.getBlock(BlockConstructionMarker.class);
        BCRegistry.INSTANCE.registerBlock(constructionMarkerBlock.func_149663_c("constructionMarkerBlock"), ItemConstructionMarker.class, false);

        BCRegistry.INSTANCE.registerTileEntity(TileConstructionMarker.class, "buildcraft.builders.ConstructionMarker",
                "net.minecraft.src.builders.TileConstructionMarker");

        SchematicRegistry.INSTANCE.readConfiguration(BuildCraftCore.mainConfiguration);

        if (BuildCraftCore.mainConfiguration.hasChanged()) {
            BuildCraftCore.mainConfiguration.save();
        }

        MinecraftForge.EVENT_BUS.register(this);

        StatementManager.registerActionProvider(new BuildersActionProvider());
    }

    @Mod.EventHandler
    public void init(FMLInitializationEvent evt) {
        NetworkRegistry.INSTANCE.registerGuiHandler(instance, new BuildersGuiHandler());

        MinecraftForge.EVENT_BUS.register(new BuilderTooltipHandler());
        EntityRegistry.registerModEntity(EntityMechanicalArm.class, "bcMechArm", EntityIds.MECHANICAL_ARM, instance, 50, 1, true);

        // Standard blocks
        ISchematicRegistry schemes = BuilderAPI.schematicRegistry;
        schemes.registerSchematicBlock(Blocks.field_150350_a, SchematicAir.class);

        schemes.registerSchematicBlock(Blocks.field_150433_aE, SchematicIgnore.class);
        schemes.registerSchematicBlock(Blocks.field_150329_H, SchematicIgnore.class);
        schemes.registerSchematicBlock(Blocks.field_150398_cm, SchematicIgnore.class);
        schemes.registerSchematicBlock(Blocks.field_150432_aD, SchematicIgnore.class);
        schemes.registerSchematicBlock(Blocks.field_150332_K, SchematicIgnore.class);

        schemes.registerSchematicBlock(Blocks.field_150346_d, SchematicDirt.class);
        schemes.registerSchematicBlock(Blocks.field_150349_c, SchematicDirt.class);

        schemes.registerSchematicBlock(Blocks.field_150434_aF, SchematicCactus.class);

        schemes.registerSchematicBlock(Blocks.field_150458_ak, SchematicFarmland.class);
        schemes.registerSchematicBlock(Blocks.field_150464_aj, SchematicSeeds.class, Items.field_151014_N);
        schemes.registerSchematicBlock(Blocks.field_150393_bb, SchematicSeeds.class, Items.field_151080_bb);
        schemes.registerSchematicBlock(Blocks.field_150394_bc, SchematicSeeds.class, Items.field_151081_bc);
        schemes.registerSchematicBlock(Blocks.field_150388_bm, SchematicSeeds.class, Items.field_151075_bm);

        schemes.registerSchematicBlock(Blocks.field_150478_aa, SchematicBlock.class);
        schemes.registerSchematicBlock(Blocks.field_150429_aA, SchematicBlock.class);
        schemes.registerSchematicBlock(Blocks.field_150437_az, SchematicBlock.class);

        schemes.registerSchematicBlock(Blocks.field_150479_bC, SchematicBlock.class);
        schemes.registerSchematicBlock(Blocks.field_150473_bD, SchematicTripwire.class);

        schemes.registerSchematicBlock(Blocks.field_150465_bP, SchematicSkull.class);

        schemes.registerSchematicBlock(Blocks.field_150364_r, SchematicLog.class);
        schemes.registerSchematicBlock(Blocks.field_150363_s, SchematicLog.class);
        schemes.registerSchematicBlock(Blocks.field_150407_cf, SchematicRotatedPillar.class);
        schemes.registerSchematicBlock(Blocks.field_150371_ca, SchematicQuartz.class);
        schemes.registerSchematicBlock(Blocks.field_150438_bZ, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150467_bQ, SchematicCustomStack.class, new ItemStack(Blocks.field_150467_bQ));

        schemes.registerSchematicBlock(Blocks.field_150395_bd, SchematicVine.class);

        schemes.registerSchematicBlock(Blocks.field_150460_al, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150470_am, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150486_ae, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150367_z, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150409_cd, SchematicTile.class);

        schemes.registerSchematicBlock(Blocks.field_150477_bB, SchematicEnderChest.class);

        schemes.registerSchematicBlock(Blocks.field_150442_at, SchematicLever.class);

        schemes.registerSchematicBlock(Blocks.field_150352_o, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150366_p, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150365_q, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150369_x, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150482_ag, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150450_ax, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150439_ay, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());
        schemes.registerSchematicBlock(Blocks.field_150412_bA, SchematicTreatAsOther.class, Blocks.field_150348_b.func_176223_P());

        schemes.registerSchematicBlock(Blocks.field_150351_n, SchematicGravel.class);

        schemes.registerSchematicBlock(Blocks.field_150488_af, SchematicRedstoneWire.class, new ItemStack(Items.field_151137_ax));
        schemes.registerSchematicBlock(Blocks.field_150414_aQ, SchematicCustomStack.class, new ItemStack(Items.field_151105_aU));
        schemes.registerSchematicBlock(Blocks.field_150426_aN, SchematicCustomStack.class, new ItemStack(Blocks.field_150426_aN));

        schemes.registerSchematicBlock(Blocks.field_150416_aS, SchematicCustomStackFloored.class, new ItemStack(Items.field_151107_aW));
        schemes.registerSchematicBlock(Blocks.field_150413_aR, SchematicCustomStackFloored.class, new ItemStack(Items.field_151107_aW));
        schemes.registerSchematicBlock(Blocks.field_150455_bV, SchematicCustomStackFloored.class, new ItemStack(Items.field_151132_bS));
        schemes.registerSchematicBlock(Blocks.field_150441_bU, SchematicCustomStackFloored.class, new ItemStack(Items.field_151132_bS));

        schemes.registerSchematicBlock(Blocks.field_150453_bW, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_180402_cm, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150421_aI, SchematicJukebox.class);
        schemes.registerSchematicBlock(Blocks.field_150323_B, SchematicTile.class);

        schemes.registerSchematicBlock(Blocks.field_150379_bu, SchematicRedstoneLamp.class);
        schemes.registerSchematicBlock(Blocks.field_150374_bv, SchematicRedstoneLamp.class);

        schemes.registerSchematicBlock(Blocks.field_150410_aZ, SchematicGlassPane.class);
        schemes.registerSchematicBlock(Blocks.field_150397_co, SchematicGlassPane.class);

        schemes.registerSchematicBlock(Blocks.field_150331_J, SchematicPiston.class);
        schemes.registerSchematicBlock(Blocks.field_180384_M, SchematicIgnore.class);
        schemes.registerSchematicBlock(Blocks.field_150320_F, SchematicPiston.class);

        schemes.registerSchematicBlock(Blocks.field_150476_ad, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150446_ar, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150389_bf, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150390_bg, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150387_bl, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150372_bz, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150485_bF, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150487_bG, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150481_bH, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150370_cb, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150400_ck, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150401_cl, SchematicStandalone.class);

        schemes.registerSchematicBlock(Blocks.field_180410_as, SchematicDoor.class, new ItemStack(Items.field_179572_au));
        schemes.registerSchematicBlock(Blocks.field_180412_aq, SchematicDoor.class, new ItemStack(Items.field_179568_as));
        schemes.registerSchematicBlock(Blocks.field_180409_at, SchematicDoor.class, new ItemStack(Items.field_179571_av));
        schemes.registerSchematicBlock(Blocks.field_180411_ar, SchematicDoor.class, new ItemStack(Items.field_179567_at));
        schemes.registerSchematicBlock(Blocks.field_180413_ao, SchematicDoor.class, new ItemStack(Items.field_179570_aq));
        schemes.registerSchematicBlock(Blocks.field_180414_ap, SchematicDoor.class, new ItemStack(Items.field_179569_ar));
        schemes.registerSchematicBlock(Blocks.field_150454_av, SchematicDoor.class, new ItemStack(Items.field_151139_aw));

        schemes.registerSchematicBlock(Blocks.field_150324_C, SchematicBed.class);

        schemes.registerSchematicBlock(Blocks.field_150444_as, SchematicSignLike.class, true);
        schemes.registerSchematicBlock(Blocks.field_150472_an, SchematicSignLike.class, false);

        schemes.registerSchematicBlock(Blocks.field_180394_cL, SchematicBanner.class, true);
        schemes.registerSchematicBlock(Blocks.field_180393_cK, SchematicBanner.class, false);

        schemes.registerSchematicBlock(Blocks.field_150427_aO, SchematicPortal.class);

        schemes.registerSchematicBlock(Blocks.field_150448_aq, SchematicRail.class);
        schemes.registerSchematicBlock(Blocks.field_150408_cc, SchematicRail.class);
        schemes.registerSchematicBlock(Blocks.field_150319_E, SchematicRail.class);
        schemes.registerSchematicBlock(Blocks.field_150318_D, SchematicRail.class);

        schemes.registerSchematicBlock(Blocks.field_150461_bJ, SchematicTile.class);
        schemes.registerSchematicBlock(Blocks.field_150382_bo, SchematicBrewingStand.class);
        schemes.registerSchematicBlock(Blocks.field_150381_bn, SchematicTile.class);

        schemes.registerSchematicBlock(Blocks.field_150480_ab, SchematicFire.class);

        schemes.registerSchematicBlock(Blocks.field_150357_h, SchematicBlockCreative.class);

        schemes.registerSchematicBlock(Blocks.field_150483_bI, SchematicTileCreative.class);
        schemes.registerSchematicBlock(Blocks.field_150474_ac, SchematicTileCreative.class);

        schemes.registerSchematicBlock(Blocks.field_150359_w, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150333_U, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150334_T, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150376_bx, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150373_bw, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150399_cn, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_180405_aT, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_180404_aQ, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_180406_aS, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_180403_aR, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_180407_aO, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_180408_aP, SchematicStandalone.class);
        schemes.registerSchematicBlock(Blocks.field_150411_aY, SchematicStandalone.class);

        // Standard entities

        schemes.registerSchematicEntity(EntityArmorStand.class, SchematicArmorStand.class);

        schemes.registerSchematicEntity(EntityMinecartEmpty.class, SchematicMinecart.class, Items.field_151143_au);
        schemes.registerSchematicEntity(EntityMinecartFurnace.class, SchematicMinecart.class, Items.field_151109_aJ);
        schemes.registerSchematicEntity(EntityMinecartTNT.class, SchematicMinecart.class, Items.field_151142_bV);
        schemes.registerSchematicEntity(EntityMinecartChest.class, SchematicMinecart.class, Items.field_151108_aI);
        schemes.registerSchematicEntity(EntityMinecartHopper.class, SchematicMinecart.class, Items.field_151140_bW);

        schemes.registerSchematicEntity(EntityPainting.class, SchematicHanging.class, Items.field_151159_an);
        schemes.registerSchematicEntity(EntityItemFrame.class, SchematicHanging.class, Items.field_151160_bD);

        // BuildCraft blocks

        schemes.registerSchematicBlock(architectBlock, SchematicTile.class);
        schemes.registerSchematicBlock(builderBlock, SchematicBuilderLike.class);
        schemes.registerSchematicBlock(fillerBlock, SchematicBuilderLike.class);
        schemes.registerSchematicBlock(libraryBlock, SchematicTile.class);
        schemes.registerSchematicBlock(quarryBlock, SchematicBuilderLike.class);

        // schemes.registerSchematicBlock(markerBlock, SchematicWallSide.class);
        // schemes.registerSchematicBlock(pathMarkerBlock, SchematicWallSide.class);
        // schemes.registerSchematicBlock(constructionMarkerBlock, SchematicWallSide.class);

        // Factories required to save entities in world

        SchematicFactory.registerSchematicFactory(SchematicBlock.class, new SchematicFactoryBlock());
        SchematicFactory.registerSchematicFactory(SchematicMask.class, new SchematicFactoryMask());
        SchematicFactory.registerSchematicFactory(SchematicEntity.class, new SchematicFactoryEntity());

        LibraryAPI.registerHandler(new LibraryBlueprintTypeHandler(false)); // Template
        LibraryAPI.registerHandler(new LibraryBlueprintTypeHandler(true)); // Blueprint
        LibraryAPI.registerHandler(new LibraryBookTypeHandler());

        BlueprintDeployer.instance = new RealBlueprintDeployer();

        architectAchievement = BuildCraftCore.achievementManager.registerAchievement(new Achievement("buildcraft|builders:achievement.architect",
                "architectAchievement", 11, 2, BuildCraftBuilders.architectBlock, BuildCraftCore.goldGearAchievement));
        builderAchievement = BuildCraftCore.achievementManager.registerAchievement(new Achievement("buildcraft|builders:achievement.builder",
                "builderAchievement", 13, 2, BuildCraftBuilders.builderBlock, architectAchievement));
        blueprintAchievement = BuildCraftCore.achievementManager.registerAchievement(new Achievement("buildcraft|builders:achievement.blueprint",
                "blueprintAchievement", 11, 4, BuildCraftBuilders.blueprintItem, architectAchievement));
        templateAchievement = BuildCraftCore.achievementManager.registerAchievement(new Achievement("buildcraft|builders:achievement.template",
                "templateAchievement", 13, 4, BuildCraftBuilders.templateItem, blueprintAchievement));
        libraryAchievement = BuildCraftCore.achievementManager.registerAchievement(new Achievement("buildcraft|builders:achievement.blueprintLibrary",
                "blueprintLibraryAchievement", 15, 2, BuildCraftBuilders.libraryBlock, builderAchievement));
        chunkDestroyerAchievement = BuildCraftCore.achievementManager.registerAchievement(new Achievement(
                "buildcraft|builders:achievement.chunkDestroyer", "chunkDestroyerAchievement", 9, 2, quarryBlock,
                BuildCraftCore.diamondGearAchievement));

        if (BuildCraftCore.loadDefaultRecipes) {
            loadRecipes();
        }

        BuilderProxy.proxy.registerBlockRenderers();
    }

    @Mod.EventHandler
    public void postInit(FMLPostInitializationEvent evt) {
        HeuristicBlockDetection.start();
        ForgeChunkManager.setForcedChunkLoadingCallback(instance, new QuarryChunkloadCallback());

        if (debugPrintSchematicList) {
            try {
                PrintWriter writer = new PrintWriter("SchematicDebug.txt", "UTF-8");
                writer.println("*** REGISTERED SCHEMATICS ***");
                SchematicRegistry reg = (SchematicRegistry) BuilderAPI.schematicRegistry;
                for (String s : reg.schematicBlocks.keySet()) {
                    writer.println(s + " -> " + reg.schematicBlocks.get(s).clazz.getCanonicalName());
                }
                writer.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        // Refresh the client database once all the library type handlers are registered
        // The server database is refreshed later
        clientDB.refresh();
    }

    public static void loadRecipes() {
        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(quarryBlock), "ipi", "gig", "dDd", 'i', "gearIron", 'p', "dustRedstone", 'g', "gearGold",
                'd', "gearDiamond", 'D', Items.field_151046_w);

        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(templateItem, 1), "ppp", "pip", "ppp", 'i', "dyeBlack", 'p', Items.field_151121_aF);

        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(blueprintItem, 1), "ppp", "pip", "ppp", 'i', "gemLapis", 'p', Items.field_151121_aF);

        if (constructionMarkerBlock != null) {
            BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(constructionMarkerBlock, 1), "l ", "r ", 'l', "gearGold", 'r', Blocks.field_150429_aA);
        }

        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(fillerBlock, 1), "btb", "ycy", "gCg", 'b', "dyeBlack", 't', BuildCraftCore.markerBlock,
                'y', "dyeYellow", 'c', Blocks.field_150462_ai, 'g', "gearGold", 'C', Blocks.field_150486_ae);

        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(builderBlock, 1), "btb", "ycy", "gCg", 'b', "dyeBlack", 't', BuildCraftCore.markerBlock,
                'y', "dyeYellow", 'c', Blocks.field_150462_ai, 'g', "gearDiamond", 'C', Blocks.field_150486_ae);

        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(architectBlock, 1), "btb", "ycy", "gCg", 'b', "dyeBlack", 't', BuildCraftCore.markerBlock,
                'y', "dyeYellow", 'c', Blocks.field_150462_ai, 'g', "gearDiamond", 'C', new ItemStack(blueprintItem, 1));

        BCRegistry.INSTANCE.addCraftingRecipe(new ItemStack(libraryBlock, 1), "igi", "bBb", "iri", 'B', new ItemStack(blueprintItem), 'b',
                Blocks.field_150342_X, 'i', "ingotIron", 'g', "gearIron", 'r', Items.field_151137_ax);
    }

    @Mod.EventHandler
    public void processIMCRequests(FMLInterModComms.IMCEvent event) {
        InterModComms.processIMC(event);
    }

    @Mod.EventHandler
    public void serverStop(FMLServerStoppingEvent event) {
        TilePathMarker.clearAvailableMarkersList();
    }

    @Mod.EventHandler
    public void serverAboutToStart(FMLServerAboutToStartEvent event) {
        String blueprintPath = new File(DimensionManager.getCurrentSaveRootDirectory(), "buildcraft" + File.separator + "blueprints").getPath();
        serverDB.init(new String[] { oldBlueprintServerDir, blueprintPath }, blueprintPath);
    }

    @SubscribeEvent
    @SideOnly(Side.CLIENT)
    public void loadTextures(TextureStitchEvent.Pre evt) {
        TextureMap terrainTextures = evt.map;
        BuilderProxyClient.drillTexture = terrainTextures.func_174942_a(new ResourceLocation("buildcraftbuilders:blocks/quarry/drill"));
        BuilderProxyClient.drillHeadTexture = terrainTextures.func_174942_a(new ResourceLocation("buildcraftbuilders:blocks/quarry/drill_head"));
    }

    @Mod.EventHandler
    public void whiteListAppliedEnergetics(FMLInitializationEvent event) {
        FMLInterModComms.sendMessage("appliedenergistics2", "whitelist-spatial", TileBlueprintLibrary.class.getCanonicalName());
    }

    @Mod.EventHandler
    public void remap(FMLMissingMappingsEvent event) {
        for (FMLMissingMappingsEvent.MissingMapping mapping : event.getAll()) {
            if (mapping.name.equals("BuildCraft|Builders:buildToolBlock") || mapping.name.equals("BuildCraft|Builders:null")) {
                if (mapping.type == GameRegistry.Type.ITEM) {
                    mapping.remap(Item.func_150898_a(BuildCraftCore.decoratedBlock));
                } else {
                    mapping.remap(BuildCraftCore.decoratedBlock);
                }
            } else if (mapping.name.equals("BuildCraft|Builders:markerBlock")) {
                if (mapping.type == GameRegistry.Type.ITEM) {
                    mapping.remap(Item.func_150898_a(BuildCraftCore.markerBlock));
                } else {
                    mapping.remap(BuildCraftCore.markerBlock);
                }
            } else if (mapping.name.equals("BuildCraft|Builders:pathMarkerBlock")) {
                if (mapping.type == GameRegistry.Type.ITEM) {
                    mapping.remap(Item.func_150898_a(BuildCraftCore.pathMarkerBlock));
                } else {
                    mapping.remap(BuildCraftCore.pathMarkerBlock);
                }
            } else if (mapping.name.toLowerCase(Locale.ROOT).equals("buildcraft|builders:machineblock")) {
                if (mapping.type == GameRegistry.Type.ITEM) {
                    mapping.remap(Item.func_150898_a(quarryBlock));
                } else {
                    mapping.remap(quarryBlock);
                }
            }
        }
    }
}
