/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.builders;

import java.util.ArrayList;
import java.util.LinkedList;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.BlockPos;
import net.minecraft.util.Vec3;

import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.IAreaProvider;
import buildcraft.builders.blueprints.RecursiveBlueprintReader;
import buildcraft.core.Box;
import buildcraft.core.Box.Kind;
import buildcraft.core.LaserData;
import buildcraft.core.blueprints.BlueprintReadConfiguration;
import buildcraft.core.internal.IBoxProvider;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.inventory.SimpleInventory;
import buildcraft.core.lib.network.base.Packet;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.ICommandReceiver;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class TileArchitect extends TileBuildCraft implements IInventory, IBoxProvider, ICommandReceiver {
    public enum Mode {
        NONE,
        EDIT,
        COPY
    }

    public String currentAuthorName = "";
    public Mode mode = Mode.NONE;

    public Box box = new Box();
    public String name = "";
    public BlueprintReadConfiguration readConfiguration = new BlueprintReadConfiguration();

    public LinkedList<LaserData> subLasers = new LinkedList<>();

    public ArrayList<BlockPos> subBlueprints = new ArrayList<>();

    private SimpleInventory inv = new SimpleInventory(2, "Architect", 1);
    private RecursiveBlueprintReader reader;
    private boolean clientIsWorking, initialized;

    public TileArchitect() {
        box.kind = Kind.BLUE_STRIPES;
    }

    public void storeBlueprintStack(ItemStack blueprintStack) {
        func_70299_a(1, blueprintStack);
        func_70298_a(0, 1);
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (!field_145850_b.field_72995_K) {
            if (mode == Mode.COPY && reader != null) {
                reader.iterate();

                if (reader.isDone()) {
                    reader = null;
                    sendNetworkUpdate();
                }
            }
        }
    }

    @Override
    public void initialize() {
        super.initialize();

        if (!field_145850_b.field_72995_K && !initialized) {
            if (!box.isInitialized()) {
                IAreaProvider a = Utils.getNearbyAreaProvider(field_145850_b, field_174879_c);

                if (a != null) {
                    mode = Mode.COPY;
                    box.initialize(a);
                    a.removeFromWorld();
                    sendNetworkUpdate();
                    return;
                } else {
                    if (BuildCraftCore.DEVELOPER_MODE) {
                        mode = Mode.EDIT;
                    } else {
                        mode = Mode.NONE;
                    }
                }
            } else {
                mode = Mode.COPY;
            }
            initialized = true;
            sendNetworkUpdate();
        }
    }

    @Override
    public int func_70302_i_() {
        return 2;
    }

    @Override
    public ItemStack func_70301_a(int i) {
        return inv.func_70301_a(i);
    }

    @Override
    public ItemStack func_70298_a(int i, int j) {
        ItemStack result = inv.func_70298_a(i, j);

        if (i == 0) {
            initializeBlueprint();
        }

        return result;
    }

    @Override
    public void func_70299_a(int i, ItemStack itemstack) {
        inv.func_70299_a(i, itemstack);

        if (i == 0) {
            initializeBlueprint();
        }
    }

    @Override
    public ItemStack func_70304_b(int slot) {
        return inv.func_70304_b(slot);
    }

    @Override
    public int func_70297_j_() {
        return 1;
    }

    @Override
    public boolean func_70300_a(EntityPlayer entityplayer) {
        return mode != Mode.NONE && field_145850_b.func_175625_s(field_174879_c) == this;
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);

        if (nbt.func_74764_b("box")) {
            box.initialize(nbt.func_74775_l("box"));
        }

        inv.readFromNBT(nbt);

        mode = Mode.values()[nbt.func_74771_c("mode")];
        name = nbt.func_74779_i("name");
        currentAuthorName = nbt.func_74779_i("lastAuthor");

        if (nbt.func_74764_b("readConfiguration")) {
            readConfiguration.readFromNBT(nbt.func_74775_l("readConfiguration"));
        }

        NBTTagList subBptList = nbt.func_150295_c("subBlueprints", Constants.NBT.TAG_COMPOUND);

        for (int i = 0; i < subBptList.func_74745_c(); ++i) {
            BlockPos index = NBTUtils.readBlockPos(subBptList.func_150305_b(i));

            addSubBlueprint(index);
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);

        if (box.isInitialized()) {
            NBTTagCompound boxStore = new NBTTagCompound();
            box.writeToNBT(boxStore);
            nbt.func_74782_a("box", boxStore);
        }

        inv.writeToNBT(nbt);

        nbt.func_74774_a("mode", (byte) mode.ordinal());
        nbt.func_74778_a("name", name);
        nbt.func_74778_a("lastAuthor", currentAuthorName);

        NBTTagCompound readConf = new NBTTagCompound();
        readConfiguration.writeToNBT(readConf);
        nbt.func_74782_a("readConfiguration", readConf);

        NBTTagList subBptList = new NBTTagList();

        for (BlockPos b : subBlueprints) {
            subBptList.func_74742_a(NBTUtils.writeBlockPos(b));
        }

        nbt.func_74782_a("subBlueprints", subBptList);
    }

    private boolean getIsWorking() {
        return mode == Mode.COPY ? reader != null : false;
    }

    @Override
    public void writeData(ByteBuf stream) {
        box.writeData(stream);
        NetworkUtils.writeUTF(stream, name);
        stream.writeBoolean(getIsWorking());
        stream.writeByte(mode.ordinal());
        if (mode == Mode.COPY) {
            readConfiguration.writeData(stream);
            stream.writeShort(subLasers.size());
            for (LaserData ld : subLasers) {
                ld.writeData(stream);
            }
        }
    }

    @Override
    public void readData(ByteBuf stream) {
        box.readData(stream);
        name = NetworkUtils.readUTF(stream);
        clientIsWorking = stream.readBoolean();
        mode = Mode.values()[stream.readByte()];

        if (mode == Mode.COPY) {
            readConfiguration.readData(stream);
            int size = stream.readUnsignedShort();
            subLasers.clear();
            for (int i = 0; i < size; i++) {
                LaserData ld = new LaserData();
                ld.readData(stream);
                subLasers.add(ld);
            }
        }
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        destroy();
    }

    private void initializeBlueprint() {
        if (func_145831_w().field_72995_K) {
            return;
        }

        if (mode == Mode.COPY) {
            reader = new RecursiveBlueprintReader(this);
        }
        sendNetworkUpdate();
    }

    public int getComputingProgressScaled(int scale) {
        if (reader != null) {
            return (int) (reader.getComputingProgressScaled() * scale);
        } else {
            return 0;
        }
    }

    @Override
    public void func_174889_b(EntityPlayer player) {}

    @Override
    public void func_174886_c(EntityPlayer player) {}

    @Override
    public boolean hasCustomName() {
        return true;
    }

    @Override
    public boolean func_94041_b(int var1, ItemStack var2) {
        return false;
    }

    @Override
    public Box getBox() {
        return box;
    }

    @Override
    public AxisAlignedBB getRenderBoundingBox() {
        Box completeBox = new Box(this).extendToEncompass(box);

        for (LaserData d : subLasers) {
            completeBox.extendToEncompass(d.tail);
        }

        return completeBox.getBoundingBox();
    }

    public Packet getPacketSetName() {
        return new PacketCommand(this, "setName", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                NetworkUtils.writeUTF(data, name);
            }
        });
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        if ("setName".equals(command)) {
            this.name = NetworkUtils.readUTF(stream);
            if (side.isServer()) {
                BuildCraftCore.instance.sendToPlayersNear(getPacketSetName(), this);
            }
        } else if (side.isServer()) {
            if ("setReadConfiguration".equals(command)) {
                readConfiguration.readData(stream);
                sendNetworkUpdate();
            }
        }
    }

    public void rpcSetConfiguration(BlueprintReadConfiguration conf) {
        readConfiguration = conf;

        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "setReadConfiguration", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                readConfiguration.writeData(data);
            }
        }));
    }

    public void addSubBlueprint(TileEntity sub) {
        if (mode == Mode.COPY) {
            addSubBlueprint(sub.func_174877_v());
            sendNetworkUpdate();
        }
    }

    private void addSubBlueprint(BlockPos index) {
        subBlueprints.add(index);

        Vec3 point5 = new Vec3(0.5, 0.5, 0.5);

        LaserData laser = new LaserData(Utils.convert(index).func_178787_e(point5), Utils.convert(this.func_174877_v()).func_178787_e(point5));

        subLasers.add(laser);
    }

    @Override
    public String getInventoryName() {
        return "Template";
    }

    @Override
    @SideOnly(Side.CLIENT)
    public double func_145833_n() {
        return Double.MAX_VALUE;
    }
}
