/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.builders;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.List;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.CompressedStreamTools;
import net.minecraft.nbt.NBTTagCompound;

import net.minecraftforge.fml.relauncher.Side;

import buildcraft.BuildCraftBuilders;
import buildcraft.BuildCraftCore;
import buildcraft.api.library.LibraryAPI;
import buildcraft.api.library.LibraryTypeHandler;
import buildcraft.api.library.LibraryTypeHandlerByteArray;
import buildcraft.api.library.LibraryTypeHandlerNBT;
import buildcraft.core.blueprints.LibraryId;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.inventory.SimpleInventory;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.ICommandReceiver;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;

import io.netty.buffer.ByteBuf;

/** In this implementation, the blueprint library is the interface to the *local* player blueprint. The player will be
 * able to load blueprint on his environment, and save blueprints to the server environment. */
public class TileBlueprintLibrary extends TileBuildCraft implements IInventory, ICommandReceiver {
    private static final int PROGRESS_TIME = 100;
    private static final int CHUNK_SIZE = 16384;

    public SimpleInventory inv = new SimpleInventory(4, "Electronic Library", 1);

    public int progressIn = 0;
    public int progressOut = 0;

    public List<LibraryId> entries;

    public int selected = -1;

    public EntityPlayer uploadingPlayer = null;
    public EntityPlayer downloadingPlayer = null;

    private LibraryId blueprintDownloadId;
    private byte[] blueprintDownload;

    public TileBlueprintLibrary() {

    }

    public void refresh() {
        if (field_145850_b.field_72995_K) {
            BuildCraftBuilders.clientDB.refresh();
            entries = BuildCraftBuilders.clientDB.getBlueprintIds();
            selected = -1;
        }
    }

    @Override
    public void initialize() {
        super.initialize();

        refresh();
    }

    public void deleteSelectedBpt() {
        if (selected != -1) {
            BuildCraftBuilders.clientDB.deleteBlueprint(entries.get(selected));
            entries = BuildCraftBuilders.clientDB.getBlueprintIds();
            if (selected >= entries.size()) {
                selected--;
            }
        }
    }

    @Override
    public void func_145839_a(NBTTagCompound nbttagcompound) {
        super.func_145839_a(nbttagcompound);

        inv.readFromNBT(nbttagcompound);
    }

    @Override
    public void func_145841_b(NBTTagCompound nbttagcompound) {
        super.func_145841_b(nbttagcompound);

        inv.writeToNBT(nbttagcompound);
    }

    @Override
    public int func_70302_i_() {
        return 4;
    }

    @Override
    public ItemStack func_70301_a(int i) {
        return inv.func_70301_a(i);
    }

    @Override
    public ItemStack func_70298_a(int i, int j) {
        ItemStack result = inv.func_70298_a(i, j);

        if (i == 0) {
            if (func_70301_a(0) == null) {
                progressIn = 0;
            }
        }

        if (i == 2) {
            if (func_70301_a(2) == null) {
                progressOut = 0;
            }
        }

        return result;
    }

    @Override
    public void func_70299_a(int i, ItemStack itemstack) {
        inv.func_70299_a(i, itemstack);

        if (i == 0) {
            if (func_70301_a(0) != null && findHandler(0, LibraryTypeHandler.HandlerType.STORE) != null) {
                progressIn = 1;
            } else {
                progressIn = 0;
            }
        }

        if (i == 2) {
            if (func_70301_a(2) != null && findHandler(2, LibraryTypeHandler.HandlerType.LOAD) != null) {
                progressOut = 1;
            } else {
                progressOut = 0;
            }
        }
    }

    @Override
    public ItemStack func_70304_b(int slot) {
        return inv.func_70304_b(slot);
    }

    @Override
    public String getInventoryName() {
        return "";
    }

    @Override
    public int func_70297_j_() {
        return 1;
    }

    @Override
    public boolean func_94041_b(int i, ItemStack itemstack) {
        return false;
    }

    @Override
    public boolean func_70300_a(EntityPlayer entityplayer) {
        return field_145850_b.func_175625_s(field_174879_c) == this;
    }

    @Override
    public void func_174889_b(EntityPlayer player) {}

    @Override
    public void func_174886_c(EntityPlayer player) {}

    private LibraryTypeHandler findHandler(int slot, LibraryTypeHandler.HandlerType type) {
        if (!field_145850_b.field_72995_K) {
            ItemStack stack = func_70301_a(slot);

            for (LibraryTypeHandler h : LibraryAPI.getHandlerSet()) {
                if (h.isHandler(stack, type)) {
                    return h;
                }
            }
        }

        return null;
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (progressIn > 0 && progressIn < PROGRESS_TIME) {
            progressIn++;
        }

        if (progressOut > 0 && progressOut < PROGRESS_TIME) {
            if (selected != -1) {
                progressOut++;
            } else {
                progressOut = 1;
            }
        }

        // On progress IN, we'll download the blueprint from the server to the
        // client, and then store it to the client.
        if (progressIn == 100 && func_70301_a(1) == null) {
            LibraryTypeHandler handler = findHandler(0, LibraryTypeHandler.HandlerType.STORE);

            if (handler == null) {
                uploadingPlayer = null;
                return;
            }

            byte[] data = null;

            if (handler instanceof LibraryTypeHandlerNBT) {
                final NBTTagCompound nbt = new NBTTagCompound();
                if (((LibraryTypeHandlerNBT) handler).store(func_70301_a(0), nbt)) {
                    data = NBTUtils.save(nbt);
                }
            } else if (handler instanceof LibraryTypeHandlerByteArray) {
                data = ((LibraryTypeHandlerByteArray) handler).store(func_70301_a(0));
            }

            if (data == null) {
                uploadingPlayer = null;
                return;
            }

            func_70299_a(1, func_70301_a(0));
            func_70299_a(0, null);

            final byte[] dataOut = data;
            final LibraryId id = new LibraryId();
            id.name = handler.getName(func_70301_a(1));
            id.extension = handler.getOutputExtension();

            if (uploadingPlayer != null) {
                BuildCraftCore.instance.sendToPlayer(uploadingPlayer, new PacketCommand(this, "downloadBlueprintToClient", new CommandWriter() {
                    @Override
                    public void write(ByteBuf data) {
                        id.generateUniqueId(dataOut);
                        id.writeData(data);
                        NetworkUtils.writeByteArray(data, dataOut);
                    }
                }));
                uploadingPlayer = null;
            }
        }

        if (progressOut == 100 && func_70301_a(3) == null) {
            BuildCraftCore.instance.sendToPlayer(downloadingPlayer, new PacketCommand(this, "requestSelectedBlueprint", null));
            progressOut = 0;
        }
    }

    @Override
    public boolean hasCustomName() {
        return false;
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        if (side.isClient()) {
            if ("requestSelectedBlueprint".equals(command)) {
                if (isOutputConsistent()) {
                    if (selected > -1 && selected < entries.size()) {
                        // Work around 32k max limit on client->server
                        final NBTTagCompound compound = BuildCraftBuilders.clientDB.load(entries.get(selected));
                        compound.func_74778_a("__filename", entries.get(selected).name);
                        final byte[] bptData = NBTUtils.save(compound);
                        final int chunks = (bptData.length + CHUNK_SIZE - 1) / CHUNK_SIZE;

                        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "uploadServerBegin", new CommandWriter() {
                            @Override
                            public void write(ByteBuf data) {
                                entries.get(selected).writeData(data);
                                data.writeShort(chunks);
                            }
                        }));

                        for (int i = 0; i < chunks; i++) {
                            final int chunk = i;
                            final int start = CHUNK_SIZE * chunk;
                            final int length = Math.min(CHUNK_SIZE, bptData.length - start);
                            BuildCraftCore.instance.sendToServer(new PacketCommand(this, "uploadServerChunk", new CommandWriter() {
                                @Override
                                public void write(ByteBuf data) {
                                    data.writeShort(chunk);
                                    data.writeShort(length);
                                    data.writeBytes(bptData, start, length);
                                }
                            }));
                        }

                        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "uploadServerEnd", null));
                    } else {
                        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "uploadNothingToServer", null));
                    }
                }
            } else if ("downloadBlueprintToClient".equals(command)) {
                LibraryId id = new LibraryId();
                id.readData(stream);
                byte[] data = NetworkUtils.readByteArray(stream);

                try {
                    LibraryTypeHandler handler = LibraryAPI.getHandlerFor(id.extension);
                    if (handler == null) {
                        return;
                    }

                    NBTTagCompound nbt = CompressedStreamTools.func_74796_a(new ByteArrayInputStream(data));
                    BuildCraftBuilders.clientDB.add(id, nbt);
                    entries = BuildCraftBuilders.clientDB.getBlueprintIds();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } else if (side.isServer()) {
            if ("uploadNothingToServer".equals(command)) {
                func_70299_a(3, func_70301_a(2));
                func_70299_a(2, null);

                downloadingPlayer = null;
            } else if ("uploadServerBegin".equals(command)) {
                blueprintDownloadId = new LibraryId();
                blueprintDownloadId.readData(stream);
                blueprintDownload = new byte[CHUNK_SIZE * stream.readUnsignedShort()];
            } else if ("uploadServerChunk".equals(command)) {
                int start = stream.readUnsignedShort() * CHUNK_SIZE;
                int length = stream.readUnsignedShort();
                if (blueprintDownload != null) {
                    stream.readBytes(blueprintDownload, start, length);
                } else {
                    stream.skipBytes(length);
                }
            } else if ("uploadServerEnd".equals(command)) {
                try {
                    LibraryTypeHandler handler = LibraryAPI.getHandlerFor(blueprintDownloadId.extension);

                    if (handler != null) {
                        ItemStack output = null;

                        if (handler instanceof LibraryTypeHandlerNBT) {
                            NBTTagCompound nbt = CompressedStreamTools.func_74796_a(new ByteArrayInputStream(blueprintDownload));
                            output = ((LibraryTypeHandlerNBT) handler).load(func_70301_a(2), nbt);
                        } else if (handler instanceof LibraryTypeHandlerByteArray) {
                            output = ((LibraryTypeHandlerByteArray) handler).load(func_70301_a(2), blueprintDownload);
                        }

                        if (output != null) {
                            func_70299_a(3, output);
                            func_70299_a(2, null);
                        }
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }

                blueprintDownloadId = null;
                blueprintDownload = null;
                downloadingPlayer = null;
            } else if ("selectBlueprint".equals(command)) {
                selected = stream.readInt();
            }
        }
    }

    public void selectBlueprint(int index) {
        selected = index;
        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "selectBlueprint", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                data.writeInt(selected);
            }
        }));
    }

    private boolean isOutputConsistent() {
        if (selected <= -1 || selected >= entries.size() || func_70301_a(2) == null) {
            return false;
        }

        return LibraryAPI.getHandlerFor(entries.get(selected).extension).isHandler(func_70301_a(2), LibraryTypeHandler.HandlerType.LOAD);
    }
}
