/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.builders;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.BCLog;
import buildcraft.api.core.IInvSlot;
import buildcraft.api.core.IPathProvider;
import buildcraft.api.core.SafeTimeTracker;
import buildcraft.api.enums.EnumBlueprintType;
import buildcraft.api.properties.BuildCraftProperties;
import buildcraft.api.robots.IRequestProvider;
import buildcraft.api.tiles.IControllable;
import buildcraft.api.tiles.IHasWork;
import buildcraft.builders.blueprints.RecursiveBlueprintBuilder;
import buildcraft.core.Box;
import buildcraft.core.Box.Kind;
import buildcraft.core.LaserData;
import buildcraft.core.blueprints.*;
import buildcraft.core.builders.TileAbstractBuilder;
import buildcraft.core.lib.fluids.Tank;
import buildcraft.core.lib.fluids.TankManager;
import buildcraft.core.lib.inventory.*;
import buildcraft.core.lib.network.base.Packet;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;
import io.netty.buffer.ByteBuf;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Vec3;
import net.minecraft.world.WorldSettings.GameType;
import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fluids.*;
import net.minecraftforge.fml.relauncher.Side;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import net.minecraft.world.WorldSettings.GameType;
public class TileBuilder extends TileAbstractBuilder implements IHasWork, IFluidHandler, IRequestProvider, IControllable, IInventoryListener {

    private static int POWER_ACTIVATION = 500;

    public Box box = new Box();
    public PathIterator currentPathIterator;
    public Tank[] fluidTanks = new Tank[] {
        //@formatter:off
            new Tank("fluid1", FluidContainerRegistry.BUCKET_VOLUME * 8, this),
            new Tank("fluid2", FluidContainerRegistry.BUCKET_VOLUME * 8, this),
            new Tank("fluid3", FluidContainerRegistry.BUCKET_VOLUME * 8, this),
            new Tank("fluid4", FluidContainerRegistry.BUCKET_VOLUME * 8, this)
        //@formatter:on
    };
    public TankManager<Tank> fluidTank = new TankManager<>(fluidTanks);

    private SafeTimeTracker networkUpdateTracker = new SafeTimeTracker(BuildCraftCore.updateFactor / 2);
    private boolean shouldUpdateRequirements;

    private SimpleInventory inv = new SimpleInventory(28, "Builder", 64);
    private BptBuilderBase currentBuilder;
    private RecursiveBlueprintBuilder recursiveBuilder;
    private List<BlockPos> path;
    private List<RequirementItemStack> requiredToBuild;
    private NBTTagCompound initNBT = null;
    private boolean done = true;
    private boolean isBuilding = false;
    /** A cached value used at the client for the block state */
    private EnumBlueprintType type = EnumBlueprintType.NONE;

    private class PathIterator {

        public Iterator<BlockPos> currentIterator;
        public double cx, cy, cz;
        public float ix, iy, iz;
        public BlockPos to;
        public double lastDistance;
        AxisAlignedBB oldBoundingBox = null;
        EnumFacing o = null;

        public PathIterator(BlockPos from, Iterator<BlockPos> it, EnumFacing initialDir) {
            this.to = it.next();

            currentIterator = it;

            double dx = to.func_177958_n() - from.func_177958_n();
            double dy = to.func_177956_o() - from.func_177956_o();
            double dz = to.func_177952_p() - from.func_177952_p();

            double size = Math.sqrt(dx * dx + dy * dy + dz * dz);

            cx = dx / size / 10;
            cy = dy / size / 10;
            cz = dz / size / 10;

            ix = from.func_177958_n();
            iy = from.func_177956_o();
            iz = from.func_177952_p();

            lastDistance = (ix - to.func_177958_n()) * (ix - to.func_177958_n()) + (iy - to.func_177956_o()) * (iy - to.func_177956_o()) + (iz - to.func_177952_p()) * (iz - to.func_177952_p());

            if (dx == 0 && dz == 0) {
                o = initialDir;
            } else if (Math.abs(dx) > Math.abs(dz)) {
                if (dx > 0) {
                    o = EnumFacing.EAST;
                } else {
                    o = EnumFacing.WEST;
                }
            } else {
                if (dz > 0) {
                    o = EnumFacing.SOUTH;
                } else {
                    o = EnumFacing.NORTH;
                }
            }
        }

        /** Return false when reached the end of the iteration */
        public BptBuilderBase next() {
            while (true) {
                BptBuilderBase bpt;

                int newX = Math.round(ix);
                int newY = Math.round(iy);
                int newZ = Math.round(iz);

                bpt = instanciateBluePrintBuilder(new BlockPos(newX, newY, newZ), o);

                if (bpt == null) {
                    return null;
                }

                AxisAlignedBB boundingBox = bpt.getBoundingBox();

                if (oldBoundingBox == null || !collision(oldBoundingBox, boundingBox)) {
                    oldBoundingBox = boundingBox;
                    return bpt;
                }

                ix += cx;
                iy += cy;
                iz += cz;

                double distance = (ix - to.func_177958_n()) * (ix - to.func_177958_n()) + (iy - to.func_177956_o()) * (iy - to.func_177956_o()) + (iz - to.func_177952_p()) * (iz - to.func_177952_p());

                if (distance > lastDistance) {
                    return null;
                } else {
                    lastDistance = distance;
                }
            }
        }

        public PathIterator iterate() {
            if (currentIterator.hasNext()) {
                PathIterator next = new PathIterator(to, currentIterator, o);
                next.oldBoundingBox = oldBoundingBox;

                return next;
            } else {
                return null;
            }
        }

        public boolean collision(AxisAlignedBB left, AxisAlignedBB right) {
            if (left.field_72336_d < right.field_72340_a || left.field_72340_a > right.field_72336_d) {
                return false;
            }
            if (left.field_72337_e < right.field_72338_b || left.field_72338_b > right.field_72337_e) {
                return false;
            }
            if (left.field_72334_f < right.field_72339_c || left.field_72339_c > right.field_72334_f) {
                return false;
            }
            return true;
        }
    }

    public TileBuilder() {
        super();

        box.kind = Kind.STRIPES;
        inv.addInvListener(this);
    }

    @Override
    public void initialize() {
        super.initialize();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (initNBT != null) {
            iterateBpt(true);

            if (initNBT.func_74764_b("iterator")) {
                BlockPos expectedTo = NBTUtils.readBlockPos(initNBT.func_74781_a("iterator"));

                while (!done && currentBuilder != null && currentPathIterator != null) {
                    BlockPos bi = new BlockPos((int) currentPathIterator.ix, (int) currentPathIterator.iy, (int) currentPathIterator.iz);

                    if (bi.equals(expectedTo)) {
                        break;
                    }

                    iterateBpt(true);
                }
            }

            if (currentBuilder != null) {
                currentBuilder.loadBuildStateToNBT(initNBT.func_74775_l("builderState"), this);
            }

            initNBT = null;
        }

        box.kind = Kind.STRIPES;

        for (EnumFacing face : EnumFacing.values()) {
            TileEntity tile = field_145850_b.func_175625_s(field_174879_c.func_177972_a(face));

            if (tile instanceof IPathProvider) {
                path = ((IPathProvider) tile).getPath();
                ((IPathProvider) tile).removeFromWorld();

                break;
            }
        }

        if (path != null && pathLasers.size() == 0) {
            createLasersForPath();
            sendNetworkUpdate();
        }

        iterateBpt(false);
    }

    public void createLasersForPath() {
        pathLasers = new LinkedList<>();
        BlockPos previous = null;

        for (BlockPos b : path) {
            if (previous != null) {
                Vec3 point5 = new Vec3(0.5, 0.5, 0.5);
                LaserData laser = new LaserData(Utils.convert(previous).func_178787_e(point5), Utils.convert(b).func_178787_e(point5));

                pathLasers.add(laser);
            }

            previous = b;
        }
    }

    public BlueprintBase instanciateBlueprint() {
        BlueprintBase bpt;

        try {
            bpt = ItemBlueprint.loadBlueprint(func_70301_a(0));
        } catch (Throwable t) {
            t.printStackTrace();
            return null;
        }

        return bpt;
    }

    @Deprecated
    public BptBuilderBase instanciateBluePrintBuilder(BlockPos pos, EnumFacing o) {
        BlueprintBase bpt = instanciateBlueprint();
        if (bpt == null) {
            return null;
        }

        bpt = bpt.adjustToWorld(field_145850_b, pos, o);

        if (bpt != null) {
            if (func_70301_a(0).func_77973_b() instanceof ItemBlueprintStandard) {
                return new BptBuilderBlueprint((Blueprint) bpt, field_145850_b, pos);
            } else if (func_70301_a(0).func_77973_b() instanceof ItemBlueprintTemplate) {
                return new BptBuilderTemplate(bpt, field_145850_b, pos);
            }
        }
        return null;
    }

    public void iterateBpt(boolean forceIterate) {
        if (func_70301_a(0) == null || !(func_70301_a(0).func_77973_b() instanceof ItemBlueprint)) {
            if (box.isInitialized()) {
                if (currentBuilder != null) {
                    currentBuilder = null;
                }

                if (box.isInitialized()) {
                    box.reset();
                }

                if (currentPathIterator != null) {
                    currentPathIterator = null;
                }

                scheduleRequirementUpdate();

                sendNetworkUpdate();

                return;
            }
        }

        if (currentBuilder == null || (currentBuilder.isDone(this) || forceIterate)) {
            if (path != null && path.size() > 1) {
                if (currentPathIterator == null) {
                    Iterator<BlockPos> it = path.iterator();
                    BlockPos start = it.next();

                    EnumFacing face = field_145850_b.func_180495_p(field_174879_c).func_177229_b(BuildCraftProperties.BLOCK_FACING);
                    currentPathIterator = new PathIterator(start, it, face);
                }

                if (currentBuilder != null && currentBuilder.isDone(this)) {
                    currentBuilder.postProcessing(field_145850_b);
                }

                currentBuilder = currentPathIterator.next();

                if (currentBuilder != null) {
                    box.reset();
                    box.initialize(currentBuilder);
                    sendNetworkUpdate();
                }

                if (currentBuilder == null) {
                    currentPathIterator = currentPathIterator.iterate();
                }

                if (currentPathIterator == null) {
                    done = true;
                } else {
                    done = false;
                }

                scheduleRequirementUpdate();
            } else {
                if (currentBuilder != null && currentBuilder.isDone(this)) {
                    currentBuilder.postProcessing(field_145850_b);
                    currentBuilder = recursiveBuilder.nextBuilder();

                    scheduleRequirementUpdate();
                } else {
                    BlueprintBase bpt = instanciateBlueprint();

                    if (bpt != null) {
                        EnumFacing face = field_145850_b.func_180495_p(field_174879_c).func_177229_b(BuildCraftProperties.BLOCK_FACING);
                        recursiveBuilder = new RecursiveBlueprintBuilder(bpt, field_145850_b, field_174879_c, face.func_176734_d());

                        currentBuilder = recursiveBuilder.nextBuilder();

                        scheduleRequirementUpdate();
                    }
                }

                if (currentBuilder == null) {
                    done = true;
                } else {
                    box.initialize(currentBuilder);
                    sendNetworkUpdate();
                    done = false;
                }
            }
        }

        if (done && func_70301_a(0) != null) {
            boolean dropBlueprint = true;
            for (int i = 1; i < func_70302_i_(); ++i) {
                if (func_70301_a(i) == null) {
                    func_70299_a(i, func_70301_a(0));
                    dropBlueprint = false;
                    break;
                }
            }

            if (dropBlueprint) {
                InvUtils.dropItems(func_145831_w(), func_70301_a(0), field_174879_c);
            }

            func_70299_a(0, null);
            box.reset();
            sendNetworkUpdate();
        }
    }

    @Override
    public int func_70302_i_() {
        return inv.func_70302_i_();
    }

    @Override
    public ItemStack func_70301_a(int i) {
        return inv.func_70301_a(i);
    }

    @Override
    public ItemStack func_70298_a(int i, int j) {
        ItemStack result = inv.func_70298_a(i, j);

        if (!field_145850_b.field_72995_K) {
            if (i == 0) {
                BuildCraftCore.instance.sendToWorld(new PacketCommand(this, "clearItemRequirements", null), field_145850_b);
                iterateBpt(false);
            }
        }

        return result;
    }

    @Override
    public void func_70299_a(int i, ItemStack itemstack) {
        inv.func_70299_a(i, itemstack);

        if (!field_145850_b.field_72995_K) {
            if (i == 0) {
                iterateBpt(false);
                done = false;
            }
        }
    }

    @Override
    public ItemStack func_70304_b(int slot) {
        return inv.func_70304_b(slot);
    }

    @Override
    public String getInventoryName() {
        return "Builder";
    }

    @Override
    public int func_70297_j_() {
        return 64;
    }

    @Override
    public boolean func_70300_a(EntityPlayer entityplayer) {
        return field_145850_b.func_175625_s(field_174879_c) == this;
    }

    @Override
    public void func_145839_a(NBTTagCompound nbttagcompound) {
        super.func_145839_a(nbttagcompound);

        inv.readFromNBT(nbttagcompound);

        if (nbttagcompound.func_74764_b("box")) {
            box.initialize(nbttagcompound.func_74775_l("box"));
        }

        type = EnumBlueprintType.getType(func_70301_a(0));

        if (nbttagcompound.func_74764_b("path")) {
            path = new LinkedList<>();
            NBTTagList list = nbttagcompound.func_150295_c("path", Constants.NBT.TAG_COMPOUND);

            for (int i = 0; i < list.func_74745_c(); ++i) {
                path.add(NBTUtils.readBlockPos(list.func_179238_g(i)));
            }
        }

        done = nbttagcompound.func_74767_n("done");
        fluidTank.readFromNBT(nbttagcompound);

        // The rest of load has to be done upon initialize.
        initNBT = (NBTTagCompound) nbttagcompound.func_74775_l("bptBuilder").func_74737_b();
    }

    @Override
    public void func_145841_b(NBTTagCompound nbttagcompound) {
        super.func_145841_b(nbttagcompound);

        inv.writeToNBT(nbttagcompound);

        if (box.isInitialized()) {
            NBTTagCompound boxStore = new NBTTagCompound();
            box.writeToNBT(boxStore);
            nbttagcompound.func_74782_a("box", boxStore);
        }

        if (path != null) {
            NBTTagList list = new NBTTagList();

            for (BlockPos i : path) {
                list.func_74742_a(NBTUtils.writeBlockPos(i));
            }

            nbttagcompound.func_74782_a("path", list);
        }

        nbttagcompound.func_74757_a("done", done);
        fluidTank.writeToNBT(nbttagcompound);

        NBTTagCompound bptNBT = new NBTTagCompound();

        if (currentBuilder != null) {
            NBTTagCompound builderCpt = new NBTTagCompound();
            currentBuilder.saveBuildStateToNBT(builderCpt, this);
            bptNBT.func_74782_a("builderState", builderCpt);
        }

        if (currentPathIterator != null) {
            BlockPos nPos = new BlockPos((int) currentPathIterator.ix, (int) currentPathIterator.iy, (int) currentPathIterator.iz);
            bptNBT.func_74782_a("iterator", NBTUtils.writeBlockPos(nPos));
        }

        nbttagcompound.func_74782_a("bptBuilder", bptNBT);
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        destroy();
    }

    @Override
    public void func_174889_b(EntityPlayer player) {}

    @Override
    public void func_174886_c(EntityPlayer player) {}

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (shouldUpdateRequirements && networkUpdateTracker.markTimeIfDelay(field_145850_b)) {
            updateRequirements();
            shouldUpdateRequirements = false;
        }

        if ((currentBuilder == null || currentBuilder.isDone(this)) && box.isInitialized()) {
            box.reset();

            sendNetworkUpdate();

            return;
        }

        iterateBpt(false);

        if (mode != Mode.Off) {
            if (func_145831_w().func_72912_H().func_76077_q() == GameType.CREATIVE || getBattery().getEnergyStored() > POWER_ACTIVATION) {
                build();
            }
        }

        if (!isBuilding && this.isBuildingBlueprint()) {
            scheduleRequirementUpdate();
        }
        isBuilding = this.isBuildingBlueprint();

        if (done) {// TODO (PASS 3): This is useless right? Is/was this needed for anything?
            return;
        } else if (getBattery().getEnergyStored() < 25) {
            return;
        }
    }

    @Override
    public boolean hasWork() {
        return !done;
    }

    public boolean isBuildingBlueprint() {
        return func_70301_a(0) != null && func_70301_a(0).func_77973_b() instanceof ItemBlueprint;
    }

    public List<RequirementItemStack> getNeededItems() {
        return field_145850_b.field_72995_K ? requiredToBuild : (currentBuilder instanceof BptBuilderBlueprint ? ((BptBuilderBlueprint) currentBuilder)
                .getNeededItems() : null);
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        super.receiveCommand(command, side, sender, stream);
        if (side.isClient()) {
            if ("clearItemRequirements".equals(command)) {
                requiredToBuild = null;
            } else if ("setItemRequirements".equals(command)) {
                int size = stream.readUnsignedMedium();
                requiredToBuild = new ArrayList<>();
                for (int i = 0; i < size; i++) {
                    int itemId = stream.readUnsignedShort();
                    int itemDamage = stream.readShort();
                    int stackSize = stream.readUnsignedMedium();
                    boolean hasCompound = stackSize >= 0x800000;

                    ItemStack stack = new ItemStack(Item.func_150899_d(itemId), 1, itemDamage);
                    if (hasCompound) {
                        stack.func_77982_d(NetworkUtils.readNBT(stream));
                    }

                    if (stack.func_77973_b() != null) {
                        requiredToBuild.add(new RequirementItemStack(stack, stackSize & 0x7FFFFF));
                    } else {
                        BCLog.logger.error("Corrupt ItemStack in TileBuilder.receiveCommand! This should not happen! (ID " + itemId + ", damage "
                            + itemDamage + ")");
                    }
                }
            }
        } else if (side.isServer()) {
            EntityPlayer player = (EntityPlayer) sender;
            if ("eraseFluidTank".equals(command)) {
                int id = stream.readInt();
                if (id < 0 || id >= fluidTanks.length) {
                    return;
                }
                if (func_70300_a(player) && player.func_174818_b(field_174879_c) <= 64) {
                    fluidTanks[id].setFluid(null);
                    sendNetworkUpdate();
                }
            }
        }
    }

    private Packet getItemRequirementsPacket(List<RequirementItemStack> itemsIn) {
        if (itemsIn != null) {
            final List<RequirementItemStack> items = new ArrayList<>();
            items.addAll(itemsIn);

            return new PacketCommand(this, "setItemRequirements", new CommandWriter() {
                @Override
                public void write(ByteBuf data) {
                    data.writeMedium(items.size());
                    for (RequirementItemStack rb : items) {
                        data.writeShort(Item.func_150891_b(rb.stack.func_77973_b()));
                        data.writeShort(rb.stack.func_77952_i());
                        data.writeMedium((rb.stack.func_77942_o() ? 0x800000 : 0x000000) | Math.min(0x7FFFFF, rb.size));
                        if (rb.stack.func_77942_o()) {
                            NetworkUtils.writeNBT(data, rb.stack.func_77978_p());
                        }
                    }
                }
            });
        } else {
            return new PacketCommand(this, "clearItemRequirements", null);
        }
    }

    @Override
    public boolean isBuildingMaterialSlot(int i) {
        return i != 0;
    }

    @Override
    public boolean hasCustomName() {
        return false;
    }

    @Override
    public boolean func_94041_b(int slot, ItemStack stack) {
        if (slot == 0) {
            return stack.func_77973_b() instanceof ItemBlueprint;
        } else {
            return true;
        }
    }

    @Override
    public Box getBox() {
        return box;
    }

    @Override
    public AxisAlignedBB getRenderBoundingBox() {
        Box renderBox = new Box(this).extendToEncompass(box);

        for (LaserData l : pathLasers) {
            renderBox = renderBox.extendToEncompass(l.head);
            renderBox = renderBox.extendToEncompass(l.tail);
        }

        return renderBox.expand(50).getBoundingBox();
    }

    public void build() {
        if (currentBuilder != null) {
            if (currentBuilder.buildNextSlot(field_145850_b, this)) {
                scheduleRequirementUpdate();
            }
        }
    }

    private void updateRequirements() {
        List<RequirementItemStack> reqCopy = null;
        if (currentBuilder instanceof BptBuilderBlueprint) {
            currentBuilder.initialize();
            reqCopy = ((BptBuilderBlueprint) currentBuilder).getNeededItems();
        }

        for (EntityPlayer p : guiWatchers) {
            BuildCraftCore.instance.sendToPlayer(p, getItemRequirementsPacket(reqCopy));
        }
    }

    public void scheduleRequirementUpdate() {
        shouldUpdateRequirements = true;
    }

    public void updateRequirementsOnGuiOpen(EntityPlayer caller) {
        List<RequirementItemStack> reqCopy = null;
        if (currentBuilder instanceof BptBuilderBlueprint) {
            currentBuilder.initialize();
            reqCopy = ((BptBuilderBlueprint) currentBuilder).getNeededItems();
        }

        BuildCraftCore.instance.sendToPlayer(caller, getItemRequirementsPacket(reqCopy));
    }

    public BptBuilderBase getBlueprint() {
        if (currentBuilder != null) {
            return currentBuilder;
        } else {
            return null;
        }
    }

    @Override
    public boolean canDrain(EnumFacing from, Fluid fluid) {
        return false;
    }

    @Override
    public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain) {
        return null;
    }

    @Override
    public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain) {
        return null;
    }

    @Override
    public boolean drainBuild(FluidStack fluidStack, boolean realDrain) {
        for (Tank tank : fluidTanks) {
            if (tank.getFluidType() == fluidStack.getFluid()) {
                return tank.getFluidAmount() >= fluidStack.amount && tank.drain(fluidStack.amount, realDrain).amount > 0;
            }
        }
        return false;
    }

    @Override
    public int fill(EnumFacing from, FluidStack resource, boolean doFill) {
        Fluid fluid = resource.getFluid();
        Tank emptyTank = null;
        for (Tank tank : fluidTanks) {
            Fluid type = tank.getFluidType();
            if (type == fluid) {
                int used = tank.fill(resource, doFill);
                if (used > 0 && doFill) {
                    sendNetworkUpdate();
                }
                return used;
            } else if (emptyTank == null && tank.isEmpty()) {
                emptyTank = tank;
            }
        }
        if (emptyTank != null) {
            int used = emptyTank.fill(resource, doFill);
            if (used > 0 && doFill) {
                sendNetworkUpdate();
            }
            return used;
        }
        return 0;
    }

    @Override
    public boolean canFill(EnumFacing from, Fluid fluid) {
        boolean emptyAvailable = false;
        for (Tank tank : fluidTanks) {
            Fluid type = tank.getFluidType();
            if (type == fluid) {
                return !tank.isFull();
            } else if (!emptyAvailable) {
                emptyAvailable = tank.isEmpty();
            }
        }
        return emptyAvailable;
    }

    @Override
    public FluidTankInfo[] getTankInfo(EnumFacing from) {
        return fluidTank.getTankInfo(from);
    }

    @Override
    public int getRequestsCount() {
        if (currentBuilder == null) {
            return 0;
        } else if (!(currentBuilder instanceof BptBuilderBlueprint)) {
            return 0;
        } else {
            BptBuilderBlueprint bpt = (BptBuilderBlueprint) currentBuilder;

            return bpt.getNeededItems().size();
        }
    }

    @Override
    public ItemStack getRequest(int slot) {
        if (currentBuilder == null) {
            return null;
        } else if (!(currentBuilder instanceof BptBuilderBlueprint)) {
            return null;
        } else {
            BptBuilderBlueprint bpt = (BptBuilderBlueprint) currentBuilder;
            List<RequirementItemStack> neededItems = bpt.getNeededItems();

            if (neededItems.size() <= slot) {
                return null;
            }

            RequirementItemStack requirement = neededItems.get(slot);

            int qty = quantityMissing(requirement.stack, requirement.size);

            if (qty <= 0) {
                return null;
            }
            ItemStack requestStack = requirement.stack.func_77946_l();
            requestStack.field_77994_a = qty;
            return requestStack;
        }
    }

    @Override
    public ItemStack offerItem(int slot, ItemStack stack) {
        if (currentBuilder == null) {
            return stack;
        } else if (!(currentBuilder instanceof BptBuilderBlueprint)) {
            return stack;
        } else {
            BptBuilderBlueprint bpt = (BptBuilderBlueprint) currentBuilder;
            List<RequirementItemStack> neededItems = bpt.getNeededItems();

            if (neededItems.size() <= slot) {
                return stack;
            }

            RequirementItemStack requirement = neededItems.get(slot);

            int qty = quantityMissing(requirement.stack, requirement.size);

            if (qty <= 0) {
                return stack;
            }

            ItemStack toAdd = stack.func_77946_l();

            if (qty < toAdd.field_77994_a) {
                toAdd.field_77994_a = qty;
            }

            ITransactor t = Transactor.getTransactorFor(this, null);
            ItemStack added = t.add(toAdd, true);

            if (added.field_77994_a >= stack.field_77994_a) {
                return null;
            } else {
                stack.field_77994_a -= added.field_77994_a;
                return stack;
            }
        }
    }

    private int quantityMissing(ItemStack requirement, int amount) {
        int left = amount <= 0 ? requirement.field_77994_a : amount;

        for (IInvSlot slot : InventoryIterator.getIterable(this)) {
            if (slot.getStackInSlot() != null) {
                // TODO: This should also be using the Schematic version of the function!
                if (StackHelper.isEqualItem(requirement, slot.getStackInSlot())) {
                    if (slot.getStackInSlot().stackSize >= left) {
                        return 0;
                    } else {
                        left -= slot.getStackInSlot().stackSize;
                    }
                }
            }
        }

        return left;
    }

    @Override
    public boolean acceptsControlMode(Mode mode) {
        return mode == Mode.Off || mode == Mode.On;
    }

    @Override
    public void writeData(ByteBuf stream) {
        super.writeData(stream);
        box.writeData(stream);
        fluidTank.writeData(stream);
        stream.writeByte(getType().ordinal());
    }

    @Override
    public void readData(ByteBuf stream) {
        super.readData(stream);
        box.readData(stream);
        fluidTank.readData(stream);
        byte type = stream.readByte();
        EnumBlueprintType old = this.type;
        this.type = EnumBlueprintType.valueOf(type);
        if (old != this.type) {
            // Needed because BLUEPRINT_TYPE is not part of the integer block meta, so we must tell minecraft that this
            // changed ourselves
            field_145850_b.func_175704_b(field_174879_c, field_174879_c);
        }
    }

    public EnumBlueprintType getType() {
        return type;
    }

    @Override
    public void onChange(int slot, ItemStack before, ItemStack after) {
        if (slot == 0 && field_145850_b != null && !field_145850_b.field_72995_K) {
            EnumBlueprintType beforeType = EnumBlueprintType.getType(before);
            EnumBlueprintType afterType = EnumBlueprintType.getType(after);
            if (beforeType != afterType) {
                type = afterType;
                sendNetworkUpdate();
            }
        }
    }

    @Override
    public Tank[] getFluidTanks() {
        return fluidTanks;
    }
}
