/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.builders;

import java.util.ArrayList;
import java.util.HashSet;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Vec3;

import net.minecraftforge.fml.relauncher.Side;

import buildcraft.BuildCraftCore;
import buildcraft.core.Box;
import buildcraft.core.Box.Kind;
import buildcraft.core.LaserData;
import buildcraft.core.blueprints.*;
import buildcraft.core.builders.BuildingItem;
import buildcraft.core.builders.IBuildingItemsProvider;
import buildcraft.core.internal.IBoxProvider;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.network.base.Packet;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.ICommandReceiver;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class TileConstructionMarker extends TileBuildCraft implements IBuildingItemsProvider, IBoxProvider, ICommandReceiver {

    public static HashSet<TileConstructionMarker> currentMarkers = new HashSet<>();

    public EnumFacing direction = null;

    public LaserData laser;
    public ItemStack itemBlueprint;
    public Box box = new Box();

    public BptBuilderBase bluePrintBuilder;
    public BptContext bptContext;

    private ArrayList<BuildingItem> buildersInAction = new ArrayList<>();
    private NBTTagCompound initNBT;

    @Override
    public void initialize() {
        super.initialize();
        box.kind = Kind.BLUE_STRIPES;

        if (field_145850_b.field_72995_K) {
            BuildCraftCore.instance.sendToServer(new PacketCommand(this, "uploadBuildersInAction", null));
        }
    }

    private Packet createLaunchItemPacket(final BuildingItem i) {
        return new PacketCommand(this, "launchItem", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                i.writeData(data);
            }
        });
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        BuildingItem toRemove = null;

        for (BuildingItem i : buildersInAction) {
            i.update();

            if (i.isDone) {
                toRemove = i;
            }
        }

        if (toRemove != null) {
            buildersInAction.remove(toRemove);
        }

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (itemBlueprint != null && ItemBlueprint.getId(itemBlueprint) != null && bluePrintBuilder == null) {
            BlueprintBase bpt = ItemBlueprint.loadBlueprint(itemBlueprint);
            if (bpt != null && bpt instanceof Blueprint) {
                bpt = bpt.adjustToWorld(field_145850_b, field_174879_c, direction);
                if (bpt != null) {
                    bluePrintBuilder = new BptBuilderBlueprint((Blueprint) bpt, field_145850_b, field_174879_c);
                    bptContext = bluePrintBuilder.getContext();
                    box.initialize(bluePrintBuilder);
                    sendNetworkUpdate();
                }
            } else {
                return;
            }
        }

        if (laser == null && direction != null) {
            Vec3 point5 = new Vec3(0.5, 0.5, 0.5);
            laser = new LaserData();
            laser.head = Utils.convert(field_174879_c).func_178787_e(point5);
            laser.tail = laser.head.func_178787_e(Utils.convert(direction, 0.5));
            laser.isVisible = true;
            sendNetworkUpdate();
        }

        if (initNBT != null) {
            if (bluePrintBuilder != null) {
                bluePrintBuilder.loadBuildStateToNBT(initNBT.func_74775_l("builderState"), this);
            }

            initNBT = null;
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);

        nbt.func_74782_a("direction", NBTUtils.writeEnum(direction));

        if (itemBlueprint != null) {
            NBTTagCompound bptNBT = new NBTTagCompound();
            itemBlueprint.func_77955_b(bptNBT);
            nbt.func_74782_a("itemBlueprint", bptNBT);
        }

        NBTTagCompound bptNBT = new NBTTagCompound();

        if (bluePrintBuilder != null) {
            NBTTagCompound builderCpt = new NBTTagCompound();
            bluePrintBuilder.saveBuildStateToNBT(builderCpt, this);
            bptNBT.func_74782_a("builderState", builderCpt);
        }

        nbt.func_74782_a("bptBuilder", bptNBT);
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);

        direction = NBTUtils.readEnum(nbt.func_74781_a("direction"), EnumFacing.class);

        if (nbt.func_74764_b("itemBlueprint")) {
            itemBlueprint = ItemStack.func_77949_a(nbt.func_74775_l("itemBlueprint"));
        }

        // The rest of load has to be done upon initialize.
        initNBT = (NBTTagCompound) nbt.func_74775_l("bptBuilder").func_74737_b();
    }

    public void setBlueprint(ItemStack currentItem) {
        itemBlueprint = currentItem;
        sendNetworkUpdate();
    }

    @Override
    public ArrayList<BuildingItem> getBuilders() {
        return buildersInAction;
    }

    @Override
    public void func_145829_t() {
        super.func_145829_t();
        if (!field_145850_b.field_72995_K) {
            currentMarkers.add(this);
        }
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        if (!field_145850_b.field_72995_K) {
            currentMarkers.remove(this);
        }
    }

    public boolean needsToBuild() {
        return !func_145837_r() && bluePrintBuilder != null && !bluePrintBuilder.isDone(this);
    }

    public BptContext getContext() {
        return bptContext;
    }

    @Override
    public void addAndLaunchBuildingItem(BuildingItem item) {
        buildersInAction.add(item);
        BuildCraftCore.instance.sendToPlayersNear(createLaunchItemPacket(item), this);
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        if (side.isServer() && "uploadBuildersInAction".equals(command)) {
            for (BuildingItem i : buildersInAction) {
                BuildCraftCore.instance.sendToPlayer((EntityPlayer) sender, createLaunchItemPacket(i));
            }
        } else if (side.isClient() && "launchItem".equals(command)) {
            BuildingItem item = new BuildingItem();
            item.readData(stream);
            buildersInAction.add(item);
        }
    }

    @Override
    public Box getBox() {
        return box;
    }

    @Override
    public AxisAlignedBB getRenderBoundingBox() {
        Box renderBox = new Box(this).extendToEncompass(box);

        return renderBox.expand(50).getBoundingBox();
    }

    @Override
    public double func_145833_n() {
        return Double.MAX_VALUE;
    }

    @Override
    public void writeData(ByteBuf stream) {
        box.writeData(stream);
        stream.writeByte((laser != null ? 1 : 0) | (itemBlueprint != null ? 2 : 0));
        if (laser != null) {
            laser.writeData(stream);
        }
        if (itemBlueprint != null) {
            NetworkUtils.writeStack(stream, itemBlueprint);
        }
    }

    @Override
    public void readData(ByteBuf stream) {
        box.readData(stream);
        int flags = stream.readUnsignedByte();
        if ((flags & 1) != 0) {
            laser = new LaserData();
            laser.readData(stream);
        } else {
            laser = null;
        }
        if ((flags & 2) != 0) {
            itemBlueprint = NetworkUtils.readStack(stream);
        } else {
            itemBlueprint = null;
        }
    }
}
