/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.builders;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.BlockPos;

import net.minecraftforge.fml.relauncher.Side;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.IAreaProvider;
import buildcraft.api.filler.FillerManager;
import buildcraft.api.properties.BuildCraftProperties;
import buildcraft.api.statements.IStatementContainer;
import buildcraft.api.statements.IStatementParameter;
import buildcraft.api.statements.StatementManager;
import buildcraft.api.tiles.IControllable;
import buildcraft.api.tiles.IHasWork;
import buildcraft.core.Box;
import buildcraft.core.Box.Kind;
import buildcraft.core.blueprints.BptBuilderTemplate;
import buildcraft.core.builders.TileAbstractBuilder;
import buildcraft.core.builders.patterns.FillerPattern;
import buildcraft.core.builders.patterns.PatternNone;
import buildcraft.core.lib.inventory.SimpleInventory;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.ICommandReceiver;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class TileFiller extends TileAbstractBuilder implements IHasWork, IControllable, ICommandReceiver, IStatementContainer {
    private static int POWER_ACTIVATION = 500;

    public FillerPattern currentPattern = PatternNone.INSTANCE;
    public IStatementParameter[] patternParameters;
    private int patternLocked;

    private BptBuilderTemplate currentTemplate;

    private final Box box = new Box();
    private boolean done = false;
    private boolean excavate = true;
    private SimpleInventory inv = new SimpleInventory(27, "Filler", 64);

    private NBTTagCompound initNBT = null;

    public TileFiller() {
        setControlMode(Mode.On);
        inv.addListener(this);
        box.kind = Kind.STRIPES;
        initPatternParameters();
    }

    public boolean isExcavate() {
        return excavate;
    }

    @Override
    public void initialize() {
        super.initialize();

        if (field_145850_b.field_72995_K) {
            return;
        }

        IAreaProvider a = Utils.getNearbyAreaProvider(field_145850_b, field_174879_c);

        if (a != null) {
            box.initialize(a);
            a.removeFromWorld();
            sendNetworkUpdate();
        }

        if (currentTemplate == null) {
            initTemplate();
        }

        if (initNBT != null && currentTemplate != null) {
            currentTemplate.loadBuildStateToNBT(initNBT.func_74775_l("builderState"), this);
        }

        initNBT = null;
    }

    private void initTemplate() {
        if (currentPattern != null && box.size().func_177951_i(BlockPos.field_177992_a) > 0) {
            currentTemplate = currentPattern.getTemplateBuilder(box, func_145831_w(), patternParameters);
            currentTemplate.blueprint.excavate = excavate;
        }
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (patternLocked > 0) patternLocked--;

        if (mode == Mode.Off) {
            return;
        }

        if (!box.isInitialized()) {
            return;
        }

        if (getBattery().getEnergyStored() < POWER_ACTIVATION) {
            return;
        }

        boolean oldDone = isDone();

        if (isDone()) {
            if (mode == Mode.Loop) {
                setDone(false);
            } else {
                return;
            }
        }

        if (currentTemplate == null) {
            initTemplate();
        }

        if (currentTemplate != null) {
            currentTemplate.buildNextSlot(field_145850_b, this);

            if (currentTemplate.isDone(this)) {
                setDone(true);
                currentTemplate = null;
            }
        }

        if (oldDone != isDone()) {
            sendNetworkUpdate();
        }
    }

    @Override
    public final int func_70302_i_() {
        return inv.func_70302_i_();
    }

    @Override
    public ItemStack func_70301_a(int slot) {
        return inv.func_70301_a(slot);
    }

    @Override
    public ItemStack func_70298_a(int slot, int amount) {
        return inv.func_70298_a(slot, amount);
    }

    @Override
    public void func_70299_a(int slot, ItemStack stack) {
        inv.func_70299_a(slot, stack);
    }

    @Override
    public ItemStack func_70304_b(int slot) {
        return inv.func_70304_b(slot);
    }

    @Override
    public String getInventoryName() {
        return "Filler";
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);

        inv.readFromNBT(nbt);

        if (nbt.func_74764_b("pattern")) {
            currentPattern = (FillerPattern) FillerManager.registry.getPattern(nbt.func_74779_i("pattern"));
        }

        if (currentPattern == null) {
            currentPattern = PatternNone.INSTANCE;
        }

        if (nbt.func_74764_b("pp")) {
            readParametersFromNBT(nbt.func_74775_l("pp"));
        } else {
            initPatternParameters();
        }

        if (nbt.func_74764_b("box")) {
            box.initialize(nbt.func_74775_l("box"));
        }

        done = nbt.func_74767_n("done");
        excavate = nbt.func_74764_b("excavate") ? nbt.func_74767_n("excavate") : true;

        // The rest of load has to be done upon initialize.
        initNBT = (NBTTagCompound) nbt.func_74775_l("bpt").func_74737_b();
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);

        inv.writeToNBT(nbt);

        if (currentPattern != null) {
            nbt.func_74778_a("pattern", currentPattern.getUniqueTag());
        }

        NBTTagCompound boxStore = new NBTTagCompound();
        box.writeToNBT(boxStore);
        nbt.func_74782_a("box", boxStore);

        nbt.func_74757_a("done", done);
        nbt.func_74757_a("excavate", excavate);

        NBTTagCompound bptNBT = new NBTTagCompound();

        if (currentTemplate != null) {
            NBTTagCompound builderCpt = new NBTTagCompound();
            currentTemplate.saveBuildStateToNBT(builderCpt, this);
            bptNBT.func_74782_a("builderState", builderCpt);
        }

        nbt.func_74782_a("bpt", bptNBT);

        NBTTagCompound ppNBT = new NBTTagCompound();
        writeParametersToNBT(ppNBT);
        nbt.func_74782_a("pp", ppNBT);
    }

    @Override
    public int func_70297_j_() {
        return inv.func_70297_j_();
    }

    @Override
    public boolean func_70300_a(EntityPlayer entityplayer) {
        if (field_145850_b.func_175625_s(field_174879_c) != this) {
            return false;
        }

        return entityplayer.func_70092_e(field_174879_c.func_177958_n() + 0.5D, field_174879_c.func_177956_o() + 0.5D, field_174879_c.func_177952_p() + 0.5D) <= 64D;
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        destroy();
    }

    private void initPatternParameters() {
        patternParameters = new IStatementParameter[currentPattern.maxParameters()];
        for (int i = 0; i < currentPattern.minParameters(); i++) {
            patternParameters[i] = currentPattern.createParameter(i);
        }
    }

    public boolean setPattern(FillerPattern pattern, boolean lock) {
        if (pattern != null && currentPattern != pattern) {
            currentPattern = pattern;
            currentTemplate = null;
            if (lock) patternLocked = 2;
            else patternLocked = 0;
            setDone(false);
            initPatternParameters();
            sendNetworkUpdate();
            return true;
        } else if (pattern != null && lock) {
            patternLocked = 2;
        }
        return false;
    }

    public boolean isPatternLocked() {
        return patternLocked > 0;
    }

    private void writeParametersToNBT(NBTTagCompound nbt) {
        IStatementParameter[] patternParameters = this.patternParameters;
        nbt.func_74774_a("length", (byte) (patternParameters != null ? patternParameters.length : 0));
        if (patternParameters != null) {
            for (int i = 0; i < patternParameters.length; i++) {
                if (patternParameters[i] != null) {
                    NBTTagCompound patternData = new NBTTagCompound();
                    patternData.func_74778_a("kind", patternParameters[i].getUniqueTag());
                    patternParameters[i].writeToNBT(patternData);// ArrayIndexOutOfBounds
                    nbt.func_74782_a("p" + i, patternData);
                }
            }
        }
    }

    private void readParametersFromNBT(NBTTagCompound nbt) {
        IStatementParameter[] patternParameters = new IStatementParameter[nbt.func_74771_c("length")];
        for (int i = 0; i < patternParameters.length; i++) {
            if (nbt.func_74764_b("p" + i)) {
                NBTTagCompound patternData = nbt.func_74775_l("p" + i);
                patternParameters[i] = StatementManager.createParameter(patternData.func_74779_i("kind"));
                patternParameters[i].readFromNBT(patternData);
            }
        }
        this.patternParameters = patternParameters;
    }

    @Override
    public void writeData(ByteBuf data) {
        super.writeData(data);
        box.writeData(data);
        data.writeByte((done ? 1 : 0) | (excavate ? 2 : 0) | (isPatternLocked() ? 4 : 0));
        NetworkUtils.writeUTF(data, currentPattern.getUniqueTag());

        NBTTagCompound parameterData = new NBTTagCompound();
        writeParametersToNBT(parameterData);
        NetworkUtils.writeNBT(data, parameterData);
    }

    @Override
    public void readData(ByteBuf data) {
        super.readData(data);
        box.readData(data);
        int flags = data.readUnsignedByte();
        done = (flags & 1) > 0;
        excavate = (flags & 2) > 0;
        patternLocked = (flags & 4) > 0 ? 2 : 0;
        FillerPattern pattern = (FillerPattern) FillerManager.registry.getPattern(NetworkUtils.readUTF(data));
        NBTTagCompound parameterData = NetworkUtils.readNBT(data);
        readParametersFromNBT(parameterData);
        if (setPattern(pattern, isPatternLocked())) {
            field_145850_b.func_175689_h(field_174879_c);
        }
    }

    @Override
    public boolean hasWork() {
        return !isDone() && mode != Mode.Off;
    }

    @Override
    public void func_174889_b(EntityPlayer player) {}

    @Override
    public void func_174886_c(EntityPlayer player) {}

    @Override
    public boolean func_94041_b(int slot, ItemStack stack) {
        return true;
    }

    public void rpcSetPatternFromString(final String name) {
        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "setPattern", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                NetworkUtils.writeUTF(data, name);
            }
        }));
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        super.receiveCommand(command, side, sender, stream);
        if (side.isServer()) {
            if ("setPattern".equals(command)) {
                // You cannot set the pattern if it is locked
                if (isPatternLocked()) return;
                String name = NetworkUtils.readUTF(stream);
                setPattern((FillerPattern) FillerManager.registry.getPattern(name), false);

                done = false;
            } else if ("setParameters".equals(command)) {
                // You cannot set the pattern if it is locked
                if (isPatternLocked()) return;
                NBTTagCompound patternData = NetworkUtils.readNBT(stream);
                readParametersFromNBT(patternData);

                currentTemplate = null;
                done = false;
            } else if ("setFlags".equals(command)) {
                excavate = stream.readBoolean();
                currentTemplate = null;

                sendNetworkUpdate();
                done = false;
            }
        }
    }

    @Override
    public Box getBox() {
        return box;
    }

    @Override
    public AxisAlignedBB getRenderBoundingBox() {
        return new Box(this).extendToEncompass(box).expand(50).getBoundingBox();
    }

    @Override
    public boolean isBuildingMaterialSlot(int i) {
        return true;
    }

    @Override
    public boolean acceptsControlMode(Mode mode) {
        return mode == IControllable.Mode.On || mode == IControllable.Mode.Off || mode == IControllable.Mode.Loop;
    }

    @Override
    public TileEntity getTile() {
        return this;
    }

    public void rpcSetParameter(int i, IStatementParameter patternParameter) {
        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "setParameters", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                NBTTagCompound parameterData = new NBTTagCompound();
                writeParametersToNBT(parameterData);
                NetworkUtils.writeNBT(data, parameterData);
            }
        }));
    }

    private boolean isDone() {
        return done;
    }

    private void setDone(boolean done) {
        this.done = done;
        if (field_145850_b != null) {
            field_145850_b.func_175656_a(field_174879_c, field_145850_b.func_180495_p(field_174879_c).func_177226_a(BuildCraftProperties.LED_DONE, done));
        }
    }

    public void setExcavate(boolean excavate) {
        this.excavate = excavate;
    }
}
