/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.builders;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import net.minecraft.block.Block;
import net.minecraft.block.BlockLiquid;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.*;
import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.world.ChunkCoordIntPair;

import net.minecraftforge.common.ForgeChunkManager;
import net.minecraftforge.common.ForgeChunkManager.Ticket;
import net.minecraftforge.common.ForgeChunkManager.Type;
import net.minecraftforge.common.util.FakePlayer;
import net.minecraftforge.fluids.IFluidBlock;

import buildcraft.BuildCraftBuilders;
import buildcraft.BuildCraftCore;
import buildcraft.api.blueprints.BuilderAPI;
import buildcraft.api.core.BuildCraftAPI;
import buildcraft.api.core.IAreaProvider;
import buildcraft.api.core.SafeTimeTracker;
import buildcraft.api.properties.BuildCraftProperties;
import buildcraft.api.tiles.IControllable;
import buildcraft.api.tiles.IDebuggable;
import buildcraft.api.tiles.IHasWork;
import buildcraft.api.transport.IPipeConnection;
import buildcraft.api.transport.IPipeTile;
import buildcraft.core.Box;
import buildcraft.core.Box.Kind;
import buildcraft.core.CoreConstants;
import buildcraft.core.DefaultAreaProvider;
import buildcraft.core.blueprints.Blueprint;
import buildcraft.core.blueprints.BptBuilderBase;
import buildcraft.core.blueprints.BptBuilderBlueprint;
import buildcraft.core.builders.TileAbstractBuilder;
import buildcraft.core.internal.IDropControlInventory;
import buildcraft.core.lib.RFBattery;
import buildcraft.core.lib.utils.BCStringUtils;
import buildcraft.core.lib.utils.BlockMiner;
import buildcraft.core.lib.utils.BlockUtils;
import buildcraft.core.lib.utils.Utils;
import buildcraft.core.lib.utils.Utils.EnumAxisOrder;
import buildcraft.core.proxy.CoreProxy;

import io.netty.buffer.ByteBuf;

public class TileQuarry extends TileAbstractBuilder implements IHasWork, ISidedInventory, IDropControlInventory, IPipeConnection, IControllable,
        IDebuggable {

    private static enum Stage {
        BUILDING,
        DIGGING,
        MOVING,
        IDLE,
        DONE
    }

    public EntityMechanicalArm arm;
    public EntityPlayer placedBy;

    /** The box for the setup */
    protected Box box = new Box();
    /** The box of what will be mined. Goes from y=0 to the very top layer of the mining box */
    private Box miningBox = new Box();
    private BlockPos target = BlockPos.field_177992_a;
    private Vec3 headPos = Utils.VEC_ZERO;
    private double speed = 0.03;
    private Stage stage = Stage.BUILDING;
    private boolean movingHorizontally;
    private boolean movingVertically;
    private float headTrajectory;

    private SafeTimeTracker updateTracker = new SafeTimeTracker(BuildCraftCore.updateFactor);

    private BptBuilderBase builder;

    private final LinkedList<int[]> visitList = Lists.newLinkedList();

    private boolean loadDefaultBoundaries = false;
    private Ticket chunkTicket;

    private boolean frameProducer = true;

    private NBTTagCompound initNBT = null;

    private BlockMiner miner;
    private int ledState;

    // TMP
    private int buildCallsS = 0, buildCallsF = 0;

    public TileQuarry() {
        box.kind = Kind.STRIPES;
        this.setBattery(new RFBattery((int) (2 * 64 * BuilderAPI.BREAK_ENERGY * BuildCraftCore.miningMultiplier), (int) (1000
            * BuildCraftCore.miningMultiplier), 0));
    }

    public void createUtilsIfNeeded() {
        if (!field_145850_b.field_72995_K) {
            if (builder == null) {
                if (!box.isInitialized()) {
                    setBoundaries(loadDefaultBoundaries);
                }

                initializeBlueprintBuilder();
            }
        }

        if (getStage() != Stage.BUILDING) {
            box.isVisible = false;

            if (arm == null) {
                createArm();
            }

            if (miningBox == null || !miningBox.isInitialized()) {
                miningBox = new Box(box.min(), box.max());
                miningBox.contract(1);
                miningBox.setMin(Utils.withValue(miningBox.min(), Axis.Y, 0));
                miningBox.setMax(miningBox.max().func_177982_a(0, 1, 0));
            }

            if (findTarget(false)) {
                AxisAlignedBB union = miningBox.getBoundingBox().func_111270_a(box.getBoundingBox());
                if (!union.func_72318_a(headPos)) {
                    headPos = Utils.withValue(headPos, Axis.Y, miningBox.max().func_177956_o() - 2);
                    BlockPos nearestPos = miningBox.closestInsideTo(Utils.convertFloor(headPos));
                    headPos = Utils.convert(nearestPos);
                }
            }
        } else {
            box.isVisible = true;
        }
    }

    private void createArm() {
        Vec3 vec = Utils.convert(box.min()).func_178787_e(Utils.vec3(CoreConstants.PIPE_MAX_POS));
        vec = vec.func_178787_e(Utils.convert(EnumFacing.UP, box.size().func_177956_o() - 1.5));

        double width = box.size().func_177958_n() - 2 + CoreConstants.PIPE_MIN_POS * 2;
        double height = box.size().func_177952_p() - 2 + CoreConstants.PIPE_MIN_POS * 2;

        field_145850_b.func_72838_d(new EntityMechanicalArm(field_145850_b, vec, width, height, this));
    }

    // Callback from the arm once it's created
    public void setArm(EntityMechanicalArm arm) {
        this.arm = arm;
    }

    public boolean areChunksLoaded() {
        if (BuildCraftBuilders.quarryLoadsChunks) {
            // Small optimization
            return true;
        }

        return Utils.checkChunksExist(field_145850_b, box.min(), box.max());
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            if (getStage() != Stage.DONE) {
                moveHead(speed);
            }

            return;
        }

        if (getStage() == Stage.DONE) {
            if (mode == Mode.Loop) {
                setStage(Stage.IDLE);
            } else {
                return;
            }
        }

        if (!areChunksLoaded()) {
            return;
        }

        if (mode == Mode.Off && getStage() != Stage.MOVING) {
            return;
        }

        createUtilsIfNeeded();
        if (getStage() == Stage.BUILDING) {
            if (builder != null && !builder.isDone(this)) {
                if (builder.buildNextSlot(field_145850_b, this)) buildCallsS++;
                else buildCallsF++;
            } else {
                setStage(Stage.IDLE);
            }
        } else if (getStage() == Stage.DIGGING) {
            dig();
        } else if (getStage() == Stage.IDLE) {
            idling();

            // We are sending a network packet update ONLY below.
            // In this case, since idling() does it anyway, we should return.
            return;
        } else if (stage == Stage.MOVING) {
            int energyUsed = this.getBattery().useEnergy(20, (int) Math.ceil(20D + (double) getBattery().getEnergyStored() / 10), false);

            if (energyUsed >= 20) {

                speed = 0.1 + energyUsed / 2000F;

                // If it's raining or snowing above the head, slow down.
                if (field_145850_b.func_72896_J()) {
                    if (field_145850_b.func_175645_m(Utils.convertFloor(headPos)).func_177956_o() < headPos.field_72448_b) {
                        speed *= 0.7;
                    }
                }

                moveHead(speed);
            } else {
                speed = 0;
            }
        }

        if (updateTracker.markTimeIfDelay(field_145850_b)) {
            sendNetworkUpdate();
        }
    }

    protected void dig() {
        if (field_145850_b.field_72995_K) {
            return;
        }

        if (miner == null) {
            // Hmm. Probably shouldn't be mining if there's no miner.
            stage = Stage.IDLE;
            return;
        }

        int rfTaken = miner.acceptEnergy(getBattery().getEnergyStored());
        getBattery().useEnergy(rfTaken, rfTaken, false);

        if (miner.hasMined()) {
            // Collect any lost items laying around.
            double[] head = getHead();
            AxisAlignedBB axis = new AxisAlignedBB(head[0] - 2, head[1] - 2, head[2] - 2, head[0] + 3, head[1] + 3, head[2] + 3);
            List<EntityItem> result = field_145850_b.func_72872_a(EntityItem.class, axis);
            for (EntityItem entity : result) {
                if (entity.field_70128_L) {
                    continue;
                }

                ItemStack mineable = entity.func_92059_d();
                if (mineable.field_77994_a <= 0) {
                    continue;
                }
                CoreProxy.proxy.removeEntity(entity);
                miner.mineStack(mineable);
            }
            setStage(Stage.IDLE);
            miner = null;
            return;
        }

        if (!findFrame()) {
            initializeBlueprintBuilder();
            stage = Stage.BUILDING;
        } else if (miner.hasFailed()) {
            setStage(Stage.IDLE);
            miner = null;
        }
    }

    protected boolean findFrame() {
        return field_145850_b.func_180495_p(box.min()).func_177230_c() == BuildCraftBuilders.frameBlock;
    }

    protected void idling() {
        if (!findTarget(true)) {
            // I believe the issue is box going null becuase of bad chunkloader positioning
            if (arm != null && box != null) {
                setTarget(new BlockPos(box.min().func_177958_n() + 1, field_174879_c.func_177956_o() + 2, box.min().func_177952_p() + 1));
            }

            setStage(Stage.DONE);
        } else {
            setStage(Stage.MOVING);
        }

        movingHorizontally = true;
        movingVertically = true;
        double[] head = getHead();
        int[] target = getTarget();
        headTrajectory = (float) Math.atan2(target[2] - head[2], target[0] - head[0]);
        sendNetworkUpdate();
    }

    public boolean findTarget(boolean doSet) {
        if (field_145850_b.field_72995_K) {
            return false;
        }

        boolean columnVisitListIsUpdated = false;

        if (visitList.isEmpty()) {
            createColumnVisitList();
            columnVisitListIsUpdated = true;
        }

        if (!doSet) {
            return !visitList.isEmpty();
        }

        if (visitList.isEmpty()) {
            return false;
        }

        int[] nextTarget = visitList.removeFirst();

        if (!columnVisitListIsUpdated) { // nextTarget may not be accurate, at least search the target column for
                                         // changes
            for (int y = nextTarget[1] + 1; y < func_174877_v().func_177956_o() + 3; y++) {
                if (isQuarriableBlock(new BlockPos(nextTarget[0], y, nextTarget[2]))) {
                    createColumnVisitList();
                    columnVisitListIsUpdated = true;
                    nextTarget = null;
                    break;
                }
            }
        }

        if (columnVisitListIsUpdated && nextTarget == null && !visitList.isEmpty()) {
            nextTarget = visitList.removeFirst();
        } else if (columnVisitListIsUpdated && nextTarget == null) {
            return false;
        }

        setTarget(new BlockPos(nextTarget[0], nextTarget[1] + 1, nextTarget[2]));

        return true;
    }

    /** Make the column visit list: called once per layer */
    private void createColumnVisitList() {
        visitList.clear();
        boolean[][] blockedColumns = new boolean[builder.blueprint.size.func_177958_n() - 2][builder.blueprint.size.func_177952_p() - 2];

        for (int searchY = field_174879_c.func_177956_o() + 3; searchY >= 1; --searchY) {
            int startX, endX, incX;

            if (searchY % 2 == 0) {
                startX = 0;
                endX = builder.blueprint.size.func_177958_n() - 2;
                incX = 1;
            } else {
                startX = builder.blueprint.size.func_177958_n() - 3;
                endX = -1;
                incX = -1;
            }

            for (int searchX = startX; searchX != endX; searchX += incX) {
                int startZ, endZ, incZ;

                if (searchX % 2 == searchY % 2) {
                    startZ = 0;
                    endZ = builder.blueprint.size.func_177952_p() - 2;
                    incZ = 1;
                } else {
                    startZ = builder.blueprint.size.func_177952_p() - 3;
                    endZ = -1;
                    incZ = -1;
                }

                for (int searchZ = startZ; searchZ != endZ; searchZ += incZ) {
                    if (!blockedColumns[searchX][searchZ]) {
                        int bx = box.min().func_177958_n() + searchX + 1;
                        int by = searchY;
                        int bz = box.min().func_177952_p() + searchZ + 1;

                        BlockPos pos = new BlockPos(bx, by, bz);
                        IBlockState state = field_145850_b.func_180495_p(pos);
                        Block block = state.func_177230_c();

                        if (!BlockUtils.canChangeBlock(state, field_145850_b, pos)) {
                            blockedColumns[searchX][searchZ] = true;
                        } else if (!BuildCraftAPI.isSoftBlock(field_145850_b, pos) && !(block instanceof BlockLiquid) && !(block instanceof IFluidBlock)) {
                            visitList.add(new int[] { bx, by, bz });
                        }

                        // Stop at two planes - generally any obstructions will have been found and will force a
                        // recompute prior to this
                        if (visitList.size() > builder.blueprint.size.func_177952_p() * builder.blueprint.size.func_177958_n() * 2) {
                            return;
                        }
                    }
                }
            }
        }

    }

    @Override
    public void func_145839_a(NBTTagCompound nbttagcompound) {
        super.func_145839_a(nbttagcompound);

        if (nbttagcompound.func_74764_b("box")) {
            box.initialize(nbttagcompound.func_74775_l("box"));

            loadDefaultBoundaries = false;
        } else if (nbttagcompound.func_74764_b("xSize")) {
            // This is a legacy save, get old data

            int xMin = nbttagcompound.func_74762_e("xMin");
            int zMin = nbttagcompound.func_74762_e("zMin");

            int xSize = nbttagcompound.func_74762_e("xSize");
            int ySize = nbttagcompound.func_74762_e("ySize");
            int zSize = nbttagcompound.func_74762_e("zSize");

            box.reset();
            box.setMin(new BlockPos(xMin, field_174879_c.func_177956_o(), zMin));
            box.setMax(new BlockPos(xMin + xSize - 1, field_174879_c.func_177956_o() + ySize - 1, zMin + zSize - 1));

            loadDefaultBoundaries = false;
        } else {
            // This is a legacy save, compute boundaries

            loadDefaultBoundaries = true;
        }

        int targetX = nbttagcompound.func_74762_e("targetX");
        int targetY = nbttagcompound.func_74762_e("targetY");
        int targetZ = nbttagcompound.func_74762_e("targetZ");
        target = new BlockPos(targetX, targetY, targetZ);

        double headPosX = nbttagcompound.func_74769_h("headPosX");
        double headPosY = nbttagcompound.func_74769_h("headPosY");
        double headPosZ = nbttagcompound.func_74769_h("headPosZ");
        headPos = new Vec3(headPosX, headPosY, headPosZ);

        // The rest of load has to be done upon initialize.
        initNBT = (NBTTagCompound) nbttagcompound.func_74775_l("bpt").func_74737_b();
    }

    @Override
    public void func_145841_b(NBTTagCompound nbttagcompound) {
        super.func_145841_b(nbttagcompound);

        nbttagcompound.func_74768_a("targetX", target.func_177958_n());
        nbttagcompound.func_74768_a("targetY", target.func_177956_o());
        nbttagcompound.func_74768_a("targetZ", target.func_177952_p());
        nbttagcompound.func_74780_a("headPosX", headPos.field_72450_a);
        nbttagcompound.func_74780_a("headPosY", headPos.field_72448_b);
        nbttagcompound.func_74780_a("headPosZ", headPos.field_72449_c);

        NBTTagCompound boxTag = new NBTTagCompound();
        box.writeToNBT(boxTag);
        nbttagcompound.func_74782_a("box", boxTag);

        NBTTagCompound bptNBT = new NBTTagCompound();

        if (builder != null) {
            NBTTagCompound builderCpt = new NBTTagCompound();
            builder.saveBuildStateToNBT(builderCpt, this);
            bptNBT.func_74782_a("builderState", builderCpt);
        }

        nbttagcompound.func_74782_a("bpt", bptNBT);
    }

    public void positionReached() {
        if (field_145850_b.field_72995_K) {
            return;
        }

        BlockPos pos = target.func_177977_b();
        if (isQuarriableBlock(pos)) {
            miner = new BlockMiner(field_145850_b, this, pos);
            setStage(Stage.DIGGING);
        } else {
            setStage(Stage.IDLE);
        }
    }

    private boolean isQuarriableBlock(BlockPos pos) {
        IBlockState state = field_145850_b.func_180495_p(pos);
        Block block = state.func_177230_c();
        return BlockUtils.canChangeBlock(state, field_145850_b, pos) && !BuildCraftAPI.isSoftBlock(field_145850_b, pos) && !(block instanceof BlockLiquid)
            && !(block instanceof IFluidBlock);
    }

    @Override
    public void func_145843_s() {
        if (chunkTicket != null) {
            ForgeChunkManager.releaseTicket(chunkTicket);
        }

        super.func_145843_s();
        destroy();
    }

    @Override
    public void onChunkUnload() {
        destroy();
    }

    @Override
    public void destroy() {
        if (arm != null) {
            arm.func_70106_y();
        }

        arm = null;

        frameProducer = false;

        if (miner != null) {
            miner.invalidate();
        }
    }

    @Override
    public boolean hasWork() {
        return getStage() != Stage.DONE;
    }

    private Stage getStage() {
        return stage;
    }

    private void setStage(Stage stage) {
        this.stage = stage;
        IBlockState state = field_145850_b.func_180495_p(field_174879_c);
        if (stage == Stage.DONE) {
            field_145850_b.func_175656_a(field_174879_c, state.func_177226_a(BuildCraftProperties.LED_DONE, true));
        } else if (state.func_177229_b(BuildCraftProperties.LED_DONE) == true) {
            field_145850_b.func_175656_a(field_174879_c, state.func_177226_a(BuildCraftProperties.LED_DONE, false));
        }
    }

    private void setBoundaries(boolean useDefaultI) {
        boolean useDefault = useDefaultI;

        if (BuildCraftBuilders.quarryLoadsChunks && chunkTicket == null) {
            chunkTicket = ForgeChunkManager.requestTicket(BuildCraftBuilders.instance, field_145850_b, Type.NORMAL);
        }

        if (chunkTicket != null) {
            chunkTicket.getModData().func_74768_a("quarryX", field_174879_c.func_177958_n());
            chunkTicket.getModData().func_74768_a("quarryY", field_174879_c.func_177956_o());
            chunkTicket.getModData().func_74768_a("quarryZ", field_174879_c.func_177952_p());
            ForgeChunkManager.forceChunk(chunkTicket, new ChunkCoordIntPair(field_174879_c.func_177958_n() >> 4, field_174879_c.func_177952_p() >> 4));
        }

        IAreaProvider a = null;

        if (!useDefault) {
            a = Utils.getNearbyAreaProvider(field_145850_b, field_174879_c);
        }

        if (a == null) {
            a = new DefaultAreaProvider(field_174879_c, field_174879_c.func_177971_a(new BlockPos(10, 4, 10)));

            useDefault = true;
        }

        int xSize = a.max().getX() - a.min().getX() + 1;
        int zSize = a.max().getZ() - a.min().getZ() + 1;

        if (xSize < 3 || zSize < 3 || (chunkTicket != null && ((xSize * zSize) >> 8) >= chunkTicket.getMaxChunkListDepth())) {
            if (placedBy != null) {
                placedBy.func_145747_a(new ChatComponentTranslation("chat.buildcraft.quarry.tooSmall", xSize, zSize, chunkTicket != null
                    ? chunkTicket.getMaxChunkListDepth() : 0));
            }

            a = new DefaultAreaProvider(field_174879_c, field_174879_c.func_177971_a(new BlockPos(10, 4, 10)));
            useDefault = true;
        }

        xSize = a.max().getX() - a.min().getX() + 1;
        int ySize = a.max().getY() - a.min().getY() + 1;
        zSize = a.max().getZ() - a.min().getZ() + 1;

        box.initialize(a);

        if (ySize < 5) {
            ySize = 5;
            box.setMax(Utils.withValue(box.max(), Axis.Y, box.min().func_177956_o() + ySize - 1));
        }

        if (useDefault) {
            int xMin, zMin;

            EnumFacing face = field_145850_b.func_180495_p(field_174879_c).func_177229_b(BuildCraftProperties.BLOCK_FACING).getOpposite();

            switch (face) {
                case EAST:
                    xMin = field_174879_c.func_177958_n() + 1;
                    zMin = field_174879_c.func_177952_p() - 4 - 1;
                    break;
                case WEST:
                    xMin = field_174879_c.func_177958_n() - 9 - 2;
                    zMin = field_174879_c.func_177952_p() - 4 - 1;
                    break;
                case SOUTH:
                    xMin = field_174879_c.func_177958_n() - 4 - 1;
                    zMin = field_174879_c.func_177952_p() + 1;

                    break;
                case NORTH:
                default:
                    xMin = field_174879_c.func_177958_n() - 4 - 1;
                    zMin = field_174879_c.func_177952_p() - 9 - 2;
                    break;
            }

            box.reset();
            box.setMin(new BlockPos(xMin, field_174879_c.func_177956_o(), zMin));
            box.setMax(new BlockPos(xMin + xSize - 1, field_174879_c.func_177956_o() + ySize - 1, zMin + zSize - 1));
        }

        a.removeFromWorld();
        if (chunkTicket != null) {
            forceChunkLoading(chunkTicket);
        }

        sendNetworkUpdate();
    }

    private void initializeBlueprintBuilder() {
        PatternQuarryFrame pqf = PatternQuarryFrame.INSTANCE;

        Blueprint bpt = pqf.getBlueprint(box, field_145850_b);
        builder = new BptBuilderBlueprint(bpt, field_145850_b, box.min());
        builder.setOrder(new Utils.AxisOrder(EnumAxisOrder.XZY, true, true, false));
        speed = 0;
        stage = Stage.BUILDING;
        sendNetworkUpdate();
    }

    @Override
    public void writeData(ByteBuf stream) {
        super.writeData(stream);
        box.writeData(stream);
        stream.writeInt(target.func_177958_n());
        stream.writeShort(target.func_177956_o());
        stream.writeInt(target.func_177952_p());
        stream.writeDouble(headPos.field_72450_a);
        stream.writeDouble(headPos.field_72448_b);
        stream.writeDouble(headPos.field_72449_c);
        stream.writeFloat((float) speed);
        stream.writeFloat(headTrajectory);
        int flags = stage.ordinal();
        flags |= movingHorizontally ? 0x10 : 0;
        flags |= movingVertically ? 0x20 : 0;
        stream.writeByte(flags);
        ledState = (getBattery().getEnergyStored() * 3 / getBattery().getMaxEnergyStored());
        stream.writeByte(ledState);
    }

    @Override
    public void readData(ByteBuf stream) {
        super.readData(stream);
        box.readData(stream);
        int targetX = stream.readInt();
        int targetY = stream.readUnsignedShort();
        int targetZ = stream.readInt();
        target = new BlockPos(targetX, targetY, targetZ);

        double headPosX = stream.readDouble();
        double headPosY = stream.readDouble();
        double headPosZ = stream.readDouble();
        headPos = new Vec3(headPosX, headPosY, headPosZ);

        speed = stream.readFloat();
        headTrajectory = stream.readFloat();
        int flags = stream.readUnsignedByte();
        setStage(Stage.values()[flags & 0x07]);
        movingHorizontally = (flags & 0x10) != 0;
        movingVertically = (flags & 0x20) != 0;
        ledState = stream.readUnsignedByte();

        createUtilsIfNeeded();

        if (arm != null) {
            arm.setHead(headPos);
            arm.updatePosition();
        }
    }

    @Override
    public void initialize() {
        super.initialize();

        if (!this.func_145831_w().field_72995_K && !box.isInitialized()) {
            setBoundaries(false);
        }

        createUtilsIfNeeded();

        if (initNBT != null && builder != null) {
            builder.loadBuildStateToNBT(initNBT.func_74775_l("builderState"), this);
        }

        initNBT = null;

        sendNetworkUpdate();
    }

    public void reinitalize() {
        initializeBlueprintBuilder();
    }

    @Override
    public int func_70302_i_() {
        return 1;
    }

    @Override
    public ItemStack func_70301_a(int i) {
        if (frameProducer) {
            return new ItemStack(BuildCraftBuilders.frameBlock);
        } else {
            return null;
        }
    }

    @Override
    public ItemStack func_70298_a(int i, int j) {
        if (frameProducer) {
            return new ItemStack(BuildCraftBuilders.frameBlock, j);
        } else {
            return null;
        }
    }

    @Override
    public void func_70299_a(int i, ItemStack itemstack) {}

    @Override
    public ItemStack func_70304_b(int slot) {
        return null;
    }

    @Override
    public String getInventoryName() {
        return "";
    }

    @Override
    public int func_70297_j_() {
        return 0;
    }

    @Override
    public boolean func_94041_b(int i, ItemStack itemstack) {
        return false;
    }

    @Override
    public boolean func_70300_a(EntityPlayer entityplayer) {
        return false;
    }

    @Override
    public void func_174889_b(EntityPlayer player) {}

    @Override
    public void func_174886_c(EntityPlayer player) {}

    @Override
    public boolean isBuildingMaterialSlot(int i) {
        return true;
    }

    public void moveHead(double instantSpeed) {
        int[] target = getTarget();
        double[] head = getHead();

        if (movingHorizontally) {
            if (Math.abs(target[0] - head[0]) < instantSpeed * 2 && Math.abs(target[2] - head[2]) < instantSpeed * 2) {
                head[0] = target[0];
                head[2] = target[2];

                movingHorizontally = false;

                if (!movingVertically) {
                    positionReached();
                    head[1] = target[1];
                }
            } else {
                head[0] += MathHelper.func_76134_b(headTrajectory) * instantSpeed;
                head[2] += MathHelper.func_76126_a(headTrajectory) * instantSpeed;
            }
            setHead(head[0], head[1], head[2]);
        }

        if (movingVertically) {
            if (Math.abs(target[1] - head[1]) < instantSpeed * 2) {
                head[1] = target[1];

                movingVertically = false;
                if (!movingHorizontally) {
                    positionReached();
                    head[0] = target[0];
                    head[2] = target[2];
                }
            } else {
                if (target[1] > head[1]) {
                    head[1] += instantSpeed;
                } else {
                    head[1] -= instantSpeed;
                }
            }
            setHead(head[0], head[1], head[2]);
        }

        updatePosition();
    }

    private void updatePosition() {
        if (arm != null && field_145850_b.field_72995_K) {
            arm.setHead(headPos);
            arm.updatePosition();
        }
    }

    private void setHead(double x, double y, double z) {
        headPos = new Vec3(x, y, z);
    }

    private double[] getHead() {
        return new double[] { headPos.field_72450_a, headPos.field_72448_b, headPos.field_72449_c };
    }

    private int[] getTarget() {
        return new int[] { target.func_177958_n(), target.func_177956_o(), target.func_177952_p() };
    }

    private void setTarget(BlockPos pos) {
        target = pos;
    }

    public void forceChunkLoading(Ticket ticket) {
        if (chunkTicket == null) {
            chunkTicket = ticket;
        }

        Set<ChunkCoordIntPair> chunks = Sets.newHashSet();
        ChunkCoordIntPair quarryChunk = new ChunkCoordIntPair(field_174879_c.func_177958_n() >> 4, field_174879_c.func_177952_p() >> 4);
        chunks.add(quarryChunk);
        ForgeChunkManager.forceChunk(ticket, quarryChunk);

        if (box.isInitialized()) {
            for (int chunkX = box.min().func_177958_n() >> 4; chunkX <= box.max().func_177958_n() >> 4; chunkX++) {
                for (int chunkZ = box.min().func_177952_p() >> 4; chunkZ <= box.max().func_177952_p() >> 4; chunkZ++) {
                    ChunkCoordIntPair chunk = new ChunkCoordIntPair(chunkX, chunkZ);
                    ForgeChunkManager.forceChunk(ticket, chunk);
                    chunks.add(chunk);
                }
            }
        }

        if (placedBy != null && !(placedBy instanceof FakePlayer)) {
            placedBy.func_145747_a(new ChatComponentTranslation("chat.buildcraft.quarry.chunkloadInfo", func_174877_v().func_177958_n(), func_174877_v().func_177956_o(), func_174877_v()
                    .func_177952_p(), chunks.size()));
        }
    }

    @Override
    public boolean hasCustomName() {
        return false;
    }

    @Override
    public AxisAlignedBB getRenderBoundingBox() {
        // return Utils.boundingBox(Utils.vec3(-100000d), Utils.vec3(100000d));
        if (func_174877_v() == null) return null;
        return new Box(this).extendToEncompass(box).expand(50).getBoundingBox();
    }

    @Override
    public Box getBox() {
        return box;
    }

    @Override
    public boolean acceptsControlMode(Mode mode) {
        return mode == Mode.Off || mode == Mode.On || mode == Mode.Loop;
    }

    @Override
    public boolean doDrop() {
        return false;
    }

    @Override
    public ConnectOverride overridePipeConnection(IPipeTile.PipeType type, EnumFacing with) {
        if (with == field_145850_b.func_180495_p(field_174879_c).func_177229_b(BuildCraftProperties.BLOCK_FACING)) {
            return ConnectOverride.DISCONNECT;
        }
        return type == IPipeTile.PipeType.ITEM ? ConnectOverride.CONNECT : ConnectOverride.DEFAULT;
    }

    @Override
    public int[] func_180463_a(EnumFacing side) {
        return new int[0];
    }

    @Override
    public boolean func_180462_a(int index, ItemStack itemStackIn, EnumFacing direction) {
        return false;
    }

    @Override
    public boolean func_180461_b(int index, ItemStack stack, EnumFacing direction) {
        return false;
    }

    @Override
    public void getDebugInfo(List<String> left, List<String> right, EnumFacing side) {
        TileQuarry server = CoreProxy.proxy.getServerTile(this);

        left.add("");
        left.add("  - IsServer = " + (server != this));
        left.add("  - Stage = " + server.getStage());
        left.add("  - Mode = " + server.mode);
        if (server.builder == null) {
            left.add("  - Builder = null");
        } else {
            left.add("  - Builder");
            left.add("    - IsDone = " + (server.builder.isDone(server)));
            left.add("    - Min = " + BCStringUtils.vec3ToDispString(server.builder.min()));
            left.add("    - Max = " + BCStringUtils.vec3ToDispString(server.builder.max()));
            left.add("    - Successes = " + server.buildCallsS);
            left.add("    - Failures = " + server.buildCallsF);
        }
        if (server.box == null || !server.box.isInitialized()) {
            left.add("  - BuildingBox = null");
        } else {
            left.add("  - BuildingBox");
            left.add("    - Min = " + BCStringUtils.vec3ToDispString(server.box.min()));
            left.add("    - Max = " + BCStringUtils.vec3ToDispString(server.box.max()));
        }
        if (server.miningBox == null || !server.miningBox.isInitialized()) {
            left.add("  - MiningBox = null");
        } else {
            left.add("  - MiningBox");
            left.add("    - Min = " + BCStringUtils.vec3ToDispString(server.miningBox.min()));
            left.add("    - Max = " + BCStringUtils.vec3ToDispString(server.miningBox.max()));
        }
        left.add("  - Head = " + BCStringUtils.vec3ToDispString(server.headPos));
        left.add("  - Target = " + BCStringUtils.vec3ToDispString(server.target));
    }
}
