/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core;

import java.util.ArrayList;
import java.util.List;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.World;

import net.minecraftforge.common.util.Constants;

import buildcraft.api.core.IAreaProvider;
import buildcraft.api.core.IBox;
import buildcraft.api.core.IPathProvider;
import buildcraft.api.core.IZone;
import buildcraft.api.items.IMapLocation;
import buildcraft.core.lib.items.ItemBuildCraft;
import buildcraft.core.lib.utils.ModelHelper;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.BCStringUtils;
import buildcraft.robotics.ZonePlan;

public class ItemMapLocation extends ItemBuildCraft implements IMapLocation {
    public ItemMapLocation() {
        super(BCCreativeTab.get("main"));
        func_77627_a(true);
    }

    @Override
    public int getItemStackLimit(ItemStack stack) {
        return MapLocationType.getFromStack(stack) == MapLocationType.CLEAN ? 16 : 1;
    }

    @Override
    public void func_77624_a(ItemStack stack, EntityPlayer player, @SuppressWarnings("rawtypes") List list, boolean advanced) {
        @SuppressWarnings("unchecked")
        List<String> strings = list;
        NBTTagCompound cpt = NBTUtils.getItemData(stack);

        if (cpt.func_74764_b("name")) {
            String name = cpt.func_74779_i("name");
            if (name.length() > 0) {
                strings.add(name);
            }
        }

        MapLocationType type = MapLocationType.getFromStack(stack);
        switch (type) {
            case SPOT: {
                int x = cpt.func_74762_e("x");
                int y = cpt.func_74762_e("y");
                int z = cpt.func_74762_e("z");
                EnumFacing side = EnumFacing.values()[cpt.func_74771_c("side")];

                strings.add(BCStringUtils.localize("{" + x + ", " + y + ", " + z + ", " + side + "}"));
                break;
            }
            case AREA: {
                int x = cpt.func_74762_e("xMin");
                int y = cpt.func_74762_e("yMin");
                int z = cpt.func_74762_e("zMin");
                int xLength = cpt.func_74762_e("xMax") - x + 1;
                int yLength = cpt.func_74762_e("yMax") - y + 1;
                int zLength = cpt.func_74762_e("zMax") - z + 1;

                strings.add(BCStringUtils.localize("{" + x + ", " + y + ", " + z + "} + {" + xLength + " x " + yLength + " x " + zLength + "}"));
                break;
            }
            case PATH: {
                NBTTagList pathNBT = cpt.func_150295_c("path", Constants.NBT.TAG_COMPOUND);
                BlockPos first = NBTUtils.readBlockPos(pathNBT);

                int x = first.func_177958_n();
                int y = first.func_177956_o();
                int z = first.func_177952_p();

                strings.add(BCStringUtils.localize("{" + x + ", " + y + ", " + z + "} + " + pathNBT.func_74745_c() + " elements"));
                break;
            }
            default: {
                break;
            }
        }
    }

    @Override
    public boolean onItemUseFirst(ItemStack stack, EntityPlayer player, World world, BlockPos pos, EnumFacing side, float par8, float par9,
            float par10) {
        if (world.field_72995_K) {
            return false;
        }

        TileEntity tile = world.func_175625_s(pos);
        NBTTagCompound cpt = NBTUtils.getItemData(stack);

        if (tile instanceof IPathProvider) {
            MapLocationType.PATH.setToStack(stack);

            NBTTagList pathNBT = new NBTTagList();

            for (BlockPos index : ((IPathProvider) tile).getPath()) {
                pathNBT.func_74742_a(NBTUtils.writeBlockPos(index));
            }

            cpt.func_74782_a("path", pathNBT);
        } else if (tile instanceof IAreaProvider) {
            MapLocationType.AREA.setToStack(stack);

            IAreaProvider areaTile = (IAreaProvider) tile;

            cpt.func_74768_a("xMin", areaTile.min().getX());
            cpt.func_74768_a("yMin", areaTile.min().getY());
            cpt.func_74768_a("zMin", areaTile.min().getZ());
            cpt.func_74768_a("xMax", areaTile.max().getX());
            cpt.func_74768_a("yMax", areaTile.max().getY());
            cpt.func_74768_a("zMax", areaTile.max().getZ());

        } else {
            MapLocationType.SPOT.setToStack(stack);

            cpt.func_74774_a("side", (byte) side.func_176745_a());
            cpt.func_74768_a("x", pos.func_177958_n());
            cpt.func_74768_a("y", pos.func_177956_o());
            cpt.func_74768_a("z", pos.func_177952_p());
        }

        return true;
    }

    @Override
    public IBox getBox(ItemStack item) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType type = MapLocationType.getFromStack(item);

        switch (type) {
            case AREA: {
                int xMin = cpt.func_74762_e("xMin");
                int yMin = cpt.func_74762_e("yMin");
                int zMin = cpt.func_74762_e("zMin");
                BlockPos min = new BlockPos(xMin, yMin, zMin);

                int xMax = cpt.func_74762_e("xMax");
                int yMax = cpt.func_74762_e("yMax");
                int zMax = cpt.func_74762_e("zMax");
                BlockPos max = new BlockPos(xMax, yMax, zMax);

                return new Box(min, max);
            }
            case SPOT: {
                return getPointBox(item);
            }
            default: {
                return null;
            }
        }
    }

    public static IBox getPointBox(ItemStack item) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType type = MapLocationType.getFromStack(item);

        switch (type) {
            case SPOT: {
                int x = cpt.func_74762_e("x");
                int y = cpt.func_74762_e("y");
                int z = cpt.func_74762_e("z");

                BlockPos pos = new BlockPos(x, y, z);

                return new Box(pos, pos);
            }
            default: {
                return null;
            }
        }
    }

    @Override
    public EnumFacing getPointSide(ItemStack item) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType type = MapLocationType.getFromStack(item);

        if (type == MapLocationType.SPOT) {
            return EnumFacing.values()[cpt.func_74771_c("side")];
        } else {
            return null;
        }
    }

    @Override
    public BlockPos getPoint(ItemStack item) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType type = MapLocationType.getFromStack(item);

        if (type == MapLocationType.SPOT) {
            return new BlockPos(cpt.func_74762_e("x"), cpt.func_74762_e("y"), cpt.func_74762_e("z"));
        } else {
            return null;
        }
    }

    @Override
    public IZone getZone(ItemStack item) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType type = MapLocationType.getFromStack(item);
        switch (type) {
            case ZONE: {
                ZonePlan plan = new ZonePlan();
                plan.readFromNBT(cpt);
                return plan;
            }
            case AREA: {
                return getBox(item);
            }
            case PATH: {
                return getPointBox(item);
            }
            default: {
                return null;
            }
        }
    }

    @Override
    public List<BlockPos> getPath(ItemStack item) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType type = MapLocationType.getFromStack(item);
        switch (type) {
            case PATH: {
                List<BlockPos> indexList = new ArrayList<>();
                NBTTagList pathNBT = cpt.func_150295_c("path", Constants.NBT.TAG_COMPOUND);
                for (int i = 0; i < pathNBT.func_74745_c(); i++) {
                    indexList.add(NBTUtils.readBlockPos(pathNBT.func_150305_b(i)));
                }
                return indexList;
            }
            case SPOT: {
                List<BlockPos> indexList = new ArrayList<>();
                indexList.add(new BlockPos(cpt.func_74762_e("x"), cpt.func_74762_e("y"), cpt.func_74762_e("z")));
                return indexList;
            }
            default: {
                return null;
            }
        }
    }

    public static void setZone(ItemStack item, ZonePlan plan) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        MapLocationType.ZONE.setToStack(item);
        plan.writeToNBT(cpt);
    }

    @Override
    public String getName(ItemStack item) {
        return NBTUtils.getItemData(item).func_74779_i("name");
    }

    @Override
    public boolean setName(ItemStack item, String name) {
        NBTTagCompound cpt = NBTUtils.getItemData(item);
        cpt.func_74778_a("name", name);
        return true;
    }

    @Override
    public void registerModels() {
        ModelHelper.registerItemModel(this, MapLocationType.CLEAN.meta, "/clean");
        ModelHelper.registerItemModel(this, MapLocationType.SPOT.meta, "/spot");
        ModelHelper.registerItemModel(this, MapLocationType.AREA.meta, "/area");
        ModelHelper.registerItemModel(this, MapLocationType.PATH.meta, "/path");
        ModelHelper.registerItemModel(this, MapLocationType.ZONE.meta, "/zone");
    }
}
