/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core;

import net.minecraft.block.state.IBlockState;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.util.Vec3;
import net.minecraft.world.World;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.ISerializable;
import buildcraft.api.tiles.ITileAreaProvider;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class TileMarker extends TileBuildCraft implements ITileAreaProvider {
    public static class TileWrapper implements ISerializable {

        public int x, y, z;
        private TileMarker marker;

        public TileWrapper() {
            x = Integer.MAX_VALUE;
            y = Integer.MAX_VALUE;
            z = Integer.MAX_VALUE;
        }

        public TileWrapper(BlockPos pos) {
            this.x = pos.func_177958_n();
            this.y = pos.func_177956_o();
            this.z = pos.func_177952_p();
        }

        public boolean isSet() {
            return x != Integer.MAX_VALUE;
        }

        public TileMarker getMarker(World world) {
            if (!isSet()) {
                return null;
            }

            if (marker == null) {
                TileEntity tile = world.func_175625_s(new BlockPos(x, y, z));
                if (tile instanceof TileMarker) {
                    marker = (TileMarker) tile;
                }
            }

            return marker;
        }

        public void reset() {
            x = Integer.MAX_VALUE;
            y = Integer.MAX_VALUE;
            z = Integer.MAX_VALUE;
        }

        @Override
        public void readData(ByteBuf stream) {
            x = stream.readInt();
            if (isSet()) {
                y = stream.readShort();
                z = stream.readInt();
            }
        }

        @Override
        public void writeData(ByteBuf stream) {
            stream.writeInt(x);
            if (isSet()) {
                // Only X is used for checking if a vector is set, so we can save space on the Y coordinate.
                stream.writeShort(y);
                stream.writeInt(z);
            }
        }

        public BlockPos getPos() {
            return new BlockPos(x, y, z);
        }
    }

    public static class Origin implements ISerializable {
        public TileWrapper vectO = new TileWrapper();
        public TileWrapper[] vect = { new TileWrapper(), new TileWrapper(), new TileWrapper() };
        public int xMin, yMin, zMin, xMax, yMax, zMax;

        public boolean isSet() {
            return vectO.isSet();
        }

        @Override
        public void writeData(ByteBuf stream) {
            vectO.writeData(stream);
            for (TileWrapper tw : vect) {
                tw.writeData(stream);
            }
            stream.writeInt(xMin);
            stream.writeShort(yMin);
            stream.writeInt(zMin);
            stream.writeInt(xMax);
            stream.writeShort(yMax);
            stream.writeInt(zMax);
        }

        @Override
        public void readData(ByteBuf stream) {
            vectO.readData(stream);
            for (TileWrapper tw : vect) {
                tw.readData(stream);
            }
            xMin = stream.readInt();
            yMin = stream.readShort();
            zMin = stream.readInt();
            xMax = stream.readInt();
            yMax = stream.readShort();
            zMax = stream.readInt();
        }

        public BlockPos min() {
            return new BlockPos(xMin, yMin, zMin);
        }

        public BlockPos max() {
            return new BlockPos(xMax, yMax, zMax);
        }
    }

    public Origin origin = new Origin();
    public boolean showSignals = false;

    private Vec3 initVectO;
    private Vec3[] initVect;
    public LaserData[] lasers;
    public LaserData[] signals;

    public void updateSignals() {
        if (!field_145850_b.field_72995_K) {
            showSignals = field_145850_b.func_175687_A(field_174879_c) > 0;
            sendNetworkUpdate();
        }
    }

    private void switchSignals() {
        signals = null;
        if (showSignals) {
            signals = new LaserData[6];
            Vec3 cPos = Utils.convert(field_174879_c);
            int rangePlus = DefaultProps.MARKER_RANGE + 1;
            int rangeMinus = DefaultProps.MARKER_RANGE - 1;
            Vec3 offset = Utils.VEC_HALF;
            if (!origin.isSet() || !origin.vect[0].isSet()) {
                signals[0] = new LaserData(cPos.func_178787_e(offset), cPos.func_178787_e(offset).func_72441_c(rangeMinus, 0, 0));
                signals[1] = new LaserData(cPos.func_178787_e(offset).func_72441_c(-rangePlus, 0, 0), cPos.func_178787_e(offset));
            }

            if (!origin.isSet() || !origin.vect[1].isSet()) {
                signals[2] = new LaserData(cPos.func_178787_e(offset), cPos.func_178787_e(offset).func_72441_c(0, rangeMinus, 0));
                signals[3] = new LaserData(cPos.func_178787_e(offset).func_72441_c(0, -rangePlus, 0), cPos.func_178787_e(offset));
            }

            if (!origin.isSet() || !origin.vect[2].isSet()) {
                signals[4] = new LaserData(cPos.func_178787_e(offset), cPos.func_178787_e(offset).func_72441_c(0, 0, rangeMinus));
                signals[5] = new LaserData(cPos.func_178787_e(offset).func_72441_c(0, 0, -rangePlus), cPos.func_178787_e(offset));
            }
        }
    }

    @Override
    public void initialize() {
        super.initialize();

        updateSignals();

        if (initVectO != null) {
            origin = new Origin();

            origin.vectO = new TileWrapper(Utils.convertFloor(initVectO));

            for (int i = 0; i < 3; ++i) {
                if (initVect[i] != null) {
                    linkTo((TileMarker) field_145850_b.func_175625_s(Utils.convertFloor(initVect[i])), i);
                }
            }
        }
    }

    public void tryConnection() {
        if (field_145850_b.field_72995_K) {
            return;
        }

        for (int j = 0; j < 3; ++j) {
            if (!origin.isSet() || !origin.vect[j].isSet()) {
                setVect(j);
            }
        }

        sendNetworkUpdate();
    }

    void setVect(int n) {
        int[] coords = new int[3];

        coords[0] = field_174879_c.func_177958_n();
        coords[1] = field_174879_c.func_177956_o();
        coords[2] = field_174879_c.func_177952_p();
        BlockPos coord = field_174879_c;

        if (!origin.isSet() || !origin.vect[n].isSet()) {
            for (int j = 1; j < DefaultProps.MARKER_RANGE; ++j) {
                coords[n] += j;
                coord = new BlockPos(coords[0], coords[1], coords[2]);

                TileEntity tile = field_145850_b.func_175625_s(coord);

                if (tile instanceof TileMarker) {
                    if (linkTo((TileMarker) tile, n)) {
                        break;
                    }
                }

                coords[n] -= j;
                coords[n] -= j;
                coord = new BlockPos(coords[0], coords[1], coords[2]);

                tile = field_145850_b.func_175625_s(coord);

                if (tile instanceof TileMarker) {
                    if (linkTo((TileMarker) tile, n)) {
                        break;
                    }
                }

                coords[n] += j;
            }
        }
    }

    private boolean linkTo(TileMarker marker, int n) {
        if (marker == null) {
            return false;
        }

        if (origin.isSet() && marker.origin.isSet()) {
            return false;
        }

        if (!origin.isSet() && !marker.origin.isSet()) {
            origin = new Origin();
            marker.origin = origin;
            origin.vectO = new TileWrapper(field_174879_c);
            origin.vect[n] = new TileWrapper(marker.field_174879_c);
        } else if (!origin.isSet()) {
            origin = marker.origin;
            origin.vect[n] = new TileWrapper(field_174879_c);
        } else {
            marker.origin = origin;
            origin.vect[n] = new TileWrapper(marker.field_174879_c);
        }

        origin.vectO.getMarker(field_145850_b).createLasers();
        updateSignals();
        marker.updateSignals();

        return true;
    }

    private void createLasers() {
        lasers = null;

        lasers = new LaserData[12];
        Origin o = origin;

        if (!origin.vect[0].isSet()) {
            o.xMin = origin.vectO.x;
            o.xMax = origin.vectO.x;
        } else if (origin.vect[0].x < field_174879_c.func_177958_n()) {
            o.xMin = origin.vect[0].x;
            o.xMax = field_174879_c.func_177958_n();
        } else {
            o.xMin = field_174879_c.func_177958_n();
            o.xMax = origin.vect[0].x;
        }

        if (!origin.vect[1].isSet()) {
            o.yMin = origin.vectO.y;
            o.yMax = origin.vectO.y;
        } else if (origin.vect[1].y < field_174879_c.func_177956_o()) {
            o.yMin = origin.vect[1].y;
            o.yMax = field_174879_c.func_177956_o();
        } else {
            o.yMin = field_174879_c.func_177956_o();
            o.yMax = origin.vect[1].y;
        }

        if (!origin.vect[2].isSet()) {
            o.zMin = origin.vectO.z;
            o.zMax = origin.vectO.z;
        } else if (origin.vect[2].z < field_174879_c.func_177952_p()) {
            o.zMin = origin.vect[2].z;
            o.zMax = field_174879_c.func_177952_p();
        } else {
            o.zMin = field_174879_c.func_177952_p();
            o.zMax = origin.vect[2].z;
        }

        lasers = Utils.createLaserDataBox(o.xMin + 0.5, o.yMin + 0.5, o.zMin + 0.5, o.xMax + 0.5, o.yMax + 0.5, o.zMax + 0.5);
    }

    @Override
    public BlockPos min() {
        if (origin.isSet()) return origin.min();
        return field_174879_c;
    }

    @Override
    public BlockPos max() {
        if (origin.isSet()) return origin.max();
        return field_174879_c;
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        destroy();
    }

    @Override
    public void destroy() {
        TileMarker markerOrigin = null;

        if (origin.isSet()) {
            markerOrigin = origin.vectO.getMarker(field_145850_b);

            Origin o = origin;

            if (markerOrigin != null && markerOrigin.lasers != null) {
                markerOrigin.lasers = null;
            }

            for (TileWrapper m : o.vect) {
                TileMarker mark = m.getMarker(field_145850_b);

                if (mark != null) {
                    mark.lasers = null;

                    if (mark != this) {
                        mark.origin = new Origin();
                    }
                }
            }

            if (markerOrigin != this && markerOrigin != null) {
                markerOrigin.origin = new Origin();
            }

            for (TileWrapper wrapper : o.vect) {
                TileMarker mark = wrapper.getMarker(field_145850_b);

                if (mark != null) {
                    mark.updateSignals();
                }
            }
            if (markerOrigin != null) {
                markerOrigin.updateSignals();
            }
        }

        signals = null;

        if (!field_145850_b.field_72995_K && markerOrigin != null && markerOrigin != this) {
            markerOrigin.sendNetworkUpdate();
        }
    }

    @Override
    public void removeFromWorld() {
        if (!origin.isSet()) {
            return;
        }

        Origin o = origin;

        for (TileWrapper m : o.vect.clone()) {
            if (m.isSet()) {
                IBlockState state = field_145850_b.func_180495_p(m.getPos());
                field_145850_b.func_175698_g(m.getPos());
                BuildCraftCore.markerBlock.func_176226_b(field_145850_b, m.getPos(), state, 0);
            }
        }

        IBlockState state = field_145850_b.func_180495_p(o.vectO.getPos());
        field_145850_b.func_175698_g(o.vectO.getPos());

        BuildCraftCore.markerBlock.func_176226_b(field_145850_b, o.vectO.getPos(), state, 0);
    }

    @Override
    public void func_145839_a(NBTTagCompound nbttagcompound) {
        super.func_145839_a(nbttagcompound);

        if (nbttagcompound.func_74764_b("vectO")) {
            initVectO = NBTUtils.readVec3(nbttagcompound, "vectO");
            initVect = new Vec3[3];

            for (int i = 0; i < 3; ++i) {
                if (nbttagcompound.func_74764_b("vect" + i)) {
                    initVect[i] = NBTUtils.readVec3(nbttagcompound, "vect" + i);
                }
            }
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);

        if (origin.isSet() && origin.vectO.getMarker(field_145850_b) == this) {
            nbt.func_74782_a("vectO", NBTUtils.writeVec3(Utils.convert(origin.vectO.getMarker(field_145850_b).field_174879_c)));

            for (int i = 0; i < 3; ++i) {
                if (origin.vect[i].isSet()) {
                    nbt.func_74782_a("vect" + i, NBTUtils.writeVec3(Utils.convert(origin.vect[i].getPos())));
                }
            }
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        origin.writeData(stream);
        stream.writeBoolean(showSignals);
    }

    @Override
    public void readData(ByteBuf stream) {
        origin.readData(stream);
        showSignals = stream.readBoolean();

        switchSignals();

        if (origin.vectO.isSet() && origin.vectO.getMarker(field_145850_b) != null) {
            origin.vectO.getMarker(field_145850_b).updateSignals();

            for (TileWrapper w : origin.vect) {
                TileMarker m = w.getMarker(field_145850_b);

                if (m != null) {
                    m.updateSignals();
                }
            }
        }

        createLasers();

    }

    @Override
    public boolean isValidFromLocation(BlockPos pos) {
        // Rules:
        // - one or two, but not three, of the coordinates must be equal to the marker's location
        // - one of the coordinates must be either -1 or 1 away
        // - it must be physically touching the box
        // - however, it cannot be INSIDE the box
        // int equal = (pos.getX() == this.pos.getX() ? 1 : 0) + (pos.getY() == this.pos.getY() ? 1 : 0) + (pos.getZ()
        // == this.pos.getZ() ? 1 : 0);
        // int touching = 0;
        //
        // if (equal == 0 || equal == 3) {
        // return false;
        // }

        Box box = new Box(min(), max());

        if (box.contains(pos)) return false;
        return box.distanceToSquared(pos) == 1;
    }
}
