/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.util.Vec3;
import net.minecraft.world.World;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.IPathProvider;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class TilePathMarker extends TileMarker implements IPathProvider {
    // A list with the pathMarkers that aren't fully connected
    // It only contains markers within the loaded chunks
    private static ArrayList<TilePathMarker> availableMarkers = new ArrayList<>();

    public int x0, y0, z0, x1, y1, z1;
    public boolean loadLink0 = false;
    public boolean loadLink1 = false;

    public LaserData[] lasers = new LaserData[2];
    public boolean tryingToConnect = false;

    public TilePathMarker[] links = new TilePathMarker[2];

    public boolean isFullyConnected() {
        return lasers[0] != null && lasers[1] != null;
    }

    public boolean isLinkedTo(TilePathMarker pathMarker) {
        return links[0] == pathMarker || links[1] == pathMarker;
    }

    public void connect(TilePathMarker marker, LaserData laser) {
        if (lasers[0] == null) {
            lasers[0] = laser;
            links[0] = marker;
        } else if (lasers[1] == null) {
            lasers[1] = laser;
            links[1] = marker;
        }

        if (isFullyConnected()) {
            availableMarkers.remove(this);
        }
    }

    public void createLaserAndConnect(TilePathMarker pathMarker) {
        if (field_145850_b.field_72995_K) {
            return;
        }
        Vec3 point5 = new Vec3(0.5, 0.5, 0.5);

        LaserData laser = new LaserData(Utils.convert(field_174879_c).func_178787_e(point5), Utils.convert(pathMarker.field_174879_c).func_178787_e(point5));

        LaserData laser2 = new LaserData(laser.head, laser.tail);
        laser2.isVisible = false;

        connect(pathMarker, laser);
        pathMarker.connect(this, laser2);
    }

    /** Searches the availableMarkers list for the nearest available that is within searchSize */
    private TilePathMarker findNearestAvailablePathMarker() {
        TilePathMarker nearestAvailable = null;
        double nearestDistance = 0, distance;

        for (TilePathMarker available : availableMarkers) {
            if (available == this || available == this.links[0] || available == this.links[1] || available.func_145831_w().field_73011_w
                    .func_177502_q() != this.func_145831_w().field_73011_w.func_177502_q()) {
                continue;
            }

            distance = field_174879_c.func_177951_i(available.field_174879_c);

            if (distance > DefaultProps.MARKER_RANGE * DefaultProps.MARKER_RANGE) {
                continue;
            }

            if (nearestAvailable == null || distance < nearestDistance) {
                nearestAvailable = available;
                nearestDistance = distance;
            }
        }

        return nearestAvailable;
    }

    @Override
    public void tryConnection() {

        if (field_145850_b.field_72995_K || isFullyConnected()) {
            return;
        }

        // Allow the user to stop the path marker from searching for new path markers to connect
        tryingToConnect = !tryingToConnect;

        sendNetworkUpdate();
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (tryingToConnect) {
            TilePathMarker nearestPathMarker = findNearestAvailablePathMarker();

            if (nearestPathMarker != null) {
                createLaserAndConnect(nearestPathMarker);
            }

            tryingToConnect = false;

            sendNetworkUpdate();
            func_145831_w().func_147458_c(field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p(), field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p());
        }
    }

    @Override
    public List<BlockPos> getPath() {
        HashSet<BlockPos> visitedPaths = new HashSet<>();
        List<BlockPos> res = new ArrayList<>();

        TilePathMarker nextTile = this;

        while (nextTile != null) {
            visitedPaths.add(nextTile.field_174879_c);
            res.add(nextTile.field_174879_c);

            if (nextTile.links[0] != null && !visitedPaths.contains(nextTile.links[0].field_174879_c)) {
                nextTile = nextTile.links[0];
            } else if (nextTile.links[1] != null && !visitedPaths.contains(nextTile.links[1].field_174879_c)) {
                nextTile = nextTile.links[1];
            } else {
                nextTile = null;
            }
        }

        return res;

    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();

        if (links[0] != null) {
            links[0].unlink(this);
        }

        if (links[1] != null) {
            links[1].unlink(this);
        }

        availableMarkers.remove(this);
        tryingToConnect = false;
    }

    @Override
    public void initialize() {
        super.initialize();

        if (!field_145850_b.field_72995_K && !isFullyConnected()) {
            availableMarkers.add(this);
        }

        if (loadLink0) {
            TileEntity e0 = field_145850_b.func_175625_s(new BlockPos(x0, y0, z0));

            if (links[0] != e0 && links[1] != e0 && e0 instanceof TilePathMarker) {
                createLaserAndConnect((TilePathMarker) e0);
            }

            loadLink0 = false;
        }

        if (loadLink1) {
            TileEntity e1 = field_145850_b.func_175625_s(new BlockPos(x1, y1, z1));

            if (links[0] != e1 && links[1] != e1 && e1 instanceof TilePathMarker) {
                createLaserAndConnect((TilePathMarker) e1);
            }

            loadLink1 = false;
        }

        sendNetworkUpdate();
    }

    private void unlink(TilePathMarker tile) {
        if (links[0] == tile) {
            lasers[0] = null;
            links[0] = null;
        }

        if (links[1] == tile) {
            lasers[1] = null;
            links[1] = null;
        }

        if (!isFullyConnected() && !availableMarkers.contains(this) && !field_145850_b.field_72995_K) {
            availableMarkers.add(this);
        }

        sendNetworkUpdate();
    }

    @Override
    public void func_145839_a(NBTTagCompound nbttagcompound) {
        super.func_145839_a(nbttagcompound);

        if (nbttagcompound.func_74764_b("x0")) {
            x0 = nbttagcompound.func_74762_e("x0");
            y0 = nbttagcompound.func_74762_e("y0");
            z0 = nbttagcompound.func_74762_e("z0");

            loadLink0 = true;
        }

        if (nbttagcompound.func_74764_b("x1")) {
            x1 = nbttagcompound.func_74762_e("x1");
            y1 = nbttagcompound.func_74762_e("y1");
            z1 = nbttagcompound.func_74762_e("z1");

            loadLink1 = true;
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound nbttagcompound) {
        super.func_145841_b(nbttagcompound);

        if (links[0] != null) {
            nbttagcompound.func_74768_a("x0", links[0].field_174879_c.func_177958_n());
            nbttagcompound.func_74768_a("y0", links[0].field_174879_c.func_177956_o());
            nbttagcompound.func_74768_a("z0", links[0].field_174879_c.func_177952_p());
        }

        if (links[1] != null) {
            nbttagcompound.func_74768_a("x1", links[1].field_174879_c.func_177958_n());
            nbttagcompound.func_74768_a("y1", links[1].field_174879_c.func_177956_o());
            nbttagcompound.func_74768_a("z1", links[1].field_174879_c.func_177952_p());
        }
    }

    @Override
    public void onChunkUnload() {
        super.onChunkUnload();
        availableMarkers.remove(this);
    }

    public static void clearAvailableMarkersList() {
        availableMarkers.clear();
    }

    public static void clearAvailableMarkersList(World w) {
        for (Iterator<TilePathMarker> it = availableMarkers.iterator(); it.hasNext();) {
            TilePathMarker t = it.next();
            if (t.func_145831_w().field_73011_w.func_177502_q() != w.field_73011_w.func_177502_q()) {
                it.remove();
            }
        }
    }

    @Override
    public void readData(ByteBuf data) {
        boolean previousState = tryingToConnect;

        int flags = data.readUnsignedByte();
        if ((flags & 1) != 0) {
            lasers[0] = new LaserData();
            lasers[0].readData(data);
        } else {
            lasers[0] = null;
        }
        if ((flags & 2) != 0) {
            lasers[1] = new LaserData();
            lasers[1].readData(data);
        } else {
            lasers[1] = null;
        }
        tryingToConnect = (flags & 4) != 0;

        if (previousState != tryingToConnect) {
            field_145850_b.func_175689_h(field_174879_c);
        }
    }

    @Override
    public void writeData(ByteBuf data) {
        int flags = (lasers[0] != null ? 1 : 0) | (lasers[1] != null ? 2 : 0) | (tryingToConnect ? 4 : 0);
        data.writeByte(flags);
        if (lasers[0] != null) {
            lasers[0].writeData(data);
        }
        if (lasers[1] != null) {
            lasers[1].writeData(data);
        }
    }

    @Override
    public void removeFromWorld() {
        List<BlockPos> path = getPath();
        for (BlockPos b : path) {
            BuildCraftCore.pathMarkerBlock.func_176226_b(field_145850_b, b, BuildCraftCore.pathMarkerBlock.func_176223_P(), 0);

            field_145850_b.func_175698_g(b);
        }
    }
}
