/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.blueprints;

import java.util.ArrayList;

import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.util.Vec3;
import net.minecraft.world.World;
import net.minecraft.world.WorldSettings.GameType;

import net.minecraftforge.common.util.Constants;

import buildcraft.api.blueprints.BuildingPermission;
import buildcraft.api.blueprints.IBuilderContext;
import buildcraft.api.blueprints.MappingRegistry;
import buildcraft.api.blueprints.SchematicBlockBase;
import buildcraft.core.Box;
import buildcraft.core.DefaultProps;
import buildcraft.core.lib.utils.Matrix4i;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.Utils;

import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.world.WorldSettings.GameType;
public abstract class BlueprintBase {

    public ArrayList<NBTTagCompound> subBlueprintsNBT = new ArrayList<>();

    public BlockPos anchor = Utils.POS_ZERO;
    public BlockPos size = Utils.POS_ONE;
    public LibraryId id = new LibraryId();
    public String author;
    public boolean rotate = true;
    public boolean excavate = true;
    public BuildingPermission buildingPermission = BuildingPermission.ALL;
    public boolean isComplete = true;

    protected MappingRegistry mapping = new MappingRegistry();
    private SchematicBlockBase[][][] contents;

    private NBTTagCompound nbt;
    private EnumFacing mainDir = EnumFacing.EAST;

    public BlueprintBase() {}

    @Deprecated
    public BlueprintBase(int sizeX, int sizeY, int sizeZ) {
        this(new BlockPos(sizeX, sizeY, sizeZ));
    }

    public BlueprintBase(BlockPos size) {
        contents = new SchematicBlockBase[size.func_177958_n()][size.func_177956_o()][size.func_177952_p()];
        this.size = size;
        this.anchor = Utils.POS_ZERO;
    }

    public SchematicBlockBase get(BlockPos pos) {
        String error = "Tried to access the " + pos + " when the maximum ";
        if (contents.length <= pos.func_177958_n()) throw new ArrayIndexOutOfBoundsException(error + "X coord was " + (contents.length - 1));
        SchematicBlockBase[][] arr2 = contents[pos.func_177958_n()];
        if (arr2.length <= pos.func_177956_o()) throw new ArrayIndexOutOfBoundsException(error + "Y coord was " + (arr2.length - 1));
        SchematicBlockBase[] arr1 = arr2[pos.func_177956_o()];
        if (arr1.length <= pos.func_177952_p()) throw new ArrayIndexOutOfBoundsException(error + "Z coord was " + (arr1.length - 1));
        return arr1[pos.func_177952_p()];
    }

    public void set(BlockPos pos, SchematicBlockBase schematic) {
        contents[pos.func_177958_n()][pos.func_177956_o()][pos.func_177952_p()] = schematic;
    }

    public void translateToBlueprint(Vec3 transform) {
        for (SchematicBlockBase[][] arr2 : contents)
            for (SchematicBlockBase[] arr1 : arr2)
                for (SchematicBlockBase content : arr1)
                    if (content != null) content.translateToBlueprint(transform);
    }

    public void translateToWorld(Vec3 transform) {
        for (SchematicBlockBase[][] arr2 : contents)
            for (SchematicBlockBase[] arr1 : arr2)
                for (SchematicBlockBase content : arr1)
                    if (content != null) content.translateToWorld(transform);
    }

    public void rotateLeft(BptContext context) {
        SchematicBlockBase[][][] newContents = new SchematicBlockBase[size.func_177952_p()][size.func_177956_o()][size.func_177958_n()];

        Matrix4i leftRot = Matrix4i.makeRotLeftTranslatePositive(new Box(BlockPos.field_177992_a, size.func_177973_b(Utils.POS_ONE)));

        for (BlockPos internal : BlockPos.func_177980_a(Utils.POS_ZERO, size.func_177973_b(Utils.POS_ONE))) {
            BlockPos rotated = leftRot.multiplyPosition(internal);

            SchematicBlockBase oldContents = contents[internal.func_177958_n()][internal.func_177956_o()][internal.func_177952_p()];
            if (oldContents != null) {
                oldContents.rotateLeft(context);
                newContents[rotated.func_177958_n()][rotated.func_177956_o()][rotated.func_177952_p()] = oldContents;
            }
        }

        contents = newContents;
        size = new BlockPos(size.func_177952_p(), size.func_177956_o(), size.func_177958_n());

        BlockPos newAnchor = leftRot.multiplyPosition(anchor);

        for (NBTTagCompound sub : subBlueprintsNBT) {
            EnumFacing dir = EnumFacing.values()[sub.func_74771_c("dir")];

            if (dir.func_176740_k() != Axis.Y) dir = dir.func_176746_e();

            Vec3 pos = new Vec3(sub.func_74762_e("x"), sub.func_74762_e("y"), sub.func_74762_e("z"));
            Vec3 rotated = context.rotatePositionLeft(pos);

            sub.func_74768_a("x", (int) rotated.field_72450_a);
            sub.func_74768_a("z", (int) rotated.field_72449_c);
            sub.func_74774_a("dir", (byte) dir.ordinal());
        }

        context.rotateLeft();

        anchor = newAnchor;

        if (mainDir.func_176740_k() != Axis.Y) mainDir = mainDir.func_176746_e();
    }

    private void writeToNBTInternal(NBTTagCompound nbt) {
        nbt.func_74778_a("version", DefaultProps.VERSION);

        if (this instanceof Template) {
            nbt.func_74778_a("kind", "template");
        } else {
            nbt.func_74778_a("kind", "blueprint");
        }

        nbt.func_74757_a("rotate", rotate);
        nbt.func_74757_a("excavate", excavate);

        nbt.func_74782_a("size", NBTUtils.writeBlockPos(size));
        nbt.func_74782_a("anchor", NBTUtils.writeBlockPos(anchor));

        if (author != null) {
            nbt.func_74778_a("author", author);
        }

        saveContents(nbt);

        NBTTagList subBptList = new NBTTagList();

        for (NBTTagCompound subBpt : subBlueprintsNBT) {
            subBptList.func_74742_a(subBpt);
        }

        nbt.func_74782_a("subBpt", subBptList);
    }

    public static BlueprintBase loadBluePrint(NBTTagCompound nbt) {
        String kind = nbt.func_74779_i("kind");

        BlueprintBase bpt;

        if ("template".equals(kind)) {
            bpt = new Template();
        } else {
            bpt = new Blueprint();
        }

        bpt.readFromNBT(nbt);

        return bpt;
    }

    public void readFromNBT(NBTTagCompound nbt) {
        if (nbt.func_74764_b("sizeX")) {
            size = new BlockPos(nbt.func_74762_e("sizeX"), nbt.func_74762_e("sizeY"), nbt.func_74762_e("sizeZ"));
        } else size = NBTUtils.readBlockPos(nbt.func_74781_a("size"));

        if (nbt.func_74764_b("anchorX")) {
            anchor = new BlockPos(nbt.func_74762_e("anchorX"), nbt.func_74762_e("anchorY"), nbt.func_74762_e("anchorZ"));
        } else anchor = NBTUtils.readBlockPos(nbt.func_74781_a("anchor"));

        author = nbt.func_74779_i("author");

        if (nbt.func_74764_b("rotate")) {
            rotate = nbt.func_74767_n("rotate");
        } else {
            rotate = true;
        }

        if (nbt.func_74764_b("excavate")) {
            excavate = nbt.func_74767_n("excavate");
        } else {
            excavate = true;
        }

        contents = new SchematicBlockBase[size.func_177958_n()][size.func_177956_o()][size.func_177952_p()];

        try {
            loadContents(nbt);
        } catch (BptError e) {
            e.printStackTrace();
        }

        if (nbt.func_74764_b("subBpt")) {
            NBTTagList subBptList = nbt.func_150295_c("subBpt", Constants.NBT.TAG_COMPOUND);

            for (int i = 0; i < subBptList.func_74745_c(); ++i) {
                subBlueprintsNBT.add(subBptList.func_150305_b(i));
            }
        }
    }

    public Box getBoxForPos(BlockPos pos) {
        BlockPos min = pos.func_177971_a(anchor);
        BlockPos max = min.func_177971_a(size).func_177973_b(Utils.POS_ONE);
        return new Box(min, max);
    }

    public BptContext getContext(World world, Box box) {
        return new BptContext(world, box, mapping);
    }

    public void addSubBlueprint(BlueprintBase subBpt, BlockPos pos, EnumFacing dir) {
        NBTTagCompound subNBT = new NBTTagCompound();

        subNBT.func_74768_a("x", pos.func_177958_n());
        subNBT.func_74768_a("y", pos.func_177956_o());
        subNBT.func_74768_a("z", pos.func_177952_p());
        subNBT.func_74774_a("dir", (byte) dir.ordinal());
        subNBT.func_74782_a("bpt", subBpt.getNBT());

        subBlueprintsNBT.add(subNBT);
    }

    public NBTTagCompound getNBT() {
        if (nbt == null) {
            nbt = new NBTTagCompound();
            writeToNBTInternal(nbt);
        }
        return nbt;
    }

    public BlueprintBase adjustToWorld(World world, BlockPos pos, EnumFacing o) {
        if (buildingPermission == BuildingPermission.NONE || (buildingPermission == BuildingPermission.CREATIVE_ONLY && world.func_72912_H()
                .func_76077_q() != GameType.CREATIVE)) {
            return null;
        }

        BptContext context = getContext(world, getBoxForPos(pos));

        if (rotate) {
            if (o == EnumFacing.EAST) {
                // Do nothing
            } else if (o == EnumFacing.SOUTH) {
                rotateLeft(context);
            } else if (o == EnumFacing.WEST) {
                rotateLeft(context);
                rotateLeft(context);
            } else if (o == EnumFacing.NORTH) {
                rotateLeft(context);
                rotateLeft(context);
                rotateLeft(context);
            }
        }

        Vec3 transform = Utils.convert(pos).func_178788_d(new Vec3(anchor));

        translateToWorld(transform);

        return this;
    }

    public abstract void loadContents(NBTTagCompound nbt) throws BptError;

    public abstract void saveContents(NBTTagCompound nbt);

    public abstract void readFromWorld(IBuilderContext context, TileEntity anchorTile, BlockPos pos);

    public abstract ItemStack getStack();

    public void readEntitiesFromWorld(IBuilderContext context, TileEntity anchorTile) {

    }
}
