/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.builders;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.item.ItemBlock;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.BlockPos;
import net.minecraft.util.MathHelper;
import net.minecraft.util.Vec3;

import net.minecraftforge.common.util.Constants;

import buildcraft.BuildCraftCore;
import buildcraft.api.blueprints.IBuilderContext;
import buildcraft.api.blueprints.MappingNotFoundException;
import buildcraft.api.blueprints.MappingRegistry;
import buildcraft.api.core.ISerializable;
import buildcraft.core.BlockDecoration;
import buildcraft.core.StackAtPosition;
import buildcraft.core.lib.inventory.InvUtils;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class BuildingItem implements IBuildingItem, ISerializable {

    public static int ITEMS_SPACE = 2;

    public Vec3 origin, destination;
    public LinkedList<StackAtPosition> stacksToDisplay = new LinkedList<>();

    public boolean isDone = false;

    public BuildingSlot slotToBuild;
    public IBuilderContext context;

    private long previousUpdate;
    private float lifetimeDisplay = 0;
    private float maxLifetime = 0;
    private boolean initialized = false;
    private Vec3 velocity;
    private double maxHeight;
    private float lifetime = 0;

    public void initialize() {
        if (!initialized) {
            Vec3 vec = destination.func_178788_d(origin);
            double size = vec.func_72433_c();

            maxLifetime = (float) size * 4;

            maxHeight = size / 2;

            // The below computation is a much simpler version of the long commented out code. Hopefully it works :P

            velocity = Utils.multiply(vec, 1 / maxLifetime);

            // the below computation is an approximation of the distance to
            // travel for the object. It really follows a sinus, but we compute
            // the size of a triangle for simplification.

            // Vec3 middle = new Vec3();
            // middle.x = (destination.x + origin.x) / 2;
            // middle.y = (destination.y + origin.y) / 2;
            // middle.z = (destination.z + origin.z) / 2;
            //
            // Vec3 top = new Vec3();
            // top.x = middle.x;
            // top.y = middle.y + maxHeight;
            // top.z = middle.z;
            //
            // Vec3 originToTop = new Vec3();
            // originToTop.x = top.x - origin.x;
            // originToTop.y = top.y - origin.y;
            // originToTop.z = top.z - origin.z;
            //
            // Vec3 destinationToTop = new Vec3();
            // destinationToTop.x = destination.x - origin.x;
            // destinationToTop.y = destination.y - origin.y;
            // destinationToTop.z = destination.z - origin.z;
            //
            // double d1 = Math.sqrt(originToTop.x * originToTop.x + originToTop.y * originToTop.y + originToTop.z *
            // originToTop.z);
            //
            // double d2 =
            // Math.sqrt(destinationToTop.x * destinationToTop.x + destinationToTop.y * destinationToTop.y +
            // destinationToTop.z * destinationToTop.z);
            //
            // d1 = d1 / size * maxLifetime;
            // d2 = d2 / size * maxLifetime;
            //
            // maxLifetime = (float) d1 + (float) d2;
            //
            // vx = dx / maxLifetime;
            // vy = dy / maxLifetime;
            // vz = dz / maxLifetime;

            if (stacksToDisplay.size() == 0) {
                StackAtPosition sPos = new StackAtPosition();
                sPos.stack = new ItemStack(BuildCraftCore.decoratedBlock);
                stacksToDisplay.add(sPos);
            }

            initialized = true;
        }
    }

    public Vec3 getDisplayPosition(float time) {
        Vec3 pos = origin.func_178787_e(Utils.multiply(velocity, time));
        return pos.func_72441_c(0, MathHelper.func_76126_a(time / maxLifetime * (float) Math.PI) * maxHeight, 0);
    }

    public void update() {
        if (isDone) {
            return;
        }

        initialize();

        lifetime++;

        if (lifetime > maxLifetime + stacksToDisplay.size() * ITEMS_SPACE - 1) {
            isDone = true;
            build();
        }

        lifetimeDisplay = lifetime;
        previousUpdate = new Date().getTime();

        if (slotToBuild != null && lifetime > maxLifetime) {
            slotToBuild.writeCompleted(context, (lifetime - maxLifetime) / (stacksToDisplay.size() * ITEMS_SPACE));
        }
    }

    public void displayUpdate() {
        initialize();

        float tickDuration = 50.0F; // miliseconds
        long currentUpdate = new Date().getTime();
        float timeSpan = currentUpdate - previousUpdate;
        previousUpdate = currentUpdate;

        float displayPortion = timeSpan / tickDuration;

        if (lifetimeDisplay - lifetime <= 1.0) {
            lifetimeDisplay += 1.0 * displayPortion;
        }
    }

    private void build() {
        if (slotToBuild != null) {
            BlockPos dest = Utils.convertFloor(destination);

            IBlockState state = context.world().getBlockState(dest);

            /* if (BlockUtil.isToughBlock(context.world(), destX, destY, destZ)) { BlockUtil.breakBlock(context.world(),
             * destX, destY, destZ, BuildCraftBuilders.fillerLifespanTough); } else {
             * BlockUtil.breakBlock(context.world(), destX, destY, destZ, BuildCraftBuilders.fillerLifespanNormal); } */

            if (slotToBuild.writeToWorld(context)) {
                context.world().playAuxSFXAtEntity(null, 2001, dest, Block.func_176210_f(state));
            } else {
                for (ItemStack s : slotToBuild.stackConsumed) {
                    if (s != null && !(s.func_77973_b() instanceof ItemBlock && Block.func_149634_a(s.func_77973_b()) instanceof BlockDecoration)) {
                        InvUtils.dropItems(context.world(), s, dest);
                    }
                }
            }
        }
    }

    public LinkedList<StackAtPosition> getStacks() {
        int d = 0;

        for (StackAtPosition s : stacksToDisplay) {
            float stackLife = lifetimeDisplay - d;

            if (stackLife <= maxLifetime && stackLife > 0) {
                s.pos = getDisplayPosition(stackLife);
                s.display = true;
            } else {
                s.display = false;
            }

            d += ITEMS_SPACE;
        }

        return stacksToDisplay;
    }

    @Override
    public boolean isDone() {
        return isDone;
    }

    public void writeToNBT(NBTTagCompound nbt) {
        nbt.func_74782_a("origin", NBTUtils.writeVec3(origin));
        nbt.func_74782_a("destination", NBTUtils.writeVec3(destination));
        nbt.func_74776_a("lifetime", lifetime);

        NBTTagList items = new NBTTagList();

        for (StackAtPosition s : stacksToDisplay) {
            NBTTagCompound cpt = new NBTTagCompound();
            s.stack.func_77955_b(cpt);
            items.func_74742_a(cpt);
        }

        nbt.func_74782_a("items", items);

        MappingRegistry registry = new MappingRegistry();

        NBTTagCompound slotNBT = new NBTTagCompound();
        NBTTagCompound registryNBT = new NBTTagCompound();

        slotToBuild.writeToNBT(slotNBT, registry);
        registry.write(registryNBT);

        nbt.func_74782_a("registry", registryNBT);

        if (slotToBuild instanceof BuildingSlotBlock) {
            nbt.func_74774_a("slotKind", (byte) 0);
        } else {
            nbt.func_74774_a("slotKind", (byte) 1);
        }

        nbt.func_74782_a("slotToBuild", slotNBT);
    }

    public void readFromNBT(NBTTagCompound nbt) throws MappingNotFoundException {
        origin = NBTUtils.readVec3(nbt, "origin");
        destination = NBTUtils.readVec3(nbt, "destination");
        lifetime = nbt.func_74760_g("lifetime");

        NBTTagList items = nbt.func_150295_c("items", Constants.NBT.TAG_COMPOUND);

        for (int i = 0; i < items.func_74745_c(); ++i) {
            StackAtPosition sPos = new StackAtPosition();
            sPos.stack = ItemStack.func_77949_a(items.func_150305_b(i));
            stacksToDisplay.add(sPos);
        }

        MappingRegistry registry = new MappingRegistry();
        registry.read(nbt.func_74775_l("registry"));

        if (nbt.func_74771_c("slotKind") == 0) {
            slotToBuild = new BuildingSlotBlock();
        } else {
            slotToBuild = new BuildingSlotEntity();
        }

        slotToBuild.readFromNBT(nbt.func_74775_l("slotToBuild"), registry);
    }

    public void setStacksToDisplay(List<ItemStack> stacks) {
        if (stacks != null) {
            for (ItemStack s : stacks) {
                for (int i = 0; i < s.field_77994_a; ++i) {
                    StackAtPosition sPos = new StackAtPosition();
                    sPos.stack = s.func_77946_l();
                    sPos.stack.field_77994_a = 1;
                    stacksToDisplay.add(sPos);
                }
            }
        }
    }

    @Override
    public void readData(ByteBuf stream) {
        origin = NetworkUtils.readVec3(stream);
        destination = NetworkUtils.readVec3(stream);
        lifetime = stream.readFloat();
        stacksToDisplay.clear();
        int size = stream.readUnsignedShort();
        for (int i = 0; i < size; i++) {
            StackAtPosition e = new StackAtPosition();
            e.readData(stream);
            stacksToDisplay.add(e);
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        NetworkUtils.writeVec3(stream, origin);
        NetworkUtils.writeVec3(stream, destination);
        stream.writeFloat(lifetime);
        stream.writeShort(stacksToDisplay.size());
        for (StackAtPosition s : stacksToDisplay) {
            s.writeData(stream);
        }
    }

    @Override
    public int hashCode() {
        return (131 * origin.hashCode()) + destination.hashCode();
    }
}
