package buildcraft.core.lib.block;

import buildcraft.api.core.EnumColor;
import buildcraft.api.enums.*;
import buildcraft.api.properties.BuildCraftExtendedProperty;
import buildcraft.api.properties.BuildCraftProperties;
import buildcraft.api.properties.BuildCraftProperty;
import buildcraft.core.BCCreativeTab;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import net.minecraft.block.Block;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.state.BlockState;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.item.ItemStack;
import net.minecraft.util.*;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraftforge.common.property.ExtendedBlockState;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

public abstract class BlockBuildCraftBase extends Block {

    public static final BuildCraftProperty<EnumFacing> FACING_PROP = BuildCraftProperties.BLOCK_FACING;
    public static final BuildCraftProperty<EnumFacing> FACING_6_PROP = BuildCraftProperties.BLOCK_FACING_6;

    public static final BuildCraftProperty<EnumEngineType> ENGINE_TYPE = BuildCraftProperties.ENGINE_TYPE;
    public static final BuildCraftProperty<EnumColor> COLOR_PROP = BuildCraftProperties.BLOCK_COLOR;
    public static final BuildCraftProperty<EnumSpring> SPRING_TYPE = BuildCraftProperties.SPRING_TYPE;
    public static final BuildCraftProperty<EnumEnergyStage> ENERGY_STAGE = BuildCraftProperties.ENERGY_STAGE;
    public static final BuildCraftProperty<EnumFillerPattern> FILLER_PATTERN = BuildCraftProperties.FILLER_PATTERN;
    public static final BuildCraftProperty<EnumBlueprintType> BLUEPRINT_TYPE = BuildCraftProperties.BLUEPRINT_TYPE;
    public static final BuildCraftProperty<EnumLaserTableType> LASER_TABLE_TYPE = BuildCraftProperties.LASER_TABLE_TYPE;
    public static final BuildCraftProperty<EnumDecoratedBlock> DECORATED_TYPE = BuildCraftProperties.DECORATED_BLOCK;

    public static final BuildCraftProperty<Integer> GENERIC_PIPE_DATA = BuildCraftProperties.GENERIC_PIPE_DATA;

    public static final BuildCraftProperty<Boolean> JOINED_BELOW = BuildCraftProperties.JOINED_BELOW;
    public static final BuildCraftProperty<Boolean> MOVING = BuildCraftProperties.MOVING;
    public static final BuildCraftProperty<Integer> LED_POWER = BuildCraftProperties.LED_POWER;
    public static final BuildCraftProperty<Boolean> LED_DONE = BuildCraftProperties.LED_DONE;

    public static final BuildCraftProperty<Boolean> CONNECTED_UP = BuildCraftProperties.CONNECTED_UP;
    public static final BuildCraftProperty<Boolean> CONNECTED_DOWN = BuildCraftProperties.CONNECTED_DOWN;
    public static final BuildCraftProperty<Boolean> CONNECTED_EAST = BuildCraftProperties.CONNECTED_EAST;
    public static final BuildCraftProperty<Boolean> CONNECTED_WEST = BuildCraftProperties.CONNECTED_WEST;
    public static final BuildCraftProperty<Boolean> CONNECTED_NORTH = BuildCraftProperties.CONNECTED_NORTH;
    public static final BuildCraftProperty<Boolean> CONNECTED_SOUTH = BuildCraftProperties.CONNECTED_SOUTH;

    public static final Map<EnumFacing, BuildCraftProperty<Boolean>> CONNECTED_MAP = BuildCraftProperties.CONNECTED_MAP;
    @SuppressWarnings("unchecked")
    public static final BuildCraftProperty<Boolean>[] CONNECTED_ARRAY = CONNECTED_MAP.values().toArray(new BuildCraftProperty[6]);

    protected BuildCraftProperty<?>[] properties;
    protected BuildCraftProperty<?>[] nonMetaProperties;
    protected BuildCraftExtendedProperty<?>[] extendedProperties;

    protected boolean hasExtendedProperties;

    protected List<BuildCraftProperty<?>> propertyList;
    protected final Map<Integer, IBlockState> intToState = Maps.newHashMap();
    protected final Map<IBlockState, Integer> stateToInt = Maps.newHashMap();
    protected final BlockState myBlockState;

    /** True if this block can rotate in any of the horizontal directions */
    public boolean horizontallyRotatable;
    /** True if this block can rotate in any of the six facing directions */
    public boolean allRotatable;

    protected BlockBuildCraftBase(Material material) {
        this(material, BCCreativeTab.get("main"), false, new BuildCraftProperty<?>[0]);
    }

    protected BlockBuildCraftBase(Material material, BCCreativeTab creativeTab) {
        this(material, creativeTab, false, new BuildCraftProperty<?>[0]);
    }

    protected BlockBuildCraftBase(Material material, BuildCraftProperty<?>... properties) {
        this(material, BCCreativeTab.get("main"), false, properties);
    }

    protected BlockBuildCraftBase(Material material, BCCreativeTab bcCreativeTab, BuildCraftProperty<?>... properties) {
        this(material, bcCreativeTab, false, properties);
    }

    protected BlockBuildCraftBase(Material material, BCCreativeTab bcCreativeTab, boolean hasExtendedProps, BuildCraftProperty<?>... properties) {
        super(material);
        func_149647_a(bcCreativeTab);
        func_149711_c(5F);
        List<BuildCraftProperty<?>> metas = Lists.newArrayList();
        List<BuildCraftProperty<?>> nonMetas = Lists.newArrayList();
        List<BuildCraftExtendedProperty<?>> infinites = Lists.newArrayList();

        this.hasExtendedProperties = fillStateListsPre(hasExtendedProps, metas, nonMetas, infinites, properties);

        this.properties = metas.toArray(new BuildCraftProperty<?>[0]);
        this.nonMetaProperties = nonMetas.toArray(new BuildCraftProperty<?>[0]);
        this.extendedProperties = infinites.toArray(new BuildCraftExtendedProperty<?>[0]);
        this.myBlockState = func_180661_e();

        fillStateMapPost(metas, nonMetas, properties);
    }

    @SuppressWarnings("static-method")
    protected boolean fillStateListsPre(boolean hasExtendedProps, List<BuildCraftProperty<?>> metas, List<BuildCraftProperty<?>> nonMetas,
            List<BuildCraftExtendedProperty<?>> infinites, BuildCraftProperty<?>... properties) {
        int total = 1;
        for (BuildCraftProperty<?> prop : properties) {
            if (prop == null) {
                /* Used by some blocks (e.g. the filler) if they do or do not want to have a specific property at
                 * runtime (per block). Is used in the format "wantProperty ? someProp : null" */
                continue;
            }
            if (prop instanceof BuildCraftExtendedProperty<?>) {
                infinites.add((BuildCraftExtendedProperty<?>) prop);
                hasExtendedProps = true;
                continue;
            }

            total *= prop.getAllowedValues().size();

            if (total > 16) {
                nonMetas.add(prop);
            } else {
                metas.add(prop);
            }
        }

        return hasExtendedProps;
    }

    protected void fillStateMapPost(List<BuildCraftProperty<?>> metas, List<BuildCraftProperty<?>> nonMetas, BuildCraftProperty<?>... properties) {
        IBlockState defaultState = func_176194_O().func_177621_b();

        Map<IBlockState, Integer> tempValidStates = Maps.newHashMap();
        tempValidStates.put(defaultState, 0);
        boolean canRotate = false;
        boolean canSixRotate = false;

        for (BuildCraftProperty<?> prop : properties) {
            if (prop == null) {
                continue;
            }

            if (prop instanceof BuildCraftExtendedProperty<?>) {
                continue;
            }

            if (prop == FACING_PROP) {
                canRotate = true;
            }
            if (prop == FACING_6_PROP) {
                canRotate = true;
                canSixRotate = true;
            }

            List<? extends Comparable> allowedValues = prop.getAllowedValues();
            defaultState = withProperty(defaultState, prop, allowedValues.iterator().next());

            Map<IBlockState, Integer> newValidStates = Maps.newHashMap();
            int mul = metas.contains(prop) ? allowedValues.size() : 1;
            for (Entry<IBlockState, Integer> entry : tempValidStates.entrySet()) {
                int index = 0;
                Collections.sort(allowedValues);
                for (Comparable<?> comp : allowedValues) {
                    int pos = entry.getValue() * mul + index;
                    newValidStates.put(withProperty(entry.getKey(), prop, comp), pos);
                    if (mul > 1) {
                        index++;
                    }
                }
            }
            tempValidStates = newValidStates;
        }

        horizontallyRotatable = canRotate;
        allRotatable = canSixRotate;

        for (Entry<IBlockState, Integer> entry : tempValidStates.entrySet()) {
            int i = entry.getValue();
            stateToInt.put(entry.getKey(), i);
            if (!intToState.containsKey(i)) {
                intToState.put(i, entry.getKey());
            }
        }
        func_180632_j(defaultState);

        List<BuildCraftProperty<?>> allProperties = Lists.newArrayList();
        allProperties.addAll(metas);
        allProperties.addAll(nonMetas);
        propertyList = Collections.unmodifiableList(allProperties);
    }

    // Generic helper methods, these stop generics from being strange
    @SuppressWarnings("unchecked")
    private IBlockState withProperty(IBlockState state, BuildCraftProperty prop, Comparable value) {
        return withProperty0(state, prop, value);
    }

    private <V extends Comparable<V>, T extends V> IBlockState withProperty0(IBlockState state, BuildCraftProperty<V> prop, T value) {
        return state.func_177226_a(prop, value);
    }

    @Override
    public BlockState func_176194_O() {
        return this.myBlockState;
    }

    @Override
    protected BlockState func_180661_e() {
        if (properties == null) {
            // Will be overridden later
            return new BlockState(this, new IProperty[] {});
        }

        IProperty[] props = new IProperty[properties.length + nonMetaProperties.length];
        System.arraycopy(properties, 0, props, 0, properties.length);
        System.arraycopy(nonMetaProperties, 0, props, properties.length, nonMetaProperties.length);
        if (hasExtendedProperties) {
            return new ExtendedBlockState(this, props, extendedProperties);
        }
        return new BlockState(this, props);
    }

    @Override
    public int func_176201_c(IBlockState state) {
        return stateToInt.containsKey(state) ? stateToInt.get(state) : 0;
    }

    @Override
    public IBlockState func_176203_a(int meta) {
        return intToState.containsKey(meta) ? intToState.get(meta) : func_176223_P();
    }

    @Override
    public IBlockState func_180642_a(World worldIn, BlockPos pos, EnumFacing facing, float hitX, float hitY, float hitZ, int meta,
            EntityLivingBase placer) {
        if (allRotatable) {// TODO (CHECK): Do we want to do this for all blocks that have 6 facing directions
            return func_176203_a(meta).func_177226_a(FACING_6_PROP, facing);
        } else {
            return func_176203_a(meta);
        }
    }

    @Override
    public void func_180633_a(World world, BlockPos pos, IBlockState state, EntityLivingBase entity, ItemStack stack) {
        // If it was allRotatable the vertical direction would have been taken into account above
        if (horizontallyRotatable && !allRotatable) {
            EnumFacing orientation = entity.func_174811_aO();
            world.func_175656_a(pos, state.func_177226_a(FACING_PROP, orientation.func_176734_d()));
        }
    }

    /** Override this to easily allow the collision boxes and selected bounding boxes to be calculated */
    public AxisAlignedBB getBox(IBlockAccess world, BlockPos pos, IBlockState state) {
        return new AxisAlignedBB(0, 0, 0, 1, 1, 1);
    }

    /** You must override one of {@link #getBoxes(IBlockAccess, BlockPos, IBlockState)} or
     * {@link #getBox(IBlockAccess, BlockPos, IBlockState)} otherwise you will get a crash. */
    public AxisAlignedBB[] getBoxes(IBlockAccess world, BlockPos pos, IBlockState state) {
        return new AxisAlignedBB[] { getBox(world, pos, state) };
    }

    /** Exposed so subclasses can call Block's collision ray trace method without needing to resort to idk,
     * reflection? */
    public MovingObjectPosition collisionRayTrace_super(World world, BlockPos pos, Vec3 origin, Vec3 direction) {
        return super.func_180636_a(world, pos, origin, direction);
    }

    @Override
    public MovingObjectPosition func_180636_a(World world, BlockPos pos, Vec3 origin, Vec3 direction) {
        IBlockState state = world.func_180495_p(pos);
        AxisAlignedBB[] aabbs = getBoxes(world, pos, state);
        MovingObjectPosition closest = null;
        for (AxisAlignedBB aabb : aabbs) {
            aabb = aabb.func_72317_d(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p()).func_72314_b(-0.01, -0.01, -0.01);

            MovingObjectPosition mop = aabb.func_72327_a(origin, direction);
            if (mop != null) {
                if (closest != null && mop.field_72307_f.func_72438_d(origin) < closest.field_72307_f.func_72438_d(origin)) {
                    closest = mop;
                } else {
                    closest = mop;
                }
            }
        }
        if (closest == null) {
            return null;
        } else {
            return new MovingObjectPosition(closest.field_72307_f, closest.field_178784_b, pos);
        }
    }

    @Override
    public void func_180638_a(World world, BlockPos pos, IBlockState state, AxisAlignedBB mask, List<AxisAlignedBB> list, Entity par7Entity) {
        if (!func_149703_v()) {
            return;
        } else {
            for (AxisAlignedBB bb : getBoxes(world, pos, state)) {
                bb = bb.func_72317_d(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
                if (mask.func_72326_a(bb)) {
                    list.add(bb);
                }
            }
        }
    }

    @Override
    public boolean func_176209_a(IBlockState state, boolean hitIfLiquid) {
        return true;
    }

    @Override
    public AxisAlignedBB func_180640_a(World world, BlockPos pos, IBlockState state) {
        if (func_149703_v()) {
            return getBox(world, pos, state).func_72317_d(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
        } else {
            return null;
        }
    }

    @Override
    @SideOnly(Side.CLIENT)
    public AxisAlignedBB func_180646_a(World world, BlockPos pos) {
        return getBox(world, pos, world.func_180495_p(pos)).func_72317_d(pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p());
    }

    /* @Override public void setBlockBoundsBasedOnState(IBlockAccess world, BlockPos pos) { AxisAlignedBB[] bbs =
     * getBoxes(world, pos, world.getBlockState(pos)); AxisAlignedBB bb = bbs[0]; for (int i = 1; i < bbs.length; i++) {
     * bb = bb.union(bbs[i]); } minX = bb.minX; minY = bb.minY; minZ = bb.minZ; maxX = bb.maxX; maxY = bb.maxY; maxZ =
     * bb.maxZ; } */
}
