/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.block;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.BCLog;
import buildcraft.api.core.ISerializable;
import buildcraft.api.tiles.IControllable;
import buildcraft.api.tiles.IControllable.Mode;
import buildcraft.core.DefaultProps;
import buildcraft.core.lib.RFBattery;
import buildcraft.core.lib.TileBuffer;
import buildcraft.core.lib.network.PacketTileUpdate;
import buildcraft.core.lib.network.base.Packet;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;
import cofh.api.energy.IEnergyProvider;
import cofh.api.energy.IEnergyReceiver;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import net.minecraft.block.Block;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.state.IBlockState;
import net.minecraft.crash.CrashReport;
import net.minecraft.crash.CrashReportCategory;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.ISidedInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.S35PacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.*;
import net.minecraft.world.World;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.wrapper.InvWrapper;
import net.minecraftforge.items.wrapper.SidedInvWrapper;
import org.apache.commons.lang3.StringUtils;

import java.util.HashSet;

/** For future maintainers: This class intentionally does not implement just every interface out there. For some of them
 * (such as IControllable), we expect the tiles supporting it to implement it - but TileBuildCraft provides all the
 * underlying functionality to stop code repetition. */
public abstract class TileBuildCraft extends TileEntity implements IEnergyProvider, IEnergyReceiver, ISerializable, ITickable, IAdditionalDataTile {
    protected TileBuffer[] cache;
    protected HashSet<EntityPlayer> guiWatchers = new HashSet<>();
    protected IControllable.Mode mode;
    private boolean sendNetworkUpdate = false;

    protected int init = 0;
    private String owner = "[BuildCraft]";
    private RFBattery battery;

    private int receivedTick, extractedTick;
    private long worldTimeEnergyReceive;
    /** Used at the client for the power LED brightness */
    public int ledPower = 0, lastLedPower = 0;
    public boolean ledDone = false, lastLedDone = false;

    /** Used to help migrate existing worlds to whatever new blockstate format we use. Note that proper migration cannot
     * be implemented until this pre-release has gone out for a while now. */
    private NBTTagCompound lastBlockState = null;

    public String getOwner() {
        return owner;
    }

    public void addGuiWatcher(EntityPlayer player) {
        if (!guiWatchers.contains(player)) {
            guiWatchers.add(player);
        }
    }

    public void removeGuiWatcher(EntityPlayer player) {
        if (guiWatchers.contains(player)) {
            guiWatchers.remove(player);
        }
    }

    @Override
    public void func_73660_a() {
        if (field_145850_b == null) throw new NullPointerException("worldObj");
        if (init != 2 && !func_145837_r()) {
            if (init < 1) {
                init++;
                return;
            }
            initialize();
            init = 2;
        }

        if (battery != null) {
            receivedTick = 0;
            extractedTick = 0;

            if (!field_145850_b.field_72995_K) {
                int prePower = ledPower;
                int stored = battery.getEnergyStored();
                int max = battery.getMaxEnergyStored();
                ledPower = 0;
                if (stored != 0) {
                    ledPower = stored * 2 / max + 1;
                }
                if (prePower != ledPower) {
                    sendNetworkUpdate();
                }
            }
        }

        if (!field_145850_b.field_72995_K) {
            if (battery != null) {
                if (battery.getMaxEnergyStored() > 0) {
                    ledPower = 3 * battery.getEnergyStored() / battery.getMaxEnergyStored();
                } else {
                    ledPower = 0;
                }
            }
        }

        if (lastLedPower != ledPower || lastLedDone != ledDone) {
            if (field_145850_b.field_72995_K) {
                field_145850_b.func_175689_h(func_174877_v());
            } else {
                sendNetworkUpdate();
            }
            lastLedPower = ledPower;
            lastLedDone = ledDone;
        }

        if (sendNetworkUpdate) {
            if (field_145850_b != null && !field_145850_b.field_72995_K) {
                BuildCraftCore.instance.sendToPlayers(getPacketUpdate(), field_145850_b, func_174877_v(), DefaultProps.NETWORK_UPDATE_RANGE);
                sendNetworkUpdate = false;
            }
        }
    }

    public void initialize() {

    }

    @Override
    public void func_145829_t() {
        super.func_145829_t();
        cache = null;
    }

    @Override
    public void func_145843_s() {
        init = 0;
        super.func_145843_s();
        cache = null;
    }

    public void onBlockPlacedBy(EntityLivingBase entity, ItemStack stack) {
        if (entity instanceof EntityPlayer) {
            owner = ((EntityPlayer) entity).getDisplayNameString();
        }
    }

    public void destroy() {
        cache = null;
    }

    @Override
    public void sendNetworkUpdate() {
        sendNetworkUpdate = true;
    }

    @Override
    public void writeData(ByteBuf stream) {
        stream.writeByte(ledPower);
        NetworkUtils.writeEnum(stream, mode);
    }

    @Override
    public void readData(ByteBuf stream) {
        ledPower = stream.readByte();
        mode = NetworkUtils.readEnum(stream, Mode.class);
    }

    public PacketTileUpdate getPacketUpdate() {
        return new PacketTileUpdate(this, this);
    }

    @Override
    public S35PacketUpdateTileEntity func_145844_m() {
        NBTTagCompound nbt = new NBTTagCompound();
        nbt.func_74778_a("net-type", "desc-packet");
        Packet p = getPacketUpdate();
        ByteBuf buf = Unpooled.buffer();
        p.writeData(buf);
        byte[] bytes = new byte[buf.readableBytes()];
        buf.readBytes(bytes);
        nbt.func_74773_a("net-data", bytes);
        S35PacketUpdateTileEntity tileUpdate = new S35PacketUpdateTileEntity(func_174877_v(), 0, nbt);
        return tileUpdate;
    }

    @Override
    public void onDataPacket(NetworkManager net, S35PacketUpdateTileEntity pkt) {
        if (!field_145850_b.field_72995_K) return;
        if (pkt.func_148857_g() == null) throw new RuntimeException("No NBTTag compound! This is a bug!");
        NBTTagCompound nbt = pkt.func_148857_g();
        try {
            if ("desc-packet".equals(nbt.func_74779_i("net-type"))) {
                byte[] bytes = nbt.func_74770_j("net-data");
                ByteBuf data = Unpooled.wrappedBuffer(bytes);
                PacketTileUpdate p = new PacketTileUpdate();
                p.readData(data);
                // The player is not used so its fine
                p.applyData(field_145850_b, null);
            } else {
                BCLog.logger.warn("Recieved a packet with a different type that expected (" + nbt.func_74779_i("net-type") + ")");
            }
        } catch (Throwable t) {
            throw new RuntimeException("Failed to read a packet! (net-type=\"" + nbt.func_74781_a("net-type") + "\", net-data=\"" + nbt.func_74781_a("net-data")
                + "\")", t);
        }
    }

    @Override
    public boolean shouldRefresh(World world, BlockPos pos, IBlockState oldState, IBlockState newSate) {
        return oldState.func_177230_c() != newSate.func_177230_c();
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);
        nbt.func_74778_a("owner", owner);
        if (battery != null) {
            NBTTagCompound batteryNBT = new NBTTagCompound();
            battery.writeToNBT(batteryNBT);
            nbt.func_74782_a("battery", batteryNBT);
        }
        if (mode != null) {
            nbt.func_74774_a("lastMode", (byte) mode.ordinal());
        }

        // Version tag that can be used for upgrading.
        // 0 means[1.8.9] 7.2.0-pre12 or before (default value)
        // 1 means [1.8.9] 7.2.0-pre13 up until 7.2.0-preX
        // 2 means [1.8.9] 7.2.0-preX or later
        nbt.func_74768_a("data-version", 1);

        /* Also save the state of all BC tiles. This will be helpful for migration. */
        // REMOVE THIS AFTER preX
        if (func_145830_o()) {
            IBlockState blockstate = field_145850_b.func_180495_p(func_174877_v());
            Block block = blockstate.func_177230_c();
            if (block instanceof BlockBuildCraft) {
                // Assume that this is us- it would be odd for this tile to be with the wrong block.
                BlockBuildCraft bcBlock = (BlockBuildCraft) block;
                NBTTagCompound statenbt = new NBTTagCompound();
                for (IProperty<?> prop : bcBlock.properties) {
                    Object value = blockstate.func_177229_b(prop);
                    if (value == null) continue;
                    statenbt.func_74782_a(prop.func_177701_a(), NBTUtils.writeObject(value));
                }
                nbt.func_74782_a("blockstate", statenbt);
            }
        }
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);
        if (nbt.func_74764_b("owner")) {
            owner = nbt.func_74779_i("owner");
        }
        if (battery != null) {
            battery.readFromNBT(nbt.func_74775_l("battery"));
        }
        if (nbt.func_74764_b("lastMode")) {
            mode = IControllable.Mode.values()[nbt.func_74771_c("lastMode")];
        }

        int version = nbt.func_74762_e("data-version");

        // Load up the block from pre12 -> preX
        if (nbt.func_74764_b("blockstate") && version == 1) lastBlockState = nbt.func_74775_l("blockstate");
    }

    protected int getTicksSinceEnergyReceived() {
        return (int) (field_145850_b.func_82737_E() - worldTimeEnergyReceive);
    }

    @Override
    public int hashCode() {
        return field_174879_c.hashCode();
    }

    @Override
    public boolean equals(Object cmp) {
        return this == cmp;
    }

    @Override
    public boolean canConnectEnergy(EnumFacing from) {
        return battery != null;
    }

    @Override
    public int receiveEnergy(EnumFacing from, int maxReceive, boolean simulate) {
        if (battery != null && this.canConnectEnergy(from)) {
            int received = battery.receiveEnergy(Math.min(maxReceive, battery.getMaxEnergyReceive() - receivedTick), simulate);
            if (!simulate) {
                receivedTick += received;
                worldTimeEnergyReceive = field_145850_b.func_82737_E();
            }
            return received;
        } else {
            return 0;
        }
    }

    /** If you want to use this, implement IEnergyProvider. */
    @Override
    public int extractEnergy(EnumFacing from, int maxExtract, boolean simulate) {
        if (battery != null && this.canConnectEnergy(from)) {
            int extracted = battery.extractEnergy(Math.min(maxExtract, battery.getMaxEnergyExtract() - extractedTick), simulate);
            if (!simulate) {
                extractedTick += extracted;
            }
            return extracted;
        } else {
            return 0;
        }
    }

    @Override
    public int getEnergyStored(EnumFacing from) {
        if (battery != null && this.canConnectEnergy(from)) {
            return battery.getEnergyStored();
        } else {
            return 0;
        }
    }

    @Override
    public int getMaxEnergyStored(EnumFacing from) {
        if (battery != null && this.canConnectEnergy(from)) {
            return battery.getMaxEnergyStored();
        } else {
            return 0;
        }
    }

    public RFBattery getBattery() {
        return battery;
    }

    protected void setBattery(RFBattery battery) {
        this.battery = battery;
    }

    public IBlockState getBlockState(EnumFacing side) {
        if (isNotReady()) return null;
        if (cache == null) {
            cache = TileBuffer.makeBuffer(field_145850_b, field_174879_c, false);
        }
        return cache[side.ordinal()].getBlockState();
    }

    public TileEntity getTile(EnumFacing side) {
        if (isNotReady()) return null;
        if (cache == null) {
            cache = TileBuffer.makeBuffer(field_145850_b, field_174879_c, false);
        }
        return cache[side.ordinal()].getTile();
    }

    public IControllable.Mode getControlMode() {
        return mode;
    }

    public void setControlMode(IControllable.Mode mode) {
        this.mode = mode;
        sendNetworkUpdate();
    }

    // Capability wrapper

    private IItemHandler[] invWrapper;

    @Override
    public boolean hasCapability(Capability<?> capability, EnumFacing facing) {
        if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
            return this instanceof IInventory;
        } else {
            return super.hasCapability(capability, facing);
        }
    }

    @Override
    public <T> T getCapability(Capability<T> capability, EnumFacing facing) {
        if (capability == CapabilityItemHandler.ITEM_HANDLER_CAPABILITY) {
            if (this instanceof IInventory) {
                if (invWrapper == null) {
                    if (this instanceof ISidedInventory) {
                        invWrapper = new IItemHandler[7];
                        for (EnumFacing facing1 : EnumFacing.field_82609_l) {
                            invWrapper[facing1.ordinal()] = new SidedInvWrapper((ISidedInventory) this, facing);
                        }
                        invWrapper[6] = new SidedInvWrapper((ISidedInventory) this, null);
                    } else {
                        invWrapper = new IItemHandler[1];
                        invWrapper[0] = new InvWrapper((IInventory) this);
                    }
                }

                if (invWrapper.length == 7) {
                    return (T) invWrapper[facing == null ? 6 : facing.ordinal()];
                } else {
                    return (T) invWrapper[0];
                }
            }
            return null;
        } else {
            return super.getCapability(capability, facing);
        }
    }

    // IInventory

    public int getField(int id) {
        return 0;
    }

    public void setField(int id, int value) {}

    public int getFieldCount() {
        return 0;
    }

    public String getInventoryName() {
        return "";
    }

    public String getName() {
        return getInventoryName();
    }

    public IChatComponent getDisplayName() {
        return new ChatComponentText(getInventoryName());
    }

    public void clear() {}

    public boolean hasCustomName() {
        return !StringUtils.isEmpty(getInventoryName());
    }

    public boolean isNotReady() {
        return !func_145830_o() || init != 2;
    }
}
