package buildcraft.core.lib.client.model;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;

import javax.vecmath.Matrix4f;
import javax.vecmath.Vector4f;

import com.google.common.base.Function;
import com.google.common.base.Optional;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;

import org.apache.commons.lang3.tuple.Pair;

import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.block.model.BakedQuad;
import net.minecraft.client.renderer.block.model.ItemCameraTransforms;
import net.minecraft.client.renderer.block.model.ItemCameraTransforms.TransformType;
import net.minecraft.client.renderer.texture.TextureAtlasSprite;
import net.minecraft.client.renderer.vertex.VertexFormat;
import net.minecraft.client.resources.model.IBakedModel;
import net.minecraft.client.resources.model.ModelRotation;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.MathHelper;
import net.minecraft.util.ResourceLocation;

import net.minecraftforge.client.model.*;
import net.minecraftforge.client.model.pipeline.LightUtil;
import net.minecraftforge.client.model.pipeline.UnpackedBakedQuad;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.fluids.BlockFluidBase;
import net.minecraftforge.fluids.Fluid;

import net.minecraft.client.renderer.block.model.ItemCameraTransforms.TransformType;
@SuppressWarnings("deprecation")
public class ModelBCFluid implements IModel {
    private final Fluid fluid;

    public ModelBCFluid(Fluid fluid) {
        this.fluid = fluid;
    }

    @Override
    public Collection<ResourceLocation> getDependencies() {
        return Collections.emptySet();
    }

    @Override
    public Collection<ResourceLocation> getTextures() {
        return ImmutableSet.of(fluid.getStill(), fluid.getFlowing());
    }

    @Override
    public IFlexibleBakedModel bake(IModelState state, VertexFormat format, Function<ResourceLocation, TextureAtlasSprite> bakedTextureGetter) {
        ImmutableMap<TransformType, TRSRTransformation> map = IPerspectiveAwareModel.MapWrapper.getTransforms(state);
        return new BakedFluid(state.apply(Optional.<IModelPart> absent()), map, format, fluid.getColor(), bakedTextureGetter.apply(fluid.getStill()),
                bakedTextureGetter.apply(fluid.getFlowing()), fluid.isGaseous(), Optional.<IExtendedBlockState> absent());
    }

    @Override
    public IModelState getDefaultState() {
        return ModelRotation.X0_Y0;
    }

    public static class BakedFluid implements ISmartBlockModel, IPerspectiveAwareModel {
        private static final int x[] = { 0, 0, 1, 1 };
        private static final int z[] = { 0, 1, 1, 0 };
        private static final float eps = 1e-3f;

        private final LoadingCache<Long, BakedFluid> modelCache = CacheBuilder.newBuilder().maximumSize(200).build(
                new CacheLoader<Long, BakedFluid>() {
                    @Override
                    public BakedFluid load(Long key) throws Exception {
                        boolean statePresent = (key & 1) != 0;
                        key >>>= 1;
                        int[] cornerRound = new int[4];
                        for (int i = 0; i < 4; i++) {
                            cornerRound[i] = (int) (key & 0x3FF);
                            key >>>= 10;
                        }
                        int flowRound = (int) (key & 0x7FF) - 1024;
                        return new BakedFluid(transformation, transforms, format, color, still, flowing, gas, statePresent, cornerRound, flowRound);
                    }
                });

        private final Optional<TRSRTransformation> transformation;
        private final ImmutableMap<TransformType, TRSRTransformation> transforms;
        private final VertexFormat format;
        private final int color;
        private final TextureAtlasSprite still, flowing;
        private final boolean gas;
        private final EnumMap<EnumFacing, List<BakedQuad>> faceQuads;

        public BakedFluid(Optional<TRSRTransformation> transformation, VertexFormat format, int color, TextureAtlasSprite still,
                TextureAtlasSprite flowing, boolean gas) {
            this(transformation, format, color, still, flowing, gas, Optional.<IExtendedBlockState> absent());
        }

        public BakedFluid(Optional<TRSRTransformation> transformation, VertexFormat format, int color, TextureAtlasSprite still,
                TextureAtlasSprite flowing, boolean gas, Optional<IExtendedBlockState> stateOption) {
            this(transformation, ImmutableMap.<TransformType, TRSRTransformation> of(), format, color, still, flowing, gas, stateOption);
        }

        public BakedFluid(Optional<TRSRTransformation> transformation, ImmutableMap<TransformType, TRSRTransformation> transforms,
                VertexFormat format, int color, TextureAtlasSprite still, TextureAtlasSprite flowing, boolean gas,
                Optional<IExtendedBlockState> stateOption) {
            this(transformation, transforms, format, color, still, flowing, gas, stateOption.isPresent(), getCorners(stateOption), getFlow(
                    stateOption));
        }

        private static int[] getCorners(Optional<IExtendedBlockState> stateOption) {
            int[] cornerRound = new int[] { 0, 0, 0, 0 };
            if (stateOption.isPresent()) {
                IExtendedBlockState state = stateOption.get();
                for (int i = 0; i < 4; i++) {
                    cornerRound[i] = Math.round(state.getValue(BlockFluidBase.LEVEL_CORNERS[i]) * 768);
                }
            }
            return cornerRound;
        }

        private static int getFlow(Optional<IExtendedBlockState> stateOption) {
            float flow = -1000;
            if (stateOption.isPresent()) {
                flow = stateOption.get().getValue(BlockFluidBase.FLOW_DIRECTION);
            }
            int flowRound = (int) Math.round(Math.toDegrees(flow));
            flowRound = MathHelper.func_76125_a(flowRound, -1000, 1000);
            return flowRound;
        }

        public BakedFluid(Optional<TRSRTransformation> transformation, ImmutableMap<TransformType, TRSRTransformation> transforms,
                VertexFormat format, int color, TextureAtlasSprite still, TextureAtlasSprite flowing, boolean gas, boolean statePresent,
                int[] cornerRound, int flowRound) {
            this.transformation = transformation;
            this.transforms = transforms;
            this.format = format;
            this.color = color;
            this.still = still;
            this.flowing = flowing;
            this.gas = gas;

            faceQuads = Maps.newEnumMap(EnumFacing.class);
            for (EnumFacing side : EnumFacing.values()) {
                faceQuads.put(side, ImmutableList.<BakedQuad> of());
            }

            if (statePresent) {
                float[] y = new float[4];
                for (int i = 0; i < 4; i++) {
                    if (gas) {
                        y[i] = 1 - cornerRound[i] / 768f;
                    } else {
                        y[i] = cornerRound[i] / 768f;
                    }
                }

                float flow = (float) Math.toRadians(flowRound);

                // top

                TextureAtlasSprite topSprite = flowing;
                float scale = 4;
                if (flow < -17F) {
                    flow = 0;
                    scale = 8;
                    topSprite = still;
                }

                float c = MathHelper.func_76134_b(flow) * scale;
                float s = MathHelper.func_76126_a(flow) * scale;

                EnumFacing side = gas ? EnumFacing.DOWN : EnumFacing.UP;
                UnpackedBakedQuad.Builder builder = new UnpackedBakedQuad.Builder(format);
                UnpackedBakedQuad.Builder builder2 = new UnpackedBakedQuad.Builder(format);
                builder.setQuadOrientation(side);
                builder2.setQuadOrientation(side);
                builder.setQuadColored();
                builder2.setQuadColored();
                int[] inverseSpriteArray = { 0, 3, 2, 1 };
                for (int i = gas ? 3 : 0; i != (gas ? -1 : 4); i += (gas ? -1 : 1)) {
//                    putVertex(builder, side, x[i], y[i], z[i], topSprite.getInterpolatedU(8 + c * (x[i] * 2 - 1) + s * (z[i] * 2 - 1)), topSprite
//                            .getInterpolatedV(8 + c * (x[(i + 1) % 4] * 2 - 1) + s * (z[(i + 1) % 4] * 2 - 1)));
                    int a = inverseSpriteArray[i];
                    putVertex(builder2, side, x[a], y[a], z[a], topSprite.func_94214_a(8 + c * (x[a] * 2 - 1) + s * (z[a] * 2 - 1)), topSprite
                            .func_94207_b(8 + c * (x[(a + 1) % 4] * 2 - 1) + s * (z[(a + 1) % 4] * 2 - 1)));
                }
//                faceQuads.put(side, ImmutableList.<BakedQuad> of(builder.build()));
                faceQuads.put(side, ImmutableList.<BakedQuad> of(builder2.build()));

                // bottom

                side = side.func_176734_d();
                builder = new UnpackedBakedQuad.Builder(format);
                builder.setQuadOrientation(side);
                builder.setQuadColored();
                for (int i = gas ? 3 : 0; i != (gas ? -1 : 4); i += (gas ? -1 : 1)) {
                    putVertex(builder, side, z[i], gas ? 1 : 0, x[i], still.func_94214_a(z[i] * 16), still.func_94207_b(x[i] * 16));
                }
//                faceQuads.put(side, ImmutableList.<BakedQuad> of(builder.build()));

                // sides

                for (int i = 0; i < 4; i++) {
                    side = EnumFacing.func_176731_b((5 - i) % 4);
                    BakedQuad q[] = new BakedQuad[2];

                    for (int k = 0; k < 2; k++) {
                        builder = new UnpackedBakedQuad.Builder(format);
                        builder.setQuadOrientation(side);
                        builder.setQuadColored();
                        for (int j = 0; j < 4; j++) {
                            int l = (k * 3) + (1 - 2 * k) * j;
                            float yl = z[l] * y[(i + x[l]) % 4];
                            if (gas && z[l] == 0) yl = 1;
                            putVertex(builder, side, x[(i + x[l]) % 4], yl, z[(i + x[l]) % 4], flowing.func_94214_a(x[l] * 8), flowing
                                    .func_94207_b((gas ? yl : 1 - yl) * 8));
                        }
                        q[k] = builder.build();
                    }
//                    faceQuads.put(side, ImmutableList.of(q[0], q[1]));
                }
            } else {
                // 1 quad for inventory
                UnpackedBakedQuad.Builder builder = new UnpackedBakedQuad.Builder(format);
                builder.setQuadOrientation(EnumFacing.UP);
                builder.setQuadColored();
                for (int i = 0; i < 4; i++) {
                    putVertex(builder, EnumFacing.UP, z[i], x[i], 0, still.func_94214_a(z[i] * 16), still.func_94207_b(x[i] * 16));
                }
                faceQuads.put(EnumFacing.SOUTH, ImmutableList.<BakedQuad> of(builder.build()));
            }
        }

        private void putVertex(UnpackedBakedQuad.Builder builder, EnumFacing side, float x, float y, float z, float u, float v) {
            for (int e = 0; e < format.func_177345_h(); e++) {
                switch (format.func_177348_c(e).func_177375_c()) {
                    case POSITION:
                        float[] data = new float[] { x - side.func_176730_m().func_177958_n() * eps, y, z - side.func_176730_m().func_177952_p() * eps, 1 };
                        if (transformation.isPresent() && transformation.get() != TRSRTransformation.identity()) {
                            Vector4f vec = new Vector4f(data);
                            transformation.get().getMatrix().transform(vec);
                            vec.get(data);
                        }
                        builder.put(e, data);
                        break;
                    case COLOR:
                        float d = LightUtil.diffuseLight(side);
                        builder.put(e, d * ((color >> 16) & 0xFF) / 255f, d * ((color >> 8) & 0xFF) / 255f, d * (color & 0xFF) / 255f, ((color >> 24)
                            & 0xFF) / 255f);
                        break;
                    case UV:
                        if (format.func_177348_c(e).func_177369_e() == 0) {
                            builder.put(e, u, v, 0f, 1f);
                            break;
                        }
                    case NORMAL:
                        builder.put(e, (float) side.func_82601_c(), (float) side.func_96559_d(), (float) side.func_82599_e(), 0f);
                        break;
                    default:
                        builder.put(e);
                        break;
                }
            }
        }

        @Override
        public boolean func_177555_b() {
            return false;
        }

        @Override
        public boolean func_177556_c() {
            return false;
        }

        @Override
        public boolean func_177553_d() {
            return false;
        }

        @Override
        public TextureAtlasSprite func_177554_e() {
            return still;
        }

        @Override
        public ItemCameraTransforms func_177552_f() {
            return ItemCameraTransforms.field_178357_a;
        }

        @Override
        public List<BakedQuad> func_177551_a(EnumFacing side) {
            return faceQuads.get(side);
        }

        @Override
        public List<BakedQuad> func_177550_a() {
            return ImmutableList.of();
        }

        @Override
        public VertexFormat getFormat() {
            return format;
        }

        @Override
        public IBakedModel handleBlockState(IBlockState state) {
            if (state instanceof IExtendedBlockState) {
                IExtendedBlockState exState = (IExtendedBlockState) state;
                int[] cornerRound = getCorners(Optional.of(exState));
                int flowRound = getFlow(Optional.of(exState));
                long key = flowRound + 1024;
                for (int i = 3; i >= 0; i--) {
                    key <<= 10;
                    key |= cornerRound[i];
                }
                key <<= 1;
                key |= 1;
                return modelCache.getUnchecked(key);
            }
            return this;
        }

        @Override
        public Pair<? extends IFlexibleBakedModel, Matrix4f> handlePerspective(TransformType type) {
            return IPerspectiveAwareModel.MapWrapper.handlePerspective(this, transforms, type);
        }
    }
}
