/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.commands;

import net.minecraft.command.CommandException;
import net.minecraft.command.ICommandSender;
import net.minecraft.command.WrongUsageException;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.ChatComponentText;
import net.minecraft.util.ChatComponentTranslation;
import net.minecraft.util.ChatStyle;
import net.minecraft.util.EnumChatFormatting;
import net.minecraft.util.StatCollector;
import net.minecraft.world.World;

public final class CommandHelpers {
    private CommandHelpers() {

    }

    public static World getWorld(ICommandSender sender, IModCommand command, String[] args, int worldArgIndex) throws CommandException {
        // Handle passed in world argument
        if (worldArgIndex < args.length) {
            try {
                int dim = Integer.parseInt(args[worldArgIndex]);
                World world = MinecraftServer.func_71276_C().func_71218_a(dim);
                if (world != null) {
                    return world;
                }
            } catch (Exception ex) {
                throwWrongUsage(sender, command);
            }
        }
        return getWorld(sender, command);
    }

    public static World getWorld(ICommandSender sender, IModCommand command) {
        if (sender instanceof EntityPlayer) {
            EntityPlayer player = (EntityPlayer) sender;
            return player.field_70170_p;
        }
        return MinecraftServer.func_71276_C().func_71218_a(0);
    }

    public static String[] getPlayers() {
        return MinecraftServer.func_71276_C().func_71213_z();
    }

    public static void sendLocalizedChatMessage(ICommandSender sender, String locTag, Object... args) {
        sender.func_145747_a(new ChatComponentTranslation(locTag, args));
    }

    public static void sendLocalizedChatMessage(ICommandSender sender, ChatStyle chatStyle, String locTag, Object... args) {
        ChatComponentTranslation chat = new ChatComponentTranslation(locTag, args);
        chat.func_150255_a(chatStyle);
        sender.func_145747_a(chat);
    }

    /** Avoid using this function if at all possible. Commands are processed on the server, which has no localization
     * information.
     * 
     * @param sender
     * @param message */
    public static void sendChatMessage(ICommandSender sender, String message) {
        sender.func_145747_a(new ChatComponentText(message));
    }

    public static void throwWrongUsage(ICommandSender sender, IModCommand command) throws WrongUsageException {
        throw new WrongUsageException(String.format(StatCollector.func_74838_a("command.buildcraft.help"), command.func_71518_a(sender)));
    }

    public static void processChildCommand(ICommandSender sender, SubCommand child, String[] args) throws CommandException {
        if (!sender.func_70003_b(child.getMinimumPermissionLevel(), child.getFullCommandString())) {
            throw new WrongUsageException(StatCollector.func_74838_a("command.buildcraft.noperms"));
        }
        String[] newargs = new String[args.length - 1];
        System.arraycopy(args, 1, newargs, 0, newargs.length);
        child.func_71515_b(sender, newargs);
    }

    public static void printHelp(ICommandSender sender, IModCommand command) {
        ChatStyle header = new ChatStyle();
        header.func_150238_a(EnumChatFormatting.GRAY);
        header.func_150227_a(true);
        sendLocalizedChatMessage(sender, header, "command.buildcraft." + command.getFullCommandString().replace(" ", ".") + ".format", command
                .getFullCommandString());
        ChatStyle body = new ChatStyle();
        body.func_150238_a(EnumChatFormatting.GRAY);
        if (command.func_71514_a().size() > 0) {
            sendLocalizedChatMessage(sender, body, "command.buildcraft.aliases", command.func_71514_a().toString().replace("[", "").replace("]",
                    ""));
        }
        if (command.getMinimumPermissionLevel() > 0) {
            sendLocalizedChatMessage(sender, body, "command.buildcraft.permlevel", command.getMinimumPermissionLevel());
        }
        sendLocalizedChatMessage(sender, body, "command.buildcraft." + command.getFullCommandString().replace(" ", ".") + ".help");
        if (!command.getChildren().isEmpty()) {
            sendLocalizedChatMessage(sender, "command.buildcraft.list");
            for (SubCommand child : command.getChildren()) {
                sendLocalizedChatMessage(sender, "command.buildcraft." + child.getFullCommandString().replace(" ", ".") + ".desc", child
                        .func_71517_b());
            }
        }
    }

    public static boolean processStandardCommands(ICommandSender sender, IModCommand command, String[] args) throws CommandException {
        if (args.length >= 1) {
            if ("help".equals(args[0])) {
                if (args.length >= 2) {
                    for (SubCommand child : command.getChildren()) {
                        if (matches(args[1], child)) {
                            child.printHelp(sender);
                            return true;
                        }
                    }
                }
                command.printHelp(sender);
                return true;
            }
            for (SubCommand child : command.getChildren()) {
                if (matches(args[0], child)) {
                    processChildCommand(sender, child, args);
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean matches(String commandName, IModCommand command) {
        if (commandName.equals(command.func_71517_b())) {
            return true;
        } else if (command.func_71514_a() != null) {
            for (String alias : command.func_71514_a()) {
                if (commandName.equals(alias)) {
                    return true;
                }
            }
        }
        return false;
    }
}
