/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.gui;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.InventoryPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.ICrafting;
import net.minecraft.inventory.Slot;
import net.minecraft.item.ItemStack;

import buildcraft.BuildCraftCore;
import buildcraft.api.core.BCLog;
import buildcraft.core.lib.gui.slots.IPhantomSlot;
import buildcraft.core.lib.gui.slots.SlotBase;
import buildcraft.core.lib.gui.widgets.Widget;
import buildcraft.core.lib.inventory.StackHelper;
import buildcraft.core.lib.network.PacketGuiWidget;

public abstract class BuildCraftContainer extends Container {

    private final EntityPlayer player;
    private List<Widget> widgets = new ArrayList<>();
    private int inventorySize;

    public BuildCraftContainer(EntityPlayer player, int inventorySize) {
        this.inventorySize = inventorySize;
        this.player = player;
    }

    public EntityPlayer getPlayer() {
        return player;
    }

    public List<Widget> getWidgets() {
        return widgets;
    }

    public void addSlot(Slot slot) {
        func_75146_a(slot);
    }

    public void addWidget(Widget widget) {
        widget.addToContainer(this);
        widgets.add(widget);
    }

    public void sendWidgetDataToClient(Widget widget, EntityPlayer player, byte[] data) {
        PacketGuiWidget pkt = new PacketGuiWidget(this.player, field_75152_c, widgets.indexOf(widget), data);
        BuildCraftCore.instance.sendToPlayer(player, pkt);
    }

    public void sendWidgetDataToServer(Widget widget, byte[] data) {
        PacketGuiWidget pkt = new PacketGuiWidget(this.player, field_75152_c, widgets.indexOf(widget), data);
        BuildCraftCore.instance.sendToServer(pkt);
    }

    public void handleWidgetData(int widgetId, byte[] data) {
        InputStream input = new ByteArrayInputStream(data);
        DataInputStream stream = new DataInputStream(input);
        if (widgetId < 0 || widgetId >= widgets.size()) BCLog.logger.warn("Found a packet with an invalid widget ID! (" + widgetId + ")");
        else {
            Widget widget = widgets.get(widgetId);
            try {
                if (getPlayer().func_70613_aW()) widget.handleServerPacketData(stream);
                else widget.handleClientPacketData(stream);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void func_75132_a(ICrafting player) {
        super.func_75132_a(player);
        for (Widget widget : widgets) {
            widget.initWidget(player);
        }
    }

    @Override
    public void func_75142_b() {
        super.func_75142_b();
        for (Widget widget : widgets) {
            for (ICrafting player : (List<ICrafting>) field_75149_d) {
                widget.updateWidget(player);
            }
        }
    }

    @Override
    public ItemStack func_75144_a(int slotNum, int mouseButton, int modifier, EntityPlayer player) {
        Slot slot = slotNum < 0 ? null : (Slot) this.field_75151_b.get(slotNum);
        if (slot instanceof IPhantomSlot) {
            return slotClickPhantom(slot, mouseButton, modifier, player);
        }
        return super.func_75144_a(slotNum, mouseButton, modifier, player);
    }

    private ItemStack slotClickPhantom(Slot slot, int mouseButton, int modifier, EntityPlayer player) {
        ItemStack stack = null;

        if (mouseButton == 2) {
            if (((IPhantomSlot) slot).canAdjust()) {
                slot.func_75215_d(null);
            }
        } else if (mouseButton == 0 || mouseButton == 1) {
            InventoryPlayer playerInv = player.field_71071_by;
            slot.func_75218_e();
            ItemStack stackSlot = slot.func_75211_c();
            ItemStack stackHeld = playerInv.func_70445_o();

            if (stackSlot != null) {
                stack = stackSlot.func_77946_l();
            }

            if (stackSlot == null) {
                if (stackHeld != null && slot.func_75214_a(stackHeld)) {
                    fillPhantomSlot(slot, stackHeld, mouseButton, modifier);
                }
            } else if (stackHeld == null) {
                adjustPhantomSlot(slot, mouseButton, modifier);
                slot.func_82870_a(player, playerInv.func_70445_o());
            } else if (slot.func_75214_a(stackHeld)) {
                if (StackHelper.canStacksMerge(stackSlot, stackHeld)) {
                    adjustPhantomSlot(slot, mouseButton, modifier);
                } else {
                    fillPhantomSlot(slot, stackHeld, mouseButton, modifier);
                }
            }
        }
        return stack;
    }

    protected void adjustPhantomSlot(Slot slot, int mouseButton, int modifier) {
        if (!((IPhantomSlot) slot).canAdjust()) {
            return;
        }
        ItemStack stackSlot = slot.func_75211_c();
        int stackSize;
        if (modifier == 1) {
            stackSize = mouseButton == 0 ? (stackSlot.field_77994_a + 1) / 2 : stackSlot.field_77994_a * 2;
        } else {
            stackSize = mouseButton == 0 ? stackSlot.field_77994_a - 1 : stackSlot.field_77994_a + 1;
        }

        if (stackSize > slot.func_75219_a()) {
            stackSize = slot.func_75219_a();
        }

        stackSlot.field_77994_a = stackSize;

        if (stackSlot.field_77994_a <= 0) {
            slot.func_75215_d(null);
        }
    }

    protected void fillPhantomSlot(Slot slot, ItemStack stackHeld, int mouseButton, int modifier) {
        if (!((IPhantomSlot) slot).canAdjust()) {
            return;
        }
        int stackSize = mouseButton == 0 ? stackHeld.field_77994_a : 1;
        if (stackSize > slot.func_75219_a()) {
            stackSize = slot.func_75219_a();
        }
        ItemStack phantomStack = stackHeld.func_77946_l();
        phantomStack.field_77994_a = stackSize;

        slot.func_75215_d(phantomStack);
    }

    protected boolean shiftItemStack(ItemStack stackToShift, int start, int end) {
        boolean changed = false;
        if (stackToShift.func_77985_e()) {
            for (int slotIndex = start; stackToShift.field_77994_a > 0 && slotIndex < end; slotIndex++) {
                Slot slot = (Slot) field_75151_b.get(slotIndex);
                ItemStack stackInSlot = slot.func_75211_c();
                if (stackInSlot != null && StackHelper.canStacksMerge(stackInSlot, stackToShift)) {
                    int resultingStackSize = stackInSlot.field_77994_a + stackToShift.field_77994_a;
                    int max = Math.min(stackToShift.func_77976_d(), slot.func_75219_a());
                    if (resultingStackSize <= max) {
                        stackToShift.field_77994_a = 0;
                        stackInSlot.field_77994_a = resultingStackSize;
                        slot.func_75218_e();
                        changed = true;
                    } else if (stackInSlot.field_77994_a < max) {
                        stackToShift.field_77994_a -= max - stackInSlot.field_77994_a;
                        stackInSlot.field_77994_a = max;
                        slot.func_75218_e();
                        changed = true;
                    }
                }
            }
        }
        if (stackToShift.field_77994_a > 0) {
            for (int slotIndex = start; stackToShift.field_77994_a > 0 && slotIndex < end; slotIndex++) {
                Slot slot = (Slot) field_75151_b.get(slotIndex);
                ItemStack stackInSlot = slot.func_75211_c();
                if (stackInSlot == null) {
                    int max = Math.min(stackToShift.func_77976_d(), slot.func_75219_a());
                    stackInSlot = stackToShift.func_77946_l();
                    stackInSlot.field_77994_a = Math.min(stackToShift.field_77994_a, max);
                    stackToShift.field_77994_a -= stackInSlot.field_77994_a;
                    slot.func_75215_d(stackInSlot);
                    slot.func_75218_e();
                    changed = true;
                }
            }
        }
        return changed;
    }

    private boolean tryShiftItem(ItemStack stackToShift, int numSlots) {
        for (int machineIndex = 0; machineIndex < numSlots - 9 * 4; machineIndex++) {
            Slot slot = (Slot) field_75151_b.get(machineIndex);
            if (slot instanceof SlotBase && !((SlotBase) slot).canShift()) {
                continue;
            }
            if (slot instanceof IPhantomSlot) {
                continue;
            }
            if (!slot.func_75214_a(stackToShift)) {
                continue;
            }
            if (shiftItemStack(stackToShift, machineIndex, machineIndex + 1)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public ItemStack func_82846_b(EntityPlayer player, int slotIndex) {
        ItemStack originalStack = null;
        Slot slot = (Slot) field_75151_b.get(slotIndex);
        int numSlots = field_75151_b.size();
        if (slot != null && slot.func_75216_d()) {
            ItemStack stackInSlot = slot.func_75211_c();
            originalStack = stackInSlot.func_77946_l();
            if (slotIndex >= numSlots - 9 * 4 && tryShiftItem(stackInSlot, numSlots)) {
                // NOOP
            } else if (slotIndex >= numSlots - 9 * 4 && slotIndex < numSlots - 9) {
                if (!shiftItemStack(stackInSlot, numSlots - 9, numSlots)) {
                    return null;
                }
            } else if (slotIndex >= numSlots - 9 && slotIndex < numSlots) {
                if (!shiftItemStack(stackInSlot, numSlots - 9 * 4, numSlots - 9)) {
                    return null;
                }
            } else if (!shiftItemStack(stackInSlot, numSlots - 9 * 4, numSlots)) {
                return null;
            }
            slot.func_75220_a(stackInSlot, originalStack);
            if (stackInSlot.field_77994_a <= 0) {
                slot.func_75215_d(null);
            } else {
                slot.func_75218_e();
            }
            if (stackInSlot.field_77994_a == originalStack.field_77994_a) {
                return null;
            }
            slot.func_82870_a(player, stackInSlot);
        }
        return originalStack;
    }

    public int getInventorySize() {
        return inventorySize;
    }
}
