/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.inventory;

import net.minecraft.entity.item.EntityItem;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.InventoryLargeChest;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.nbt.NBTTagString;
import net.minecraft.tileentity.TileEntityChest;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.World;

import net.minecraftforge.common.util.Constants;

import buildcraft.api.core.IInvSlot;
import buildcraft.api.core.IStackFilter;
import buildcraft.core.lib.utils.BlockUtils;

public final class InvUtils {

    /** Deactivate constructor */
    private InvUtils() {}

    public static int countItems(IInventory inv, EnumFacing side, IStackFilter filter) {
        int count = 0;
        for (IInvSlot slot : InventoryIterator.getIterable(inv, side)) {
            ItemStack stack = slot.getStackInSlot();
            if (stack != null && filter.matches(stack)) {
                count += stack.field_77994_a;
            }
        }
        return count;
    }

    public static boolean containsItem(IInventory inv, EnumFacing side, IStackFilter filter) {
        for (IInvSlot slot : InventoryIterator.getIterable(inv, side)) {
            ItemStack stack = slot.getStackInSlot();
            if (stack != null && filter.matches(stack)) {
                return true;
            }
        }
        return false;
    }

    /** Checks if there is room for the ItemStack in the inventory.
     *
     * @param stack The ItemStack
     * @param dest The IInventory
     * @return true if room for stack */
    public static boolean isRoomForStack(ItemStack stack, EnumFacing side, IInventory dest) {
        if (stack == null || dest == null) {
            return false;
        }
        ITransactor tran = Transactor.getTransactorFor(dest, side);
        return tran.add(stack, false).field_77994_a > 0;
    }

    /** Attempts to move a single item from one inventory to another.
     *
     * @param source
     * @param dest
     * @param filter an IStackFilter to match against
     * @return null if nothing was moved, the stack moved otherwise */
    public static ItemStack moveOneItem(IInventory source, EnumFacing output, IInventory dest, EnumFacing input, IStackFilter filter) {
        ITransactor imSource = Transactor.getTransactorFor(source, output);
        ItemStack stack = imSource.remove(filter, false);
        if (stack != null) {
            ITransactor imDest = Transactor.getTransactorFor(dest, input);
            int moved = imDest.add(stack, true).field_77994_a;
            if (moved > 0) {
                imSource.remove(filter, true);
                return stack;
            }
        }
        return null;
    }

    /* STACK DROPS */
    public static void dropItems(World world, ItemStack stack, BlockPos pos) {
        if (stack == null || stack.field_77994_a <= 0) {
            return;
        }

        float f1 = 0.7F;
        double d = (world.field_73012_v.nextFloat() * f1) + (1.0F - f1) * 0.5D;
        double d1 = (world.field_73012_v.nextFloat() * f1) + (1.0F - f1) * 0.5D;
        double d2 = (world.field_73012_v.nextFloat() * f1) + (1.0F - f1) * 0.5D;
        EntityItem entityitem = new EntityItem(world, pos.func_177958_n() + d, pos.func_177956_o() + d1, pos.func_177952_p() + d2, stack);
        entityitem.func_174869_p();

        world.func_72838_d(entityitem);
    }

    public static void dropItems(World world, IInventory inv, BlockPos pos) {
        for (int slot = 0; slot < inv.func_70302_i_(); ++slot) {
            ItemStack items = inv.func_70301_a(slot);

            if (items != null && items.field_77994_a > 0) {
                dropItems(world, inv.func_70301_a(slot).func_77946_l(), pos);
            }
        }
    }

    public static void wipeInventory(IInventory inv) {
        for (int slot = 0; slot < inv.func_70302_i_(); ++slot) {
            inv.func_70299_a(slot, null);
        }
    }

    public static NBTTagCompound getItemData(ItemStack stack) {
        NBTTagCompound nbt = stack.func_77978_p();
        if (nbt == null) {
            nbt = new NBTTagCompound();
            stack.func_77982_d(nbt);
        }
        return nbt;
    }

    public static void addItemToolTip(ItemStack stack, String msg) {
        NBTTagCompound nbt = getItemData(stack);
        NBTTagCompound display = nbt.func_74775_l("display");
        nbt.func_74782_a("display", display);
        NBTTagList lore = display.func_150295_c("Lore", Constants.NBT.TAG_STRING);
        display.func_74782_a("Lore", lore);
        lore.func_74742_a(new NBTTagString(msg));
    }

    public static void writeInvToNBT(IInventory inv, String tag, NBTTagCompound data) {
        NBTTagList list = new NBTTagList();
        for (byte slot = 0; slot < inv.func_70302_i_(); slot++) {
            ItemStack stack = inv.func_70301_a(slot);
            if (stack != null) {
                NBTTagCompound itemTag = new NBTTagCompound();
                itemTag.func_74774_a("Slot", slot);
                stack.func_77955_b(itemTag);
                list.func_74742_a(itemTag);
            }
        }
        data.func_74782_a(tag, list);
    }

    public static void readInvFromNBT(IInventory inv, String tag, NBTTagCompound data) {
        NBTTagList list = data.func_150295_c(tag, Constants.NBT.TAG_COMPOUND);
        for (byte entry = 0; entry < list.func_74745_c(); entry++) {
            NBTTagCompound itemTag = list.func_150305_b(entry);
            int slot = itemTag.func_74771_c("Slot");
            if (slot >= 0 && slot < inv.func_70302_i_()) {
                ItemStack stack = ItemStack.func_77949_a(itemTag);
                inv.func_70299_a(slot, stack);
            }
        }
    }

    public static void readStacksFromNBT(NBTTagCompound nbt, String name, ItemStack[] stacks) {
        NBTTagList nbttaglist = nbt.func_150295_c(name, Constants.NBT.TAG_COMPOUND);

        for (int i = 0; i < stacks.length; ++i) {
            if (i < nbttaglist.func_74745_c()) {
                NBTTagCompound nbttagcompound2 = nbttaglist.func_150305_b(i);

                stacks[i] = ItemStack.func_77949_a(nbttagcompound2);
            } else {
                stacks[i] = null;
            }
        }
    }

    public static void writeStacksToNBT(NBTTagCompound nbt, String name, ItemStack[] stacks) {
        NBTTagList nbttaglist = new NBTTagList();

        for (ItemStack stack : stacks) {
            NBTTagCompound cpt = new NBTTagCompound();
            nbttaglist.func_74742_a(cpt);
            if (stack != null) {
                stack.func_77955_b(cpt);
            }

        }

        nbt.func_74782_a(name, nbttaglist);
    }

    public static ItemStack consumeItem(ItemStack stack) {
        if (stack.field_77994_a == 1) {
            if (stack.func_77973_b().hasContainerItem(stack)) {
                return stack.func_77973_b().getContainerItem(stack);
            } else {
                return null;
            }
        } else {
            stack.func_77979_a(1);

            return stack;
        }
    }

    /** Ensures that the given inventory is the full inventory, i.e. takes double chests into account.
     *
     * @param inv
     * @return Modified inventory if double chest, unmodified otherwise. */
    public static IInventory getInventory(IInventory inv) {
        if (inv instanceof TileEntityChest) {
            TileEntityChest adjacent = BlockUtils.getOtherDoubleChest((TileEntityChest) inv);
            if (adjacent != null) {
                return new InventoryLargeChest("", (TileEntityChest) inv, adjacent);
            }
            return inv;
        }
        return inv;
    }

    public static IInvSlot getItem(IInventory inv, IStackFilter filter) {
        for (IInvSlot s : InventoryIterator.getIterable(inv)) {
            if (s.getStackInSlot() != null && filter.matches(s.getStackInSlot())) {
                return s;
            }
        }

        return null;
    }
}
