/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.inventory;

import net.minecraft.item.ItemStack;

import net.minecraftforge.oredict.OreDictionary;

import buildcraft.api.items.IList;

public class StackHelper {

    protected StackHelper() {}

    /* STACK MERGING */

    /** Checks if two ItemStacks are identical enough to be merged
     *
     * @param stack1 - The first stack
     * @param stack2 - The second stack
     * @return true if stacks can be merged, false otherwise */
    public static boolean canStacksMerge(ItemStack stack1, ItemStack stack2) {
        if (stack1 == null || stack2 == null) {
            return false;
        }
        if (!stack1.func_77969_a(stack2)) {
            return false;
        }
        if (!ItemStack.func_77970_a(stack1, stack2)) {
            return false;
        }
        return true;

    }

    public static boolean canStacksOrListsMerge(ItemStack stack1, ItemStack stack2) {
        if (stack1 == null || stack2 == null) {
            return false;
        }

        if (stack1.func_77973_b() instanceof IList) {
            IList list = (IList) stack1.func_77973_b();
            return list.matches(stack1, stack2);
        } else if (stack2.func_77973_b() instanceof IList) {
            IList list = (IList) stack2.func_77973_b();
            return list.matches(stack2, stack1);
        }

        if (!stack1.func_77969_a(stack2)) {
            return false;
        }
        if (!ItemStack.func_77970_a(stack1, stack2)) {
            return false;
        }
        return true;

    }

    /** Merges mergeSource into mergeTarget
     *
     * @param mergeSource - The stack to merge into mergeTarget, this stack is not modified
     * @param mergeTarget - The target merge, this stack is modified if doMerge is set
     * @param doMerge - To actually do the merge
     * @return The number of items that was successfully merged. */
    public static int mergeStacks(ItemStack mergeSource, ItemStack mergeTarget, boolean doMerge) {
        if (!canStacksMerge(mergeSource, mergeTarget)) {
            return 0;
        }
        int mergeCount = Math.min(mergeTarget.func_77976_d() - mergeTarget.field_77994_a, mergeSource.field_77994_a);
        if (mergeCount < 1) {
            return 0;
        }
        if (doMerge) {
            mergeTarget.field_77994_a += mergeCount;
        }
        return mergeCount;
    }

    /* ITEM COMPARISONS */
    /** Determines whether the given ItemStack should be considered equivalent for crafting purposes.
     *
     * @param base The stack to compare to.
     * @param comparison The stack to compare.
     * @param oreDictionary true to take the Forge OreDictionary into account.
     * @return true if comparison should be considered a crafting equivalent for base. */
    public static boolean isCraftingEquivalent(ItemStack base, ItemStack comparison, boolean oreDictionary) {
        if (isMatchingItem(base, comparison, true, false)) {
            return true;
        }

        if (oreDictionary) {
            int[] idBase = OreDictionary.getOreIDs(base);
            if (idBase.length > 0) {
                for (int id : idBase) {
                    for (ItemStack itemstack : OreDictionary.getOres(OreDictionary.getOreName(id))) {
                        if (comparison.func_77973_b() == itemstack.func_77973_b() && (itemstack.func_77952_i() == OreDictionary.WILDCARD_VALUE || comparison
                                .func_77952_i() == itemstack.func_77952_i())) {
                            return true;
                        }
                    }
                }
            }
        }

        return false;
    }

    public static boolean isCraftingEquivalent(int[] oreIDs, ItemStack comparison) {
        if (oreIDs.length > 0) {
            for (int id : oreIDs) {
                for (ItemStack itemstack : OreDictionary.getOres(OreDictionary.getOreName(id))) {
                    if (comparison.func_77973_b() == itemstack.func_77973_b() && (itemstack.func_77952_i() == OreDictionary.WILDCARD_VALUE || comparison
                            .func_77952_i() == itemstack.func_77952_i())) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    public static boolean isMatchingItemOrList(final ItemStack a, final ItemStack b) {
        if (a == null || b == null) {
            return false;
        }

        if (a.func_77973_b() instanceof IList) {
            IList list = (IList) a.func_77973_b();
            return list.matches(a, b);
        } else if (b.func_77973_b() instanceof IList) {
            IList list = (IList) b.func_77973_b();
            return list.matches(b, a);
        }

        return isMatchingItem(a, b, true, false);
    }

    /** Compares item id, damage and NBT. Accepts wildcard damage. Ignores damage entirely if the item doesn't have
     * subtypes.
     *
     * @param base The stack to compare to.
     * @param comparison The stack to compare.
     * @return true if id, damage and NBT match. */
    public static boolean isMatchingItem(final ItemStack base, final ItemStack comparison) {
        return isMatchingItem(base, comparison, true, true);
    }

    /** This variant also checks damage for damaged items. */
    public static boolean isEqualItem(final ItemStack a, final ItemStack b) {
        if (isMatchingItem(a, b, false, true)) {
            return isWildcard(a) || isWildcard(b) || a.func_77952_i() == b.func_77952_i();
        } else {
            return false;
        }
    }

    /** Compares item id, and optionally damage and NBT. Accepts wildcard damage. Ignores damage entirely if the item
     * doesn't have subtypes.
     *
     * @param a ItemStack
     * @param b ItemStack
     * @param matchDamage
     * @param matchNBT
     * @return true if matches */
    public static boolean isMatchingItem(final ItemStack a, final ItemStack b, final boolean matchDamage, final boolean matchNBT) {
        if (a == null || b == null) {
            return false;
        }

        if (a.func_77973_b() != b.func_77973_b()) {
            return false;
        }
        if (matchDamage && a.func_77981_g()) {
            if (!isWildcard(a) && !isWildcard(b)) {
                if (a.func_77952_i() != b.func_77952_i()) {
                    return false;
                }
            }
        }
        if (matchNBT) {
            if (a.func_77978_p() != null && !a.func_77978_p().equals(b.func_77978_p())) {
                return false;
            }
        }
        return true;
    }

    public static boolean isWildcard(ItemStack stack) {
        return isWildcard(stack.func_77952_i());
    }

    public static boolean isWildcard(int damage) {
        return damage == -1 || damage == OreDictionary.WILDCARD_VALUE;
    }
}
