/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.utils;

import java.util.ArrayList;
import java.util.List;

import net.minecraft.block.Block;
import net.minecraft.block.BlockLiquid;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.init.Blocks;
import net.minecraft.item.ItemStack;
import net.minecraft.network.play.server.S27PacketExplosion;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.tileentity.TileEntityChest;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.Explosion;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.chunk.Chunk;
import net.minecraft.world.chunk.Chunk.EnumCreateEntityType;

import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.event.ForgeEventFactory;
import net.minecraftforge.event.world.BlockEvent.BreakEvent;
import net.minecraftforge.fluids.*;
import net.minecraftforge.fml.common.FMLCommonHandler;

import buildcraft.BuildCraftCore;
import buildcraft.api.blueprints.BuilderAPI;
import buildcraft.core.proxy.CoreProxy;

import net.minecraft.world.chunk.Chunk.EnumCreateEntityType;
public final class BlockUtils {
    /** Deactivate constructor */
    private BlockUtils() {}

    public static List<ItemStack> getItemStackFromBlock(WorldServer world, BlockPos pos) {
        IBlockState state = world.func_180495_p(pos);
        Block block = state.func_177230_c();
        if (block == null || block.isAir(world, pos)) {
            return null;
        }

        List<ItemStack> dropsList = block.getDrops(world, pos, state, 0);
        float dropChance = ForgeEventFactory.fireBlockHarvesting(dropsList, world, pos, state, 0, 1.0F, false, CoreProxy.proxy.getBuildCraftPlayer(
                world).get());

        ArrayList<ItemStack> returnList = new ArrayList<>();
        for (ItemStack s : dropsList) {
            if (world.field_73012_v.nextFloat() <= dropChance) {
                returnList.add(s);
            }
        }

        return returnList;
    }

    public static boolean breakBlock(WorldServer world, BlockPos pos) {
        return breakBlock(world, pos, BuildCraftCore.itemLifespan * 20);
    }

    public static boolean breakBlock(WorldServer world, BlockPos pos, int forcedLifespan) {
        List<ItemStack> items = new ArrayList<>();

        if (breakBlock(world, pos, items)) {
            for (ItemStack item : items) {
                dropItem(world, pos, forcedLifespan, item);
            }
            return true;
        }
        return false;
    }

    public static boolean harvestBlock(WorldServer world, BlockPos pos, ItemStack tool) {
        BreakEvent breakEvent = new BreakEvent(world, pos, world.func_180495_p(pos), CoreProxy.proxy.getBuildCraftPlayer(world).get());
        MinecraftForge.EVENT_BUS.post(breakEvent);

        if (breakEvent.isCanceled()) {
            return false;
        }

        IBlockState state = world.func_180495_p(pos);

        EntityPlayer player = CoreProxy.proxy.getBuildCraftPlayer(world, pos).get();

        if (!state.func_177230_c().canHarvestBlock(world, pos, player)) {
            return false;
        }

        state.func_177230_c().func_176208_a(world, pos, state, player);
        state.func_177230_c().func_180657_a(world, player, pos, state, world.func_175625_s(pos));
        world.func_175698_g(pos);

        return true;
    }

    public static EntityPlayer getFakePlayerWithTool(WorldServer world, BlockPos pos, ItemStack tool) {
        EntityPlayer player = CoreProxy.proxy.getBuildCraftPlayer(world, pos).get();
        int i = 0;

        while (player.func_70694_bm() != tool && i < 9) {
            if (i > 0) {
                player.field_71071_by.func_70299_a(i - 1, null);
            }

            player.field_71071_by.func_70299_a(i, tool);
            i++;
        }

        return player;
    }

    public static boolean breakBlock(WorldServer world, BlockPos pos, List<ItemStack> drops) {
        BreakEvent breakEvent = new BreakEvent(world, pos, world.func_180495_p(pos), CoreProxy.proxy.getBuildCraftPlayer(world).get());
        MinecraftForge.EVENT_BUS.post(breakEvent);

        if (breakEvent.isCanceled()) {
            return false;
        }

        if (!world.func_175623_d(pos) && !world.field_72995_K && world.func_82736_K().func_82766_b("doTileDrops")) {
            drops.addAll(getItemStackFromBlock(world, pos));
        }
        world.func_175698_g(pos);

        return true;
    }

    public static void dropItem(WorldServer world, BlockPos pos, int forcedLifespan, ItemStack stack) {
        float var = 0.7F;
        double dx = world.field_73012_v.nextFloat() * var + (1.0F - var) * 0.5D;
        double dy = world.field_73012_v.nextFloat() * var + (1.0F - var) * 0.5D;
        double dz = world.field_73012_v.nextFloat() * var + (1.0F - var) * 0.5D;
        EntityItem entityitem = new EntityItem(world, pos.func_177958_n() + dx, pos.func_177956_o() + dy, pos.func_177952_p() + dz, stack);

        entityitem.lifespan = forcedLifespan;
        entityitem.func_174869_p();

        world.func_72838_d(entityitem);
    }

    public static boolean canChangeBlock(World world, BlockPos pos) {
        return canChangeBlock(world.func_180495_p(pos), world, pos);
    }

    public static boolean canChangeBlock(IBlockState state, World world, BlockPos pos) {
        if (state == null) return true;

        Block block = state.func_177230_c();
        if (block == null || block.isAir(world, pos)) {
            return true;
        }

        if (isUnbreakableBlock(world, pos, block)) {
            return false;
        }

        if (block == Blocks.field_150353_l || block == Blocks.field_150356_k) {
            return false;
        } else if (block instanceof IFluidBlock && ((IFluidBlock) block).getFluid() != null) {
            Fluid f = ((IFluidBlock) block).getFluid();
            if (f.getDensity(world, pos) >= 3000) {
                return false;
            }
        }

        return true;
    }

    public static float getBlockHardnessMining(World world, BlockPos pos, Block b) {
        if (world instanceof WorldServer && !BuildCraftCore.miningAllowPlayerProtectedBlocks) {
            float relativeHardness = b.func_180647_a(CoreProxy.proxy.getBuildCraftPlayer((WorldServer) world).get(), world, pos);
            if (relativeHardness <= 0.0F) { // Forge's getPlayerRelativeBlockHardness hook returns 0.0F if the hardness
                                            // is < 0.0F.
                return -1.0F;
            }
        }
        return b.func_176195_g(world, pos);
    }

    public static boolean isUnbreakableBlock(World world, BlockPos pos, Block b) {
        if (b == null) {
            return false;
        }

        return getBlockHardnessMining(world, pos, b) < 0;
    }

    public static boolean isUnbreakableBlock(World world, BlockPos pos) {
        return isUnbreakableBlock(world, pos, world.func_180495_p(pos).func_177230_c());
    }

    /** Returns true if a block cannot be harvested without a tool. */
    public static boolean isToughBlock(World world, BlockPos pos) {
        return !world.func_180495_p(pos).func_177230_c().func_149688_o().func_76229_l();
    }

    public static boolean isFullFluidBlock(World world, BlockPos pos) {
        return isFullFluidBlock(world.func_180495_p(pos), world, pos);
    }

    public static boolean isFullFluidBlock(IBlockState state, World world, BlockPos pos) {
        Block block = state.func_177230_c();
        if (block instanceof IFluidBlock) {
            FluidStack fluid = ((IFluidBlock) block).drain(world, pos, false);
            if (fluid == null || fluid.amount > 0) {
                return true;
            }
            return false;
        } else if (block instanceof BlockLiquid) {
            int level = (Integer) state.func_177229_b(BlockLiquid.field_176367_b);
            return level == 0;
        }
        return false;
    }

    public static Fluid getFluid(Block block) {
        return FluidRegistry.lookupFluidForBlock(block);
    }

    public static FluidStack drainBlock(World world, BlockPos pos, boolean doDrain) {
        return drainBlock(world.func_180495_p(pos), world, pos, doDrain);
    }

    public static FluidStack drainBlock(IBlockState state, World world, BlockPos pos, boolean doDrain) {
        Block block = state.func_177230_c();
        Fluid fluid = FluidRegistry.lookupFluidForBlock(block);

        if (fluid != null && FluidRegistry.isFluidRegistered(fluid)) {
            if (block instanceof IFluidBlock) {
                IFluidBlock fluidBlock = (IFluidBlock) block;
                if (!fluidBlock.canDrain(world, pos)) {
                    return null;
                }
                return fluidBlock.drain(world, pos, doDrain);
            } else {
                int level = (Integer) state.func_177229_b(BlockLiquid.field_176367_b);
                if (level != 0) {
                    return null;
                }

                if (doDrain) {
                    world.func_175698_g(pos);
                }

                return new FluidStack(fluid, FluidContainerRegistry.BUCKET_VOLUME);
            }
        } else {
            return null;
        }
    }

    /** Create an explosion which only affects a single block. */
    @SuppressWarnings("unchecked")
    public static void explodeBlock(World world, BlockPos pos) {
        if (FMLCommonHandler.instance().getEffectiveSide().isClient()) {
            return;
        }

        double x = pos.func_177958_n() + 0.5;
        double y = pos.func_177956_o() + 0.5;
        double z = pos.func_177952_p() + 0.5;

        Explosion explosion = new Explosion(world, null, x, y, z, 3f, false, false);
        explosion.func_180343_e().add(pos);
        explosion.func_77279_a(true);

        for (EntityPlayer player : (List<EntityPlayer>) world.field_73010_i) {
            if (!(player instanceof EntityPlayerMP)) {
                continue;
            }

            if (player.func_174818_b(pos) < 4096) {
                ((EntityPlayerMP) player).field_71135_a.func_147359_a(new S27PacketExplosion(x, y, z, 3f, explosion.func_180343_e(),
                        null));
            }
        }
    }

    public static int computeBlockBreakEnergy(World world, BlockPos pos) {
        float hardness = world.func_180495_p(pos).func_177230_c().func_176195_g(world, pos);
        return (int) Math.floor(BuilderAPI.BREAK_ENERGY * BuildCraftCore.miningMultiplier * ((hardness + 1) * 2));
    }

    /** The following functions let you avoid unnecessary chunk loads, which is nice. */
    public static TileEntity getTileEntity(World world, BlockPos pos) {
        return getTileEntity(world, pos, false);
    }

    public static TileEntity getTileEntity(World world, BlockPos pos, boolean force) {
        if (!force) {
            if (pos.func_177956_o() < 0 || pos.func_177956_o() > 255) {
                return null;
            }
            Chunk chunk = ThreadSafeUtils.getChunk(world, pos.func_177958_n() >> 4, pos.func_177952_p() >> 4);
            return chunk != null ? chunk.func_177424_a(pos, EnumCreateEntityType.CHECK) : null;
        } else {
            return world.func_175625_s(pos);
        }
    }

    public static IBlockState getBlockState(World world, BlockPos pos) {
        return getBlockState(world, pos, false);
    }

    public static IBlockState getBlockState(World world, BlockPos pos, boolean force) {
        if (!force) {
            if (pos.func_177956_o() < 0 || pos.func_177956_o() >= world.func_72800_K()) {
                return Blocks.field_150350_a.func_176223_P();
            }
            Chunk chunk = ThreadSafeUtils.getChunk(world, pos.func_177958_n() >> 4, pos.func_177952_p() >> 4);
            return chunk != null ? chunk.func_177435_g(pos) : Blocks.field_150350_a.func_176223_P();
        } else {
            if (pos.func_177956_o() < 0 || pos.func_177956_o() > 255) {
                return Blocks.field_150350_a.func_176223_P();
            }
            Chunk chunk = ThreadSafeUtils.getChunk(world, pos.func_177958_n() >> 4, pos.func_177952_p() >> 4);
            return chunk != null ? chunk.func_177435_g(pos) : Blocks.field_150350_a.func_176223_P();
        }
    }

    // Meta is hidden internally, so we shouldn't even try
    // public static int getBlockMetadata(World world, BlockPos pos) {
    // return getBlockMetadata(world, pos, false);
    //
    // }

    // public static int getBlockMetadata(World world, BlockPos pos, boolean force) {
    // if (!force) {
    // if (y < 0 || y > 255) {
    // return 0;
    // }
    // Chunk chunk = getChunkUnforced(world, x >> 4, z >> 4);
    // return chunk != null ? chunk.getBlockMetadata(x & 15, y, z & 15) : 0;
    // } else {
    // return world.getBlockMetadata(pos);
    // }
    // }

    public static boolean useItemOnBlock(World world, EntityPlayer player, ItemStack stack, BlockPos pos, EnumFacing direction) {
        boolean done = stack.func_77973_b().onItemUseFirst(stack, player, world, pos, direction, 0.5F, 0.5F, 0.5F);

        if (!done) {
            done = stack.func_77973_b().func_180614_a(stack, player, world, pos, direction, 0.5F, 0.5F, 0.5F);
        }
        return done;
    }

    public static void onComparatorUpdate(World world, BlockPos pos, Block block) {
        world.func_175666_e(pos, block);
    }

    public static TileEntityChest getOtherDoubleChest(TileEntity inv) {
        if (inv instanceof TileEntityChest) {
            TileEntityChest chest = (TileEntityChest) inv;

            TileEntityChest adjacent = null;

            chest.func_145979_i();

            if (chest.field_145991_k != null) {
                adjacent = chest.field_145991_k;
            }

            if (chest.field_145990_j != null) {
                adjacent = chest.field_145990_j;
            }

            if (chest.field_145992_i != null) {
                adjacent = chest.field_145992_i;
            }

            if (chest.field_145988_l != null) {
                adjacent = chest.field_145988_l;
            }

            return adjacent;
        }
        return null;
    }
}
