/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.core.lib.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.UUID;

import net.minecraft.item.ItemStack;
import net.minecraft.nbt.*;
import net.minecraft.nbt.NBTBase.NBTPrimitive;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Vec3;

import net.minecraftforge.common.util.Constants;

import buildcraft.api.core.BCLog;

public final class NBTUtils {
    /** Deactivate constructor */
    private NBTUtils() {

    }

    public static NBTTagCompound load(byte[] data) {
        try {
            NBTTagCompound nbt = CompressedStreamTools.func_74796_a(new ByteArrayInputStream(data));
            return nbt;
        } catch (IOException e) {
            e.printStackTrace();
        }

        return null;
    }

    public static NBTTagCompound getItemData(ItemStack stack) {
        if (stack == null) {
            return new NBTTagCompound();
        }
        NBTTagCompound nbt = stack.func_77978_p();
        if (nbt == null) {
            nbt = new NBTTagCompound();
            stack.func_77982_d(nbt);
        }
        return nbt;
    }

    public static void writeUUID(NBTTagCompound data, String tag, UUID uuid) {
        if (uuid == null) {
            return;
        }
        NBTTagCompound nbtTag = new NBTTagCompound();
        nbtTag.func_74772_a("most", uuid.getMostSignificantBits());
        nbtTag.func_74772_a("least", uuid.getLeastSignificantBits());
        data.func_74782_a(tag, nbtTag);
    }

    public static UUID readUUID(NBTTagCompound data, String tag) {
        if (data.func_74764_b(tag)) {
            NBTTagCompound nbtTag = data.func_74775_l(tag);
            return new UUID(nbtTag.func_74763_f("most"), nbtTag.func_74763_f("least"));
        }
        return null;
    }

    public static byte[] save(NBTTagCompound compound) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            CompressedStreamTools.func_74799_a(compound, baos);
            return baos.toByteArray();
        } catch (IOException e) {
            e.printStackTrace();
            return new byte[0];
        }
    }

    public static NBTBase writeBlockPos(BlockPos pos) {
        if (pos == null) return null;
        return new NBTTagIntArray(new int[] { pos.func_177958_n(), pos.func_177956_o(), pos.func_177952_p() });
    }

    public static BlockPos readBlockPos(NBTBase base) {
        if (base == null) return null;
        switch (base.func_74732_a()) {
            case Constants.NBT.TAG_INT_ARRAY: {
                int[] array = ((NBTTagIntArray) base).func_150302_c();
                return new BlockPos(array[0], array[1], array[2]);
            }
            case Constants.NBT.TAG_COMPOUND: {
                NBTTagCompound nbt = (NBTTagCompound) base;
                BlockPos pos = BlockPos.field_177992_a;
                if (nbt.func_74764_b("i")) {
                    int i = nbt.func_74762_e("i");
                    int j = nbt.func_74762_e("j");
                    int k = nbt.func_74762_e("k");
                    pos = new BlockPos(i, j, k);
                } else if (nbt.func_74764_b("x")) {
                    int x = nbt.func_74762_e("x");
                    int y = nbt.func_74762_e("y");
                    int z = nbt.func_74762_e("z");
                    pos = new BlockPos(x, y, z);
                } else if (nbt.func_74764_b("pos")) {
                    return readBlockPos(nbt.func_74781_a("pos"));
                } else {
                    BCLog.logger.warn("Attempted to read a block positions from a compound tag without the correct sub-tags! (" + base + ")", new Throwable());
                }
                return pos;
            }
        }
        BCLog.logger.warn("Attempted to read a block position from an invalid tag! (" + base + ")", new Throwable());
        return BlockPos.field_177992_a;
    }

    public static NBTTagList writeVec3(Vec3 vec3) {
        NBTTagList list = new NBTTagList();
        list.func_74742_a(new NBTTagDouble(vec3.field_72450_a));
        list.func_74742_a(new NBTTagDouble(vec3.field_72448_b));
        list.func_74742_a(new NBTTagDouble(vec3.field_72449_c));
        return list;
    }

    public static Vec3 readVec3(NBTTagCompound nbt, String tagName) {
        return readVec3(nbt.func_150295_c(tagName, Constants.NBT.TAG_DOUBLE));
    }

    public static Vec3 readVec3(NBTTagList list) {
        return new Vec3(list.func_150309_d(0), list.func_150309_d(1), list.func_150309_d(2));
    }

    private static final String NULL_ENUM_STRING = "_NULL";

    public static <E extends Enum<E>> NBTBase writeEnum(E value) {
        if (value == null) {
            return new NBTTagString(NULL_ENUM_STRING);
        }
        return new NBTTagString(value.name());
    }

    public static <E extends Enum<E>> E readEnum(NBTBase nbt, Class<E> clazz) {
        if (nbt instanceof NBTTagString) {
            String value = ((NBTTagString) nbt).func_150285_a_();
            if (NULL_ENUM_STRING.equals(value)) {
                return null;
            }
            try {
                return Enum.valueOf(clazz, value);
            } catch (Throwable t) {
                // In case we didn't find the constant
                BCLog.logger.warn("Tried and failed to read the value(" + value + ") from " + clazz.getSimpleName(), t);
                return null;
            }
        } else if (nbt instanceof NBTTagByte) {
            byte value = ((NBTTagByte) nbt).func_150290_f();
            if (value < 0 || value >= clazz.getEnumConstants().length) {
                return null;
            } else {
                return clazz.getEnumConstants()[value];
            }
        } else if (nbt == null) {
            return null;
        } else {
            BCLog.logger.warn(new IllegalArgumentException("Tried to read an enum value when it was not a string! This is probably not good!"));
            return null;
        }
    }

    public static NBTBase writeObject(Object obj) {
        if (obj == null) throw new NullPointerException("obj");
        if (obj instanceof Byte) return new NBTTagByte((Byte) obj);
        if (obj instanceof Short) return new NBTTagShort((Short) obj);
        if (obj instanceof Integer) return new NBTTagInt((Integer) obj);
        if (obj instanceof Long) return new NBTTagLong((Long) obj);
        if (obj instanceof Float) return new NBTTagFloat((Float) obj);
        if (obj instanceof Double) return new NBTTagDouble((Double) obj);
        if (obj instanceof Boolean) {
            NBTTagCompound nbt = new NBTTagCompound();
            nbt.func_74757_a("boolean", (Boolean) obj);
            return nbt;
        }
        if (obj instanceof EnumFacing) {
            NBTTagCompound nbt = new NBTTagCompound();
            nbt.func_74778_a("type", "minecraft:enumfacing");
            nbt.func_74782_a("value", writeEnum((EnumFacing) obj));
            return nbt;
        }
        if (obj instanceof Enum<?>) {
            NBTTagCompound nbt = new NBTTagCompound();
            nbt.func_74778_a("type", "enum:" + obj.getClass().getName());
            nbt.func_74782_a("value", writeEnum((Enum) obj));
            return nbt;
        }
        throw new IllegalArgumentException("Cannot write class " + obj.getClass() + " directly to NBT!");
    }

    public static Object readObject(NBTBase nbt) {
        if (nbt == null) return null;
        if (nbt instanceof NBTPrimitive) {
            NBTPrimitive prim = (NBTPrimitive) nbt;
            if (prim instanceof NBTTagByte) return prim.func_150290_f();
            if (prim instanceof NBTTagShort) return prim.func_150289_e();
            if (prim instanceof NBTTagInt) return prim.func_150287_d();
            if (prim instanceof NBTTagLong) return prim.func_150291_c();
            if (prim instanceof NBTTagFloat) return prim.func_150288_h();
            if (prim instanceof NBTTagDouble) return prim.func_150286_g();
            else throw new Error("Seriously what? When was a new primitive NBT class added?");
        }
        if (nbt instanceof NBTTagString) return ((NBTTagString) nbt).func_150285_a_();
        if (nbt instanceof NBTTagCompound) {
            NBTTagCompound comp = (NBTTagCompound) nbt;
            if (comp.func_74764_b("boolean")) return comp.func_74767_n("boolean");
            if (comp.func_150297_b("type", Constants.NBT.TAG_STRING) && comp.func_74764_b("value")) {
                String type = ((NBTTagCompound) nbt).func_74779_i("type");
                NBTBase valueTag = comp.func_74781_a("value");
                if ("minecraft:enumfacing".equals(type)) return readEnum(valueTag, EnumFacing.class);
                if (type.startsWith("enum:")) try {
                    Class<?> clazz = Class.forName(type.replace("enum:", ""));
                    if (clazz.isEnum()) {
                        return readEnum(valueTag, (Class<Enum>) clazz);
                    } else {
                        BCLog.logger.warn("The type " + type + " refered to a class that was not an enum type!");
                    }
                } catch (ClassNotFoundException e) {
                    BCLog.logger.warn("Tried to load " + type + " but couldn't find the enum class!");
                }
            }
        }
        throw new Error("Tried to load an object from an unknown tag! " + nbt);
    }
}
