/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.energy;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.ICrafting;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.biome.BiomeGenBase;

import net.minecraftforge.fluids.*;

import buildcraft.BuildCraftEnergy;
import buildcraft.api.core.StackKey;
import buildcraft.api.enums.EnumEnergyStage;
import buildcraft.api.enums.EnumEngineType;
import buildcraft.api.fuels.BuildcraftFuelRegistry;
import buildcraft.api.fuels.ICoolant;
import buildcraft.api.fuels.IFuel;
import buildcraft.api.fuels.IFuelManager.IDirtyFuel;
import buildcraft.api.fuels.ISolidCoolant;
import buildcraft.api.transport.IItemPipe;
import buildcraft.core.GuiIds;
import buildcraft.core.lib.engines.TileEngineWithInventory;
import buildcraft.core.lib.fluids.Tank;
import buildcraft.core.lib.fluids.TankManager;
import buildcraft.core.lib.fluids.TankUtils;
import buildcraft.core.lib.inventory.InvUtils;
import buildcraft.core.statements.IBlockDefaultTriggers;

public class TileEngineIron extends TileEngineWithInventory implements IFluidHandler, IBlockDefaultTriggers {
    public static int MAX_LIQUID = FluidContainerRegistry.BUCKET_VOLUME * 10;
    public static float HEAT_PER_RF = 0.00023F;
    public static float COOLDOWN_RATE = 0.05F;
    public static int MAX_COOLANT_PER_TICK = 40;

    public Tank tankFuel = new Tank("tankFuel", MAX_LIQUID, this, this::isValidFuel);
    public Tank tankCoolant = new Tank("tankCoolant", MAX_LIQUID, this, this::isValidCoolant);
    public Tank tankResidue = new Tank("tankResidue", MAX_LIQUID, this, this::isResidue);

    private double burnTime = 0;
    private float coolingBuffer = 0.0f;
    private int tankFuelAmountCache = 0;
    private int tankCoolantAmountCache = 0;

    private TankManager<Tank> tankManager = new TankManager<>();
    private IFuel currentFuel;
    private int penaltyCooling = 0;
    private boolean lastPowered = false;
    private BiomeGenBase biomeCache;
    // private final ItemHandlerDelegate invWrapper = ItemHandlerDelegate.createFrom(super.getItemHandler(),
    // this::isItemValidForSlot);

    public TileEngineIron() {
        super(1);
        tankManager.add(tankFuel);
        tankManager.add(tankCoolant);
        tankManager.add(tankResidue);
    }

    @Override
    public boolean func_94041_b(/* IItemHandler handler, */int slot, ItemStack stack) {
        if (stack == null) {
            return false;
        } else if (BuildcraftFuelRegistry.coolant.getSolidCoolant(StackKey.stack(stack)) != null) {
            return true;
        } else {
            FluidStack fluidStack = FluidContainerRegistry.getFluidForFilledItem(stack);
            return fluidStack != null && canFill(null, fluidStack.getFluid());
        }
    }

    // @Override
    // public IItemHandler getItemHandler() {
    // return invWrapper;
    // }

    @Override
    public EnumEngineType getEngineType() {
        return EnumEngineType.IRON;
    }

    @Override
    public boolean onBlockActivated(EntityPlayer player, EnumFacing side) {
        if (super.onBlockActivated(player, side)) {
            return true;
        }

        ItemStack current = player.func_71045_bC();
        if (current != null) {
            if (current.func_77973_b() instanceof IItemPipe) {
                return false;
            }
            if (!field_145850_b.field_72995_K) {
                if (TankUtils.handleRightClick(this, side, player, true, true)) {
                    return true;
                }
            } else {
                if (FluidContainerRegistry.isContainer(current)) {
                    return true;
                }
            }
        }
        if (!field_145850_b.field_72995_K) {
            player.openGui(BuildCraftEnergy.instance, GuiIds.ENGINE_IRON, field_145850_b, field_174879_c.func_177958_n(), field_174879_c.func_177956_o(), field_174879_c.func_177952_p());
        }
        return true;
    }

    @Override
    public float getPistonSpeed() {
        if (!field_145850_b.field_72995_K) {
            return Math.max(0.07f * getHeatLevel(), 0.01f);
        }
        switch (getEnergyStage()) {
            case BLUE:
                return 0.04F;
            case GREEN:
                return 0.05F;
            case YELLOW:
                return 0.06F;
            case RED:
                return 0.07F;
            default:
                return 0;
        }
    }

    public boolean hasFuelBelowThreshold(float threshold) {
        FluidStack fuel = tankFuel.getFluid();

        if (fuel == null) {
            return true;
        }

        float percentage = (float) fuel.amount / (float) MAX_LIQUID;
        return percentage < threshold;
    }

    public boolean hasCoolantBelowThreshold(float threshold) {
        FluidStack coolant = tankCoolant.getFluid();

        if (coolant == null) {
            return true;
        }

        float percentage = (float) coolant.amount / (float) MAX_LIQUID;
        return percentage < threshold;
    }

    private float getBiomeTempScalar() {
        if (biomeCache == null) {
            biomeCache = field_145850_b.func_180494_b(field_174879_c);
        }
        float tempScalar = biomeCache.field_76750_F - 1.0F;
        tempScalar *= 0.5F;
        tempScalar += 1.0F;
        return tempScalar;
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        biomeCache = null;
    }

    @Override
    public boolean isBurning() {
        if (getEnergyStage() == EnumEnergyStage.OVERHEAT) {
            return false;
        }

        FluidStack fuel = tankFuel.getFluid();
        return fuel != null && fuel.amount > 0 && penaltyCooling == 0 && isRedstonePowered;
    }

    @Override
    public void overheat() {
        super.overheat();
        // Evaporate all remaining coolant as a form of penalty.
        tankCoolant.setFluid(null);
    }

    @Override
    public void burn() {
        FluidStack fuel = this.tankFuel.getFluid();
        if (currentFuel == null && fuel != null) {
            currentFuel = BuildcraftFuelRegistry.fuel.getFuel(fuel.getFluid());
        }

        if (currentFuel == null) {
            return;
        }

        if (penaltyCooling <= 0 && isRedstonePowered) {
            lastPowered = true;

            if (burnTime > 0 || (fuel != null && fuel.amount > 0)) {
                if (burnTime > 0) {
                    burnTime--;
                }
                if (burnTime <= 0) {
                    if (fuel != null) {
                        if (--fuel.amount <= 0) {
                            tankFuel.setFluid(null);
                        }
                        burnTime += currentFuel.getTotalBurningTime() / (double) FluidContainerRegistry.BUCKET_VOLUME;
                    } else {
                        currentFuel = null;
                        return;
                    }
                }

                currentOutput = currentFuel.getPowerPerCycle();

                addEnergy(currentFuel.getPowerPerCycle());
                heat += currentFuel.getPowerPerCycle() * HEAT_PER_RF * getBiomeTempScalar();
            }
        } else if (penaltyCooling <= 0) {
            if (lastPowered) {
                lastPowered = false;
                penaltyCooling = 10;
                // 10 tick of penalty on top of the cooling
            }
        }
    }

    @Override
    public void updateHeat() {
        if (energyStage == EnumEnergyStage.OVERHEAT && heat > MIN_HEAT) {
            heat -= COOLDOWN_RATE;
        }
    }

    @Override
    public void engineUpdate() {
        super.engineUpdate();

        ItemStack stack = func_70301_a(0);/* getItemHandler().getStackInSlot(0); */
        if (stack != null) {
            FluidStack liquid = FluidContainerRegistry.getFluidForFilledItem(stack);
            if (liquid == null && heat > MIN_HEAT * 2) {
                final ItemStack stackOne = stack.func_77946_l();
                stackOne.field_77994_a = 1;
                ISolidCoolant coolant = BuildcraftFuelRegistry.coolant.getSolidCoolant(StackKey.stack(stackOne));
                if (coolant != null) {
                    liquid = coolant.getFluidFromSolidCoolant(stackOne);
                }
            }

            if (liquid != null) {
                if (fill(null, liquid, false) == liquid.amount) {
                    fill(null, liquid, true);
                    this.func_70299_a(0, InvUtils.consumeItem(stack));
                    // itemHandler.setStackInSlot(0, InvUtils.consumeItem(stack));
                }
            }
        }

        if (heat > MIN_HEAT && (penaltyCooling > 0 || !isRedstonePowered)) {
            heat -= COOLDOWN_RATE;
            coolEngine(MIN_HEAT);
            getEnergyStage();
        } else if (heat > IDEAL_HEAT) {
            coolEngine(IDEAL_HEAT);
        }

        if (heat <= MIN_HEAT && penaltyCooling > 0) {
            penaltyCooling--;
        }

        if (heat <= MIN_HEAT) {
            heat = MIN_HEAT;
        }
    }

    private void coolEngine(float idealHeat) {
        float extraHeat = heat - idealHeat;

        if (coolingBuffer < extraHeat) {
            fillCoolingBuffer();
        }

        if (coolingBuffer >= extraHeat) {
            coolingBuffer -= extraHeat;
            heat -= extraHeat;
            return;
        }

        heat -= coolingBuffer;
        coolingBuffer = 0.0f;
    }

    private void fillCoolingBuffer() {
        FluidStack coolant = this.tankCoolant.getFluid();
        if (coolant == null) {
            return;
        }

        int coolantAmount = Math.min(MAX_COOLANT_PER_TICK, coolant.amount);
        ICoolant currentCoolant = BuildcraftFuelRegistry.coolant.getCoolant(coolant.getFluid());
        if (currentCoolant != null) {
            float cooling = currentCoolant.getDegreesCoolingPerMB(heat);
            cooling /= getBiomeTempScalar();
            coolingBuffer += coolantAmount * cooling;
            tankCoolant.drain(coolantAmount, true);
        }
    }

    @Override
    public void func_145839_a(NBTTagCompound data) {
        super.func_145839_a(data);
        tankManager.readFromNBT(data);

        burnTime = data.func_74769_h("burnTime");
        penaltyCooling = data.func_74762_e("penaltyCooling");

    }

    @Override
    public void func_145841_b(NBTTagCompound data) {
        super.func_145841_b(data);
        tankManager.writeToNBT(data);

        data.func_74780_a("burnTime", burnTime);
        data.func_74768_a("penaltyCooling", penaltyCooling);

    }

    @Override
    public void getGUINetworkData(int id, int value) {
        super.getGUINetworkData(id, value);
        switch (id) {
            // Fluid Fuel ID
            case 15:
                if (FluidRegistry.getFluid(value) != null) {
                    tankFuel.setFluid(new FluidStack(FluidRegistry.getFluid(value), tankFuelAmountCache));
                } else {
                    tankFuel.setFluid(null);
                }
                break;
            // Fluid Coolant ID
            case 16:
                if (FluidRegistry.getFluid(value) != null) {
                    tankCoolant.setFluid(new FluidStack(FluidRegistry.getFluid(value), tankCoolantAmountCache));
                } else {
                    tankCoolant.setFluid(null);
                }
                break;
            // Fluid Fuel amount
            case 17:
                tankFuelAmountCache = value;
                if (tankFuel.getFluid() != null) {
                    tankFuel.getFluid().amount = value;
                }
                break;
            // Fluid Coolant amount
            case 18:
                tankCoolantAmountCache = value;
                if (tankCoolant.getFluid() != null) {
                    tankCoolant.getFluid().amount = value;
                }
                break;
            // Fluid Fuel color
            case 19:
                tankFuel.colorRenderCache = value;
                break;
            // Fluid Coolant color
            case 20:
                tankCoolant.colorRenderCache = value;
                break;
        }
    }

    @Override
    public void sendGUINetworkData(Container containerEngine, ICrafting iCrafting) {
        super.sendGUINetworkData(containerEngine, iCrafting);
        iCrafting.func_71112_a(containerEngine, 15, tankFuel.getFluid() != null && tankFuel.getFluid().getFluid() != null ? tankFuel
                .getFluid().getFluid().getID() : 0);
        iCrafting.func_71112_a(containerEngine, 16, tankCoolant.getFluid() != null && tankCoolant.getFluid().getFluid() != null ? tankCoolant
                .getFluid().getFluid().getID() : 0);
        iCrafting.func_71112_a(containerEngine, 17, tankFuel.getFluid() != null ? tankFuel.getFluid().amount : 0);
        iCrafting.func_71112_a(containerEngine, 18, tankCoolant.getFluid() != null ? tankCoolant.getFluid().amount : 0);
        iCrafting.func_71112_a(containerEngine, 19, tankFuel.colorRenderCache);
        iCrafting.func_71112_a(containerEngine, 20, tankCoolant.colorRenderCache);
    }

    @Override
    public boolean isActive() {
        return penaltyCooling <= 0;
    }

    /* ITANKCONTAINER */
    @Override
    public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain) {
        return tankFuel.drain(maxDrain, doDrain);
    }

    @Override
    public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain) {
        if (resource == null) {
            return null;
        }
        if (tankCoolant.getFluidType() == resource.getFluid()) {
            return tankCoolant.drain(resource.amount, doDrain);
        }
        if (tankFuel.getFluidType() == resource.getFluid()) {
            return tankFuel.drain(resource.amount, doDrain);
        }
        return null;
    }

    @Override
    public boolean canDrain(EnumFacing from, Fluid fluid) {
        return from != orientation;
    }

    @Override
    public int fill(EnumFacing from, FluidStack resource, boolean doFill) {
        if (resource == null || resource.getFluid() == null) {
            return 0;
        }

        if (isValidCoolant(resource)) {
            return tankCoolant.fill(resource, doFill);
        } else if (isValidFuel(resource)) {
            int filled = tankFuel.fill(resource, doFill);
            if (filled > 0 && tankFuel.getFluid() != null && tankFuel.getFluid().getFluid() != null && (currentFuel == null || tankFuel.getFluid()
                    .getFluid() != currentFuel.getFluid())) {
                currentFuel = BuildcraftFuelRegistry.fuel.getFuel(tankFuel.getFluid().getFluid());
            }
            return filled;
        } else {
            return 0;
        }
    }

    private boolean isValidCoolant(FluidStack fluid) {
        return BuildcraftFuelRegistry.coolant.getCoolant(fluid.getFluid()) != null;
    }

    private boolean isValidFuel(FluidStack fluid) {
        return BuildcraftFuelRegistry.fuel.getFuel(fluid.getFluid()) != null;
    }

    private boolean isResidue(FluidStack fluid) {
        if (currentFuel == null) return false;
        if (currentFuel instanceof IDirtyFuel) {
            return fluid.isFluidEqual(((IDirtyFuel) currentFuel).getResidue());
        }
        return false;
    }

    @Override
    public boolean canFill(EnumFacing from, Fluid fluid) {
        return from != orientation && fluid != null && (BuildcraftFuelRegistry.coolant.getCoolant(fluid) != null || BuildcraftFuelRegistry.fuel
                .getFuel(fluid) != null);
    }

    @Override
    public FluidTankInfo[] getTankInfo(EnumFacing direction) {
        return tankManager.getTankInfo(direction);
    }

    public FluidStack getFuel() {
        return tankFuel.getFluid();
    }

    public FluidStack getCoolant() {
        return tankCoolant.getFluid();
    }

    @Override
    public int getMaxEnergy() {
        return 100000;
    }

    @Override
    public int getIdealOutput() {
        if (currentFuel == null) {
            return 0;
        } else {
            return currentFuel.getPowerPerCycle();
        }
    }

    @Override
    public boolean hasCustomName() {
        return false;
    }

    @Override
    public boolean blockInventoryTriggers(EnumFacing side) {
        return false;
    }

    @Override
    public boolean blockFluidHandlerTriggers(EnumFacing side) {
        return true;
    }
}
