/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.factory;

import java.util.*;

import net.minecraft.block.Block;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;

import net.minecraftforge.fluids.*;

import buildcraft.api.core.BuildCraftAPI;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.fluids.Tank;
import buildcraft.core.lib.fluids.TankUtils;
import buildcraft.core.lib.utils.BlockUtils;
import buildcraft.core.lib.utils.Utils;

import io.netty.buffer.ByteBuf;

public class TileFloodGate extends TileBuildCraft implements IFluidHandler {
    public static final int[] REBUILD_DELAY = new int[8];
    public static final int MAX_LIQUID = FluidContainerRegistry.BUCKET_VOLUME * 2;
    private final TreeMap<Integer, Deque<BlockPos>> pumpLayerQueues = new TreeMap<>();
    private final Set<BlockPos> visitedBlocks = new HashSet<>();
    private Deque<BlockPos> fluidsFound = new LinkedList<>();
    private final Tank tank = new Tank("tank", MAX_LIQUID, this);
    private int rebuildDelay;
    private int tick = Utils.RANDOM.nextInt();
    private boolean powered = false;
    private EnumSet<EnumFacing> blockedSides = EnumSet.noneOf(EnumFacing.class);

    static {
        REBUILD_DELAY[0] = 128;
        REBUILD_DELAY[1] = 256;
        REBUILD_DELAY[2] = 512;
        REBUILD_DELAY[3] = 1024;
        REBUILD_DELAY[4] = 2048;
        REBUILD_DELAY[5] = 4096;
        REBUILD_DELAY[6] = 8192;
        REBUILD_DELAY[7] = 16384;
    }

    public TileFloodGate() {}

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (powered) {
            return;
        }

        tick++;
        if (tick % 16 == 0) {
            FluidStack fluidtoFill = tank.drain(FluidContainerRegistry.BUCKET_VOLUME, false);
            if (fluidtoFill != null && fluidtoFill.amount == FluidContainerRegistry.BUCKET_VOLUME) {
                Fluid fluid = fluidtoFill.getFluid();
                if (fluid == null || !fluid.canBePlacedInWorld()) {
                    return;
                }

                if (fluid == FluidRegistry.WATER && field_145850_b.field_73011_w.func_177502_q() == -1) {
                    tank.drain(FluidContainerRegistry.BUCKET_VOLUME, true);
                    return;
                }

                if (tick % REBUILD_DELAY[rebuildDelay] == 0) {
                    rebuildDelay++;
                    if (rebuildDelay >= REBUILD_DELAY.length) {
                        rebuildDelay = REBUILD_DELAY.length - 1;
                    }
                    rebuildQueue();
                }
                BlockPos index = getNextIndexToFill(true);

                if (index != null && placeFluid(index, fluid)) {
                    tank.drain(FluidContainerRegistry.BUCKET_VOLUME, true);
                    rebuildDelay = 0;
                }
            }
        }
    }

    private boolean placeFluid(BlockPos pos, Fluid fluid) {
        Block block = BlockUtils.getBlockState(field_145850_b, pos).func_177230_c();

        if (canPlaceFluidAt(block, pos)) {
            boolean placed;
            Block b = TankUtils.getFluidBlock(fluid, true);

            if (b instanceof BlockFluidBase) {
                BlockFluidBase blockFluid = (BlockFluidBase) b;
                placed = field_145850_b.func_180501_a(pos, blockFluid.func_176223_P(), 3);
            } else {
                placed = field_145850_b.func_175656_a(pos, b.func_176223_P());
            }

            if (placed) {
                queueAdjacent(pos);
                expandQueue();
            }

            return placed;
        }

        return false;
    }

    private BlockPos getNextIndexToFill(boolean remove) {
        if (pumpLayerQueues.isEmpty()) {
            return null;
        }

        Deque<BlockPos> bottomLayer = pumpLayerQueues.firstEntry().getValue();

        if (bottomLayer != null) {
            if (bottomLayer.isEmpty()) {
                pumpLayerQueues.pollFirstEntry();
            }
            if (remove) {
                BlockPos index = bottomLayer.pollFirst();
                return index;
            }
            return bottomLayer.peekFirst();
        }

        return null;
    }

    private Deque<BlockPos> getLayerQueue(int layer) {
        Deque<BlockPos> pumpQueue = pumpLayerQueues.get(layer);
        if (pumpQueue == null) {
            pumpQueue = new LinkedList<>();
            pumpLayerQueues.put(layer, pumpQueue);
        }
        return pumpQueue;
    }

    /** Nasty expensive function, don't call if you don't have to. */
    public void rebuildQueue() {
        pumpLayerQueues.clear();
        visitedBlocks.clear();
        fluidsFound.clear();

        queueAdjacent(field_174879_c);

        expandQueue();
    }

    private void expandQueue() {
        if (tank.getFluidType() == null) {
            return;
        }
        while (!fluidsFound.isEmpty()) {
            Deque<BlockPos> fluidsToExpand = fluidsFound;
            fluidsFound = new LinkedList<>();

            for (BlockPos index : fluidsToExpand) {
                queueAdjacent(index);
            }
        }
    }

    public void queueAdjacent(BlockPos pos) {
        if (tank.getFluidType() == null) {
            return;
        }
        for (EnumFacing face : EnumFacing.field_82609_l) {
            if (face != EnumFacing.UP && !blockedSides.contains(face)) {
                queueForFilling(pos.func_177972_a(face));
            }
        }
    }

    public void queueForFilling(BlockPos pos) {
        if (pos.func_177956_o() < 0 || pos.func_177956_o() > 255) {
            return;
        }
        if (visitedBlocks.add(pos)) {
            if ((pos.func_177958_n() - this.field_174879_c.func_177958_n()) * (pos.func_177958_n() - this.field_174879_c.func_177958_n()) + (pos.func_177952_p() - this.field_174879_c.func_177952_p()) * (pos.func_177952_p() - this.field_174879_c.func_177952_p()) > 64
                * 64) {
                return;
            }

            Block block = BlockUtils.getBlockState(field_145850_b, pos).func_177230_c();
            if (BlockUtils.getFluid(block) == tank.getFluidType()) {
                fluidsFound.add(pos);
            }
            if (canPlaceFluidAt(block, pos)) {
                getLayerQueue(pos.func_177956_o()).addLast(pos);
            }
        }
    }

    private boolean canPlaceFluidAt(Block block, BlockPos pos) {
        return BuildCraftAPI.isSoftBlock(field_145850_b, pos) && !BlockUtils.isFullFluidBlock(field_145850_b, pos);
    }

    public void onNeighborBlockChange(Block block) {
        boolean p = field_145850_b.func_175687_A(field_174879_c) > 0;
        if (powered != p) {
            powered = p;
            if (!p) {
                rebuildQueue();
            }
        }
    }

    @Override
    public void func_145839_a(NBTTagCompound data) {
        super.func_145839_a(data);
        tank.readFromNBT(data);
        rebuildDelay = data.func_74771_c("rebuildDelay");
        powered = data.func_74767_n("powered");
        blockedSides.clear();
        for (int i = 0; i < 6; i++) {
            boolean blocked = data.func_74767_n("blocked[" + i + "]");
            if (blocked) blockedSides.add(EnumFacing.field_82609_l[i]);
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound data) {
        super.func_145841_b(data);
        tank.writeToNBT(data);
        data.func_74774_a("rebuildDelay", (byte) rebuildDelay);
        data.func_74757_a("powered", powered);
        for (int i = 0; i < 6; i++) {
            if (blockedSides.contains(EnumFacing.field_82609_l[i])) {
                data.func_74757_a("blocked[" + i + "]", true);
            }
        }
    }

    @Override
    public void readData(ByteBuf stream) {
        byte data = stream.readByte();
        blockedSides.clear();
        for (EnumFacing face : EnumFacing.field_82609_l) {
            int offset = face.ordinal();
            int isBlocked = (data >> offset) % 2;
            if (isBlocked != 0) blockedSides.add(face);
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        int offset = 0;
        byte data = 0;
        for (EnumFacing face : EnumFacing.field_82609_l) {
            int isBlocked = blockedSides.contains(face) ? 1 : 0;
            data &= isBlocked << offset;
            offset++;
        }
        stream.writeByte(data);
    }

    public void switchSide(EnumFacing side) {
        if (side != EnumFacing.UP) {
            if (blockedSides.contains(side)) blockedSides.remove(side);
            else blockedSides.add(side);

            rebuildQueue();
            sendNetworkUpdate();
            field_145850_b.func_175704_b(field_174879_c, field_174879_c);
        }
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        destroy();
    }

    @Override
    public void destroy() {
        pumpLayerQueues.clear();
    }

    // IFluidHandler implementation.
    @Override
    public int fill(EnumFacing from, FluidStack resource, boolean doFill) {
        return tank.fill(resource, doFill);
    }

    @Override
    public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain) {
        return null;
    }

    @Override
    public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain) {
        return null;
    }

    @Override
    public boolean canFill(EnumFacing from, Fluid fluid) {
        return true;
    }

    @Override
    public boolean canDrain(EnumFacing from, Fluid fluid) {
        return false;
    }

    @Override
    public FluidTankInfo[] getTankInfo(EnumFacing from) {
        return new FluidTankInfo[] { tank.getInfo() };
    }

    public boolean isSideBlocked(EnumFacing face) {
        return blockedSides.contains(face);
    }
}
