/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.factory;

import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.World;

import buildcraft.BuildCraftCore;
import buildcraft.BuildCraftFactory;
import buildcraft.api.blueprints.BuilderAPI;
import buildcraft.api.core.SafeTimeTracker;
import buildcraft.api.tiles.IControllable;
import buildcraft.api.tiles.IHasWork;
import buildcraft.api.transport.IPipeConnection;
import buildcraft.api.transport.IPipeTile;
import buildcraft.core.lib.RFBattery;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.utils.BlockMiner;
import buildcraft.core.lib.utils.BlockUtils;

import io.netty.buffer.ByteBuf;

public class TileMiningWell extends TileBuildCraft implements IHasWork, IPipeConnection, IControllable {
    private boolean isDigging = true;
    private BlockMiner miner;
    private int ledState;
    private int ticksSinceAction = 9001;

    private SafeTimeTracker updateTracker = new SafeTimeTracker(BuildCraftCore.updateFactor);

    public TileMiningWell() {
        super();
        this.setBattery(new RFBattery(2 * 64 * BuilderAPI.BREAK_ENERGY, BuilderAPI.BREAK_ENERGY * 4 + BuilderAPI.BUILD_ENERGY, 0));
    }

    /** Dig the next available piece of land if not done. As soon as it reaches bedrock, lava or goes below 0, it's
     * considered done. */
    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (updateTracker.markTimeIfDelay(field_145850_b)) {
            sendNetworkUpdate();
        }

        ticksSinceAction++;

        if (mode == Mode.Off) {
            if (miner != null) {
                miner.invalidate();
                miner = null;
            }
            isDigging = false;
            return;
        }

        if (getBattery().getEnergyStored() == 0) {
            return;
        }

        if (miner == null) {
            World world = field_145850_b;

            BlockPos search = field_174879_c.func_177977_b();

            while (world.func_180495_p(search).func_177230_c() == BuildCraftFactory.plainPipeBlock) {
                search = search.func_177977_b();
            }

            if (search.func_177956_o() < 1 || search.func_177956_o() < field_174879_c.func_177956_o() - BuildCraftFactory.miningDepth || !BlockUtils.canChangeBlock(world, search)) {
                isDigging = false;
                // Drain energy, because at 0 energy this will stop doing calculations.
                getBattery().useEnergy(0, 10, false);
                return;
            }

            if (world.func_175623_d(search) || world.func_180495_p(search).func_177230_c().func_176200_f(world, search)) {
                ticksSinceAction = 0;
                world.func_175656_a(search, BuildCraftFactory.plainPipeBlock.func_176223_P());
            } else {
                miner = new BlockMiner(world, this, search);
            }
        }

        if (miner != null) {
            isDigging = true;
            ticksSinceAction = 0;

            int usedEnergy = miner.acceptEnergy(getBattery().getEnergyStored());
            getBattery().useEnergy(usedEnergy, usedEnergy, false);

            if (miner.hasFailed()) {
                isDigging = false;
            }

            if (miner.hasFailed() || miner.hasMined()) {
                miner = null;
            }
        }
    }

    @Override
    public void func_145843_s() {
        super.func_145843_s();
        if (miner != null) {
            miner.invalidate();
        }
        if (field_145850_b != null && field_174879_c.func_177956_o() > 2) {
            BuildCraftFactory.miningWellBlock.removePipes(field_145850_b, field_174879_c);
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        super.writeData(stream);

        ledState = (ticksSinceAction < 2 ? 16 : 0) | (getBattery().getEnergyStored() * 15 / getBattery().getMaxEnergyStored());
        stream.writeByte(ledState);
    }

    @Override
    public void readData(ByteBuf stream) {
        super.readData(stream);

        int newLedState = stream.readUnsignedByte();
        if (newLedState != ledState) {
            ledState = newLedState;
            field_145850_b.func_175704_b(field_174879_c, field_174879_c);
        }
    }

    @Override
    public boolean hasWork() {
        return isDigging;
    }

    @Override
    public ConnectOverride overridePipeConnection(IPipeTile.PipeType type, EnumFacing with) {
        return type == IPipeTile.PipeType.ITEM ? ConnectOverride.CONNECT : ConnectOverride.DEFAULT;
    }

    @Override
    public boolean acceptsControlMode(Mode mode) {
        return mode == Mode.Off || mode == Mode.On;
    }
}
