/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.robotics;

import buildcraft.BuildCraftCore;
import buildcraft.api.boards.RedstoneBoardNBT;
import buildcraft.api.boards.RedstoneBoardRegistry;
import buildcraft.api.boards.RedstoneBoardRobot;
import buildcraft.api.boards.RedstoneBoardRobotNBT;
import buildcraft.api.core.BCLog;
import buildcraft.api.core.IZone;
import buildcraft.api.events.RobotEvent;
import buildcraft.api.robots.*;
import buildcraft.api.statements.StatementSlot;
import buildcraft.api.tiles.IDebuggable;
import buildcraft.core.ItemWrench;
import buildcraft.core.LaserData;
import buildcraft.core.lib.RFBattery;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.ICommandReceiver;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;
import buildcraft.core.proxy.CoreProxy;
import buildcraft.robotics.ai.AIRobotMain;
import buildcraft.robotics.ai.AIRobotShutdown;
import buildcraft.robotics.ai.AIRobotSleep;
import buildcraft.robotics.statements.ActionRobotWorkInArea;
import buildcraft.robotics.statements.ActionRobotWorkInArea.AreaType;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.properties.Property;
import io.netty.buffer.ByteBuf;
import net.minecraft.client.Minecraft;
import net.minecraft.enchantment.EnchantmentHelper;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.SharedMonsterAttributes;
import net.minecraft.entity.ai.attributes.AttributeModifier;
import net.minecraft.entity.item.EntityFallingBlock;
import net.minecraft.entity.monster.IMob;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemArmor;
import net.minecraft.item.ItemSkull;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.nbt.NBTUtil;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.*;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraftforge.common.ForgeHooks;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.util.Constants;
import net.minecraftforge.common.util.Constants.NBT;
import net.minecraftforge.event.entity.player.AttackEntityEvent;
import net.minecraftforge.fluids.*;
import net.minecraftforge.fml.common.registry.IEntityAdditionalSpawnData;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import java.util.*;

public class EntityRobot extends EntityRobotBase implements IEntityAdditionalSpawnData, IInventory, IFluidHandler, ICommandReceiver, IDebuggable {

    public static final ResourceLocation ROBOT_BASE = new ResourceLocation("buildcraftrobotics", "entities/robot_base");

    private static final int DATA_LASER_TAIL_X = 12;
    private static final int DATA_LASER_TAIL_Y = 13;
    private static final int DATA_LASER_TAIL_Z = 14;
    // 15 is used by entity living base to see if the AI is active or not
    private static final int DATA_LASER_VISIBLE = 16;
    private static final int DATA_BOARD_ID = 17;
    private static final int DATA_ITEM_AIM_YAW = 18;
    private static final int DATA_ITEM_AIM_PITCH = 19;
    private static final int DATA_ENERGY_SPEND_PER_CYCLE = 20;
    private static final int DATA_ACTIVE_CLIENT = 21;
    private static final int DATA_BATTERY_ENERGY = 22;

    public static final int MAX_WEARABLES = 8;

    private static Set<Integer> blacklistedItemsForUpdate = Sets.newHashSet();

    public LaserData laser = new LaserData();
    public DockingStation linkedDockingStation;
    public BlockPos linkedDockingStationIndex;
    public EnumFacing linkedDockingStationSide;

    public BlockPos currentDockingStationIndex;
    public EnumFacing currentDockingStationSide;

    public boolean isDocked = false;

    public RedstoneBoardRobot board;
    public AIRobotMain mainAI;

    public ItemStack itemInUse;
    public float itemAimYaw = 0;
    public float renderItemAimYaw = 0;
    public float itemAimPitch = 0;
    public boolean itemActive = false;
    public float itemActiveStage = 0;
    public long lastUpdateTime = 0;

    private DockingStation currentDockingStation;
    private List<ItemStack> wearables = new ArrayList<>();

    private boolean needsUpdate = false;
    private ItemStack[] inv = new ItemStack[4];
    private FluidStack tank;
    private int maxFluid = FluidContainerRegistry.BUCKET_VOLUME * 4;
    private ResourceLocation texture;

    private WeakHashMap<Entity, Long> unreachableEntities = new WeakHashMap<>();

    private NBTTagList stackRequestNBT;

    private RFBattery battery = new RFBattery(MAX_ENERGY, MAX_ENERGY, 100);

    private boolean firstUpdateDone = false;

    private boolean isActiveClient = false;

    private long robotId = EntityRobotBase.NULL_ROBOT_ID;

    private int energySpendPerCycle = 0;
    private int ticksCharging = 0;
    private float energyFX = 0;
    private Vec3 steamDirection = new Vec3(0, -1, 0);

    public EntityRobot(World world, RedstoneBoardRobotNBT boardNBT) {
        this(world);

        board = boardNBT.create(this);
        dataWatcher.updateObject(DATA_BOARD_ID, board.getNBTHandler().getID());

        if (!world.field_72995_K) {
            mainAI = new AIRobotMain(this);
            mainAI.start();
        }
    }

    public EntityRobot(World world) {
        super(world);

        motionX = 0;
        motionY = 0;
        motionZ = 0;

        ignoreFrustumCheck = true;
        laser.isVisible = false;
        entityCollisionReduction = 1F;

        width = 0.25F;
        height = 0.25F;
    }

    @Override
    protected void entityInit() {
        super.entityInit();

        setNullBoundingBox();

        preventEntitySpawning = false;
        noClip = true;
        isImmuneToFire = true;
        this.enablePersistence();

        dataWatcher.addObject(DATA_LASER_TAIL_X, Float.valueOf(0));
        dataWatcher.addObject(DATA_LASER_TAIL_Y, Float.valueOf(0));
        dataWatcher.addObject(DATA_LASER_TAIL_Z, Float.valueOf(0));
        dataWatcher.addObject(DATA_LASER_VISIBLE, Byte.valueOf((byte) 0));
        dataWatcher.addObject(DATA_BOARD_ID, "");
        dataWatcher.addObject(DATA_ITEM_AIM_YAW, Float.valueOf(0));
        dataWatcher.addObject(DATA_ITEM_AIM_PITCH, Float.valueOf(0));
        dataWatcher.addObject(DATA_ENERGY_SPEND_PER_CYCLE, Integer.valueOf(0));
        dataWatcher.addObject(DATA_ACTIVE_CLIENT, Byte.valueOf((byte) 0));
        dataWatcher.addObject(DATA_BATTERY_ENERGY, Integer.valueOf(0));
    }

    protected void updateDataClient() {
        float x = dataWatcher.getWatchableObjectFloat(DATA_LASER_TAIL_X);
        float y = dataWatcher.getWatchableObjectFloat(DATA_LASER_TAIL_Y);
        float z = dataWatcher.getWatchableObjectFloat(DATA_LASER_TAIL_Z);
        laser.tail = new Vec3(x, y, z);
        laser.isVisible = dataWatcher.getWatchableObjectByte(DATA_LASER_VISIBLE) == 1;

        RedstoneBoardNBT<?> boardNBT = RedstoneBoardRegistry.instance.getRedstoneBoard(dataWatcher.getWatchableObjectString(DATA_BOARD_ID));

        if (boardNBT != null) {
            ResourceLocation texOld = ((RedstoneBoardRobotNBT) boardNBT).getRobotTexture();
            texture = new ResourceLocation(texOld.func_110624_b(), "textures/" + texOld.func_110623_a() + ".png");
        }

        itemAimYaw = dataWatcher.getWatchableObjectFloat(DATA_ITEM_AIM_YAW);
        itemAimPitch = dataWatcher.getWatchableObjectFloat(DATA_ITEM_AIM_PITCH);
        energySpendPerCycle = dataWatcher.getWatchableObjectInt(DATA_ENERGY_SPEND_PER_CYCLE);
        isActiveClient = dataWatcher.getWatchableObjectByte(DATA_ACTIVE_CLIENT) == 1;
        battery.setEnergy(dataWatcher.getWatchableObjectInt(DATA_BATTERY_ENERGY));
    }

    protected void updateDataServer() {
        dataWatcher.updateObject(DATA_LASER_TAIL_X, Float.valueOf((float) laser.tail.field_72450_a));
        dataWatcher.updateObject(DATA_LASER_TAIL_Y, Float.valueOf((float) laser.tail.field_72448_b));
        dataWatcher.updateObject(DATA_LASER_TAIL_Z, Float.valueOf((float) laser.tail.field_72449_c));
        dataWatcher.updateObject(DATA_LASER_VISIBLE, Byte.valueOf((byte) (laser.isVisible ? 1 : 0)));
        dataWatcher.updateObject(DATA_ITEM_AIM_YAW, Float.valueOf(itemAimYaw));
        dataWatcher.updateObject(DATA_ITEM_AIM_PITCH, Float.valueOf(itemAimPitch));
    }

    public boolean isActive() {
        if (worldObj.isRemote) {
            return isActiveClient;
        } else {
            return mainAI.getActiveAI() instanceof AIRobotSleep || mainAI.getActiveAI() instanceof AIRobotShutdown;
        }
    }

    protected void init() {
        if (worldObj.isRemote) {
            BuildCraftCore.instance.sendToServer(new PacketCommand(this, "requestInitialization", null));
        }
    }

    public void setLaserDestination(float x, float y, float z) {
        if (x != laser.tail.field_72450_a || y != laser.tail.field_72448_b || z != laser.tail.field_72449_c) {
            laser.tail = new Vec3(x, y, z);
            needsUpdate = true;
        }
    }

    public void showLaser() {
        if (!laser.isVisible) {
            laser.isVisible = true;
            needsUpdate = true;
        }
    }

    public void hideLaser() {
        if (laser.isVisible) {
            laser.isVisible = false;
            needsUpdate = true;
        }
    }

    protected void firstUpdate() {
        if (stackRequestNBT != null) {

        }

        if (!worldObj.isRemote) {
            getRegistry().registerRobot(this);
        }
    }

    @Override
    public String func_70005_c_() {
        return StatCollector.func_74838_a("item.robot.name");
    }

    @Override
    public void onEntityUpdate() {
        this.worldObj.theProfiler.startSection("bcEntityRobot");
        if (!firstUpdateDone) {
            firstUpdate();
            firstUpdateDone = true;
        }

        if (ticksCharging > 0) {
            ticksCharging--;
        }

        if (!worldObj.isRemote) {
            // The client-side sleep indicator should also display if the robot is charging.
            // To not break gates and other things checking for sleep, this is done here.
            dataWatcher.updateObject(DATA_ACTIVE_CLIENT, Byte.valueOf((byte) ((isActive() && ticksCharging == 0) ? 1 : 0)));
            dataWatcher.updateObject(DATA_BATTERY_ENERGY, getEnergy());

            if (needsUpdate) {
                updateDataServer();
                needsUpdate = false;
            }
        }

        if (worldObj.isRemote) {
            updateDataClient();
            updateRotationYaw(60.0f);
            updateEnergyFX();
        }

        if (currentDockingStation != null) {
            motionX = 0;
            motionY = 0;
            motionZ = 0;

            Vec3 pos = Utils.convertMiddle(currentDockingStation.getPos()).add(Utils.convert(currentDockingStation.side(), 0.5));
            posX = pos.field_72450_a;
            posY = pos.field_72448_b;
            posZ = pos.field_72449_c;
        }

        if (!worldObj.isRemote) {
            if (linkedDockingStation == null) {
                if (linkedDockingStationIndex != null) {
                    linkedDockingStation = getRegistry().getStation(linkedDockingStationIndex, linkedDockingStationSide);
                }

                if (linkedDockingStation == null) {
                    shutdown("no docking station");
                } else {
                    if (linkedDockingStation.robotTaking() != this) {
                        if (linkedDockingStation.robotIdTaking() == robotId) {
                            BCLog.logger.warn("A robot entity was not properly unloaded");
                            linkedDockingStation.invalidateRobotTakingEntity();
                        }
                        if (linkedDockingStation.robotTaking() != this) {
                            shutdown("wrong docking station");
                        }
                    }
                }
            }

            if (currentDockingStationIndex != null && currentDockingStation == null) {
                currentDockingStation = getRegistry().getStation(currentDockingStationIndex, currentDockingStationSide);
            }

            if (posY < -128) {
                isDead = true;

                BCLog.logger.info("Destroying robot " + this.toString() + " - Fallen into Void");
                getRegistry().killRobot(this);
            }

            if (linkedDockingStation == null || linkedDockingStation.isInitialized()) {
                this.worldObj.theProfiler.startSection("bcRobotAI");
                mainAI.cycle();
                this.worldObj.theProfiler.endSection();

                if (energySpendPerCycle != mainAI.getActiveAI().getEnergyCost()) {
                    energySpendPerCycle = mainAI.getActiveAI().getEnergyCost();
                    dataWatcher.updateObject(DATA_ENERGY_SPEND_PER_CYCLE, energySpendPerCycle);
                }
            }
        }

        // tick all carried itemstacks
        for (int i = 0; i < inv.length; i++) {
            updateItem(inv[i], i, false);
        }

        // tick the item the robot is currently holding
        updateItem(itemInUse, 0, true);

        // do not tick wearables or equipment from EntityLiving

        super.onEntityUpdate();
        this.worldObj.theProfiler.endSection();
    }

    // @Override
    // protected void updateEntityActionState() {}

    @Override
    public boolean handleWaterMovement() {
        return false;
    }

    @SideOnly(Side.CLIENT)
    private void updateEnergyFX() {
        energyFX += energySpendPerCycle;

        if (energyFX >= (100 << (2 * Minecraft.func_71410_x().field_71474_y.field_74362_aa))) {
            energyFX = 0;
            spawnEnergyFX();
        }
    }

    @SideOnly(Side.CLIENT)
    private void spawnEnergyFX() {
        Minecraft.func_71410_x().field_71452_i.func_78873_a(new EntityRobotEnergyParticle(worldObj, posX + steamDirection.field_72450_a * 0.25, posY
            + steamDirection.field_72448_b * 0.25, posZ + steamDirection.field_72449_c * 0.25, steamDirection.field_72450_a * 0.05, steamDirection.field_72448_b * 0.05,
                steamDirection.field_72449_c * 0.05, energySpendPerCycle * 0.075F < 1 ? 1 : energySpendPerCycle * 0.075F));
    }

    @Override
    public AxisAlignedBB getEntityBoundingBox() {
        return new AxisAlignedBB(posX - 0.25F, posY - 0.25F, posZ - 0.25F, posX + 0.25F, posY + 0.25F, posZ + 0.25F);
    }

    @Override
    public AxisAlignedBB getCollisionBoundingBox() {
        return getEntityBoundingBox();
    }

    public void setNullBoundingBox() {
        width = 0F;
        height = 0F;

        setEntityBoundingBox(new AxisAlignedBB(posX, posY, posZ, posX, posY, posZ));
    }

    private void shutdown(String reason) {
        if (!(mainAI.getDelegateAI() instanceof AIRobotShutdown)) {
            BCLog.logger.info("Shutting down robot " + this.toString() + " - " + reason);
            mainAI.startDelegateAI(new AIRobotShutdown(this));
        }
    }

    @Override
    public void writeSpawnData(ByteBuf data) {
        data.writeByte(wearables.size());
        for (ItemStack s : wearables) {
            NetworkUtils.writeStack(data, s);
        }
    }

    @Override
    public void readSpawnData(ByteBuf data) {
        int amount = data.readUnsignedByte();
        while (amount > 0) {
            wearables.add(NetworkUtils.readStack(data));
            amount--;
        }
        init();
    }

    @Override
    public ItemStack getHeldItem() {
        return itemInUse;
    }

    @Override
    public void setCurrentItemOrArmor(int i, ItemStack itemstack) {}

    @Override
    public void moveEntityWithHeading(float par1, float par2) {
        this.setPosition(posX + motionX, posY + motionY, posZ + motionZ);
    }

    @Override
    public boolean isOnLadder() {
        return false;
    }

    public ResourceLocation getTexture() {
        return texture;
    }

    @Override
    public void writeEntityToNBT(NBTTagCompound nbt) {
        super.writeEntityToNBT(nbt);

        if (linkedDockingStationIndex != null) {
            NBTTagCompound linkedStationNBT = new NBTTagCompound();
            linkedStationNBT.func_74782_a("index", NBTUtils.writeBlockPos(linkedDockingStationIndex));
            linkedStationNBT.func_74774_a("side", (byte) linkedDockingStationSide.ordinal());
            nbt.func_74782_a("linkedStation", linkedStationNBT);
        }

        if (currentDockingStationIndex != null) {
            NBTTagCompound currentStationNBT = new NBTTagCompound();
            currentStationNBT.func_74782_a("index", NBTUtils.writeBlockPos(currentDockingStationIndex));
            currentStationNBT.func_74774_a("side", (byte) currentDockingStationSide.ordinal());
            nbt.func_74782_a("currentStation", currentStationNBT);
        }

        NBTTagCompound nbtLaser = new NBTTagCompound();
        laser.writeToNBT(nbtLaser);
        nbt.func_74782_a("laser", nbtLaser);

        NBTTagCompound batteryNBT = new NBTTagCompound();
        battery.writeToNBT(batteryNBT);
        nbt.func_74782_a("battery", batteryNBT);

        if (itemInUse != null) {
            NBTTagCompound itemNBT = new NBTTagCompound();
            itemInUse.func_77955_b(itemNBT);
            nbt.func_74782_a("itemInUse", itemNBT);
            nbt.func_74757_a("itemActive", itemActive);
        }

        for (int i = 0; i < inv.length; ++i) {
            NBTTagCompound stackNbt = new NBTTagCompound();

            if (inv[i] != null) {
                nbt.func_74782_a("inv[" + i + "]", inv[i].func_77955_b(stackNbt));
            }
        }

        if (wearables.size() > 0) {
            NBTTagList wearableList = new NBTTagList();

            for (ItemStack wearable : wearables) {
                NBTTagCompound item = new NBTTagCompound();
                wearable.func_77955_b(item);
                wearableList.func_74742_a(item);
            }

            nbt.func_74782_a("wearables", wearableList);
        }

        NBTTagCompound ai = new NBTTagCompound();
        mainAI.writeToNBT(ai);
        nbt.func_74782_a("mainAI", ai);

        if (mainAI.getDelegateAI() != board) {
            NBTTagCompound boardNBT = new NBTTagCompound();
            board.writeToNBT(boardNBT);
            nbt.func_74782_a("board", boardNBT);
        }

        nbt.func_74772_a("robotId", robotId);

        if (tank != null) {
            NBTTagCompound tankNBT = new NBTTagCompound();

            tank.writeToNBT(tankNBT);

            nbt.func_74782_a("tank", tankNBT);
        }
    }

    @Override
    public void readEntityFromNBT(NBTTagCompound nbt) {
        super.readEntityFromNBT(nbt);

        if (nbt.func_74764_b("linkedStation")) {
            NBTTagCompound linkedStationNBT = nbt.func_74775_l("linkedStation");
            linkedDockingStationIndex = NBTUtils.readBlockPos(linkedStationNBT.func_74781_a("index"));
            linkedDockingStationSide = EnumFacing.values()[linkedStationNBT.func_74771_c("side")];
        }

        if (nbt.func_74764_b("currentStation")) {
            NBTTagCompound currentStationNBT = nbt.func_74775_l("currentStation");
            currentDockingStationIndex = NBTUtils.readBlockPos(currentStationNBT.func_74781_a("index"));
            currentDockingStationSide = EnumFacing.values()[currentStationNBT.func_74771_c("side")];

        }

        laser.readFromNBT(nbt.func_74775_l("laser"));

        battery.readFromNBT(nbt.func_74775_l("battery"));

        wearables.clear();
        if (nbt.func_74764_b("wearables")) {
            NBTTagList list = nbt.func_150295_c("wearables", 10);
            for (int i = 0; i < list.func_74745_c(); i++) {
                ItemStack stack = ItemStack.func_77949_a(list.func_150305_b(i));
                if (stack != null) {
                    wearables.add(stack);
                }
            }
        }

        if (nbt.func_74764_b("itemInUse")) {
            itemInUse = ItemStack.func_77949_a(nbt.func_74775_l("itemInUse"));
            itemActive = nbt.func_74767_n("itemActive");
        }

        for (int i = 0; i < inv.length; ++i) {
            inv[i] = ItemStack.func_77949_a(nbt.func_74775_l("inv[" + i + "]"));
        }

        NBTTagCompound ai = nbt.func_74775_l("mainAI");
        mainAI = (AIRobotMain) AIRobot.loadAI(ai, this);

        if (nbt.func_74764_b("board")) {
            board = (RedstoneBoardRobot) AIRobot.loadAI(nbt.func_74775_l("board"), this);
        } else {
            board = (RedstoneBoardRobot) mainAI.getDelegateAI();
        }

        if (board == null) {
            board = RedstoneBoardRegistry.instance.getEmptyRobotBoard().create(this);
        }

        dataWatcher.updateObject(DATA_BOARD_ID, board.getNBTHandler().getID());

        stackRequestNBT = nbt.func_150295_c("stackRequests", Constants.NBT.TAG_COMPOUND);

        if (nbt.func_74764_b("robotId")) {
            robotId = nbt.func_74763_f("robotId");
        }

        if (nbt.func_74764_b("tank")) {
            tank = FluidStack.loadFluidStackFromNBT(nbt.func_74775_l("tank"));
        } else {
            tank = null;
        }

        // Restore robot persistence on pre-6.1.9 robotics
        this.enablePersistence();
        // this.func_110163_bv(); TODO (PASS 1): Check to make sure this is really the correct method!
    }

    @Override
    public void dock(DockingStation station) {
        currentDockingStation = station;

        setSteamDirection(Utils.convert(currentDockingStation.side));

        currentDockingStationIndex = currentDockingStation.index();
        currentDockingStationSide = currentDockingStation.side();
    }

    @Override
    public void undock() {
        if (currentDockingStation != null) {
            currentDockingStation.release(this);
            currentDockingStation = null;

            setSteamDirection(new Vec3(0, -1, 0));

            currentDockingStationIndex = null;
            currentDockingStationSide = null;
        }
    }

    @Override
    public DockingStation getDockingStation() {
        return currentDockingStation;
    }

    @Override
    public void setMainStation(DockingStation station) {
        if (linkedDockingStation != null && linkedDockingStation != station) {
            linkedDockingStation.unsafeRelease(this);
        }

        linkedDockingStation = station;
        if (station != null) {
            linkedDockingStationIndex = linkedDockingStation.index();
            linkedDockingStationSide = linkedDockingStation.side();
        } else {
            linkedDockingStationIndex = null;
            linkedDockingStationSide = null;
        }
    }

    @Override
    public ItemStack getEquipmentInSlot(int var1) {
        return null;
    }

    @Override
    public int func_70302_i_() {
        return inv.length;
    }

    @Override
    public ItemStack func_70301_a(int var1) {
        return inv[var1];
    }

    @Override
    public ItemStack func_70298_a(int var1, int var2) {
        ItemStack result = inv[var1].func_77979_a(var2);

        if (inv[var1].field_77994_a == 0) {
            inv[var1] = null;
        }

        updateClientSlot(var1);

        return result;
    }

    @Override
    public ItemStack func_70304_b(int var1) {
        ItemStack stack = inv[var1];
        inv[var1] = null;
        return stack;
    }

    @Override
    public void func_70299_a(int var1, ItemStack var2) {
        inv[var1] = var2;

        updateClientSlot(var1);
    }

    @Override
    public IChatComponent func_145748_c_() {
        return null;
    }

    @Override
    public boolean func_145818_k_() {
        return false;
    }

    @Override
    public int func_70297_j_() {
        return 64;
    }

    @Override
    public void func_70296_d() {}

    public void updateClientSlot(final int slot) {
        BuildCraftCore.instance.sendToEntity(new PacketCommand(this, "clientSetInventory", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                data.writeShort(slot);
                NetworkUtils.writeStack(data, inv[slot]);
            }
        }), this);
    }

    @Override
    public boolean func_70300_a(EntityPlayer var1) {
        return false;
    }

    @Override
    public void func_174889_b(EntityPlayer player) {}

    @Override
    public void func_174886_c(EntityPlayer player) {}

    @Override
    public boolean func_94041_b(int var1, ItemStack var2) {
        return inv[var1] == null || (inv[var1].func_77969_a(var2) && inv[var1].func_77985_e() && inv[var1].field_77994_a + var2.field_77994_a <= inv[var1]
                .func_77973_b().getItemStackLimit(inv[var1]));
    }

    @Override
    public boolean isMoving() {
        return motionX != 0 || motionY != 0 || motionZ != 0;
    }

    @Override
    public void setItemInUse(ItemStack stack) {
        itemInUse = stack;
        BuildCraftCore.instance.sendToEntity(new PacketCommand(this, "clientSetItemInUse", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                NetworkUtils.writeStack(data, itemInUse);
            }
        }), this);
    }

    private void setSteamDirection(final Vec3 direction) {
        if (!worldObj.isRemote) {
            BuildCraftCore.instance.sendToEntity(new PacketCommand(this, "setSteamDirection", new CommandWriter() {
                @Override
                public void write(ByteBuf data) {
                    data.writeDouble(direction.field_72450_a);
                    data.writeDouble(direction.field_72448_b);
                    data.writeDouble(direction.field_72449_c);
                }
            }), this);
        } else {
            steamDirection = direction.func_72432_b();
        }
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        if (side.isClient()) {
            if ("clientSetItemInUse".equals(command)) {
                itemInUse = NetworkUtils.readStack(stream);
            } else if ("clientSetInventory".equals(command)) {
                int slot = stream.readUnsignedShort();
                inv[slot] = NetworkUtils.readStack(stream);
            } else if ("initialize".equals(command)) {
                itemInUse = NetworkUtils.readStack(stream);
                itemActive = stream.readBoolean();
            } else if ("setItemActive".equals(command)) {
                itemActive = stream.readBoolean();
                itemActiveStage = 0;
                lastUpdateTime = new Date().getTime();

                if (!itemActive) {
                    setSteamDirection(new Vec3(0, -1, 0));
                }
            } else if ("setSteamDirection".equals(command)) {
                setSteamDirection(new Vec3(stream.readDouble(), stream.readDouble(), stream.readDouble()));
            } else if ("syncWearables".equals(command)) {
                wearables.clear();

                int amount = stream.readUnsignedByte();
                while (amount > 0) {
                    wearables.add(NetworkUtils.readStack(stream));
                    amount--;
                }
            }
        } else if (side.isServer()) {
            EntityPlayer p = (EntityPlayer) sender;
            if ("requestInitialization".equals(command)) {
                BuildCraftCore.instance.sendToPlayer(p, new PacketCommand(this, "initialize", new CommandWriter() {
                    @Override
                    public void write(ByteBuf data) {
                        NetworkUtils.writeStack(data, itemInUse);
                        data.writeBoolean(itemActive);
                    }
                }));

                for (int i = 0; i < inv.length; ++i) {
                    final int j = i;
                    BuildCraftCore.instance.sendToPlayer(p, new PacketCommand(this, "clientSetInventory", new CommandWriter() {
                        @Override
                        public void write(ByteBuf data) {
                            data.writeShort(j);
                            NetworkUtils.writeStack(data, inv[j]);
                        }
                    }));
                }

                if (currentDockingStation != null) {
                    setSteamDirection(Utils.convert(currentDockingStation.side()));
                } else {
                    setSteamDirection(new Vec3(0, -1, 0));
                }
            }
        }
    }

    @Override
    public void setHealth(float par1) {
        // deactivate health management
    }

    @Override
    public boolean attackEntityFrom(DamageSource source, float f) {
        // Ignore hits from mobs or when docked.
        Entity src = source.func_76364_f();
        if (src != null && !(src instanceof EntityFallingBlock) && !(src instanceof IMob) && currentDockingStation == null) {
            if (ForgeHooks.onLivingAttack(this, source, f)) {
                return false;
            }

            if (!worldObj.isRemote) {
                hurtTime = maxHurtTime = 10;

                int mul = 2600;
                for (ItemStack s : wearables) {
                    if (s.func_77973_b() instanceof ItemArmor) {
                        mul = mul * 2 / (2 + ((ItemArmor) s.func_77973_b()).field_77879_b);
                    } else {
                        mul *= 0.7;
                    }
                }

                int energy = Math.round(f * mul);
                if (battery.getEnergyStored() - energy > 0) {
                    battery.setEnergy(battery.getEnergyStored() - energy);
                    return true;
                } else {
                    onRobotHit(true);
                }
            }
            return true;
        }
        return false;
    }

    @Override
    public float getAimYaw() {
        return itemAimYaw;
    }

    @Override
    public float getAimPitch() {
        return itemAimPitch;
    }

    @Override
    public void aimItemAt(float yaw, float pitch) {
        itemAimYaw = yaw;
        itemAimPitch = pitch;

        updateDataServer();
    }

    @Override
    public void aimItemAt(BlockPos pos) {
        Vec3 delta = Utils.convert(pos).func_178788_d(Utils.getVec(this));
        if (delta.field_72450_a != 0 || delta.field_72449_c != 0) {
            itemAimYaw = (float) (Math.atan2(delta.field_72450_a, delta.field_72449_c) * 180f / Math.PI) + 180f;
        }

        double d3 = MathHelper.func_76133_a(delta.field_72450_a * delta.field_72450_a + delta.field_72449_c * delta.field_72449_c);
        itemAimPitch = (float) (-(Math.atan2(delta.field_72448_b, d3) * 180.0D / Math.PI));

        setSteamDirection(delta);

        updateDataServer();
    }

    private void updateRotationYaw(float maxStep) {
        float step = MathHelper.func_76142_g(itemAimYaw - rotationYaw);

        if (step > maxStep) {
            step = maxStep;
        }

        if (step < -maxStep) {
            step = -maxStep;
        }

        rotationYaw = rotationYaw + step;
    }

    @Override
    protected float updateDistance(float targetYaw, float dist) {
        if (worldObj.isRemote) {
            float f2 = MathHelper.func_76142_g(this.rotationYaw - this.renderYawOffset);
            this.renderYawOffset += f2 * 0.5F;
            float f3 = MathHelper.func_76142_g(this.rotationYaw - this.renderYawOffset);
            boolean flag = f3 < -90.0F || f3 >= 90.0F;

            this.renderYawOffset = this.rotationYaw - f3;

            if (f3 * f3 > 2500.0F) {
                this.renderYawOffset += f3 * 0.2F;
            }

            float newDist = dist;
            if (flag) {
                newDist *= -1.0F;
            }

            return newDist;
        }
        return 0;
    }

    @Override
    public void setItemActive(final boolean isActive) {
        if (isActive != itemActive) {
            itemActive = isActive;
            BuildCraftCore.instance.sendToEntity(new PacketCommand(this, "setItemActive", new CommandWriter() {
                @Override
                public void write(ByteBuf data) {
                    data.writeBoolean(isActive);
                }
            }), this);
        }
    }

    @Override
    public RedstoneBoardRobot getBoard() {
        return board;
    }

    @Override
    public DockingStation getLinkedStation() {
        return linkedDockingStation;
    }

    @SideOnly(Side.CLIENT)
    @Override
    public boolean isInRangeToRenderDist(double par1) {
        return true;
    }

    @Override
    public int getEnergy() {
        return battery.getEnergyStored();
    }

    @Override
    public RFBattery getBattery() {
        return battery;
    }

    @Override
    protected boolean canDespawn() {
        return false;
    }

    public AIRobot getOverridingAI() {
        return mainAI.getOverridingAI();
    }

    public void overrideAI(AIRobot ai) {
        mainAI.setOverridingAI(ai);
    }

    public void attackTargetEntityWithCurrentItem(Entity par1Entity) {
        BlockPos entPos = Utils.convertFloor(Utils.getVec(par1Entity));
        if (MinecraftForge.EVENT_BUS.post(new AttackEntityEvent(CoreProxy.proxy.getBuildCraftPlayer((WorldServer) worldObj, entPos).get(),
                par1Entity))) {
            return;
        }
        if (par1Entity.func_70075_an()) {
            if (!par1Entity.func_85031_j(this)) {
                Multimap<String, AttributeModifier> attributes = itemInUse != null ? (Multimap<String, AttributeModifier>) itemInUse.func_111283_C() : null;
                float attackDamage = 2.0F;
                int knockback = 0;

                if (attributes != null) {
                    for (AttributeModifier modifier : attributes.get(SharedMonsterAttributes.field_111264_e.func_111108_a())) {
                        switch (modifier.func_111169_c()) {
                            case 0:
                                attackDamage += modifier.func_111164_d();
                                break;
                            case 1:
                                attackDamage *= modifier.func_111164_d();
                                break;
                            case 2:
                                attackDamage *= 1.0F + modifier.func_111164_d();
                                break;
                        }
                    }
                }

                if (par1Entity instanceof EntityLivingBase) {
                    // FIXME: This was probably meant to do something at some point
                    // attackDamage += EnchantmentHelper.getEnchantmentModifierDamage(this, (EntityLivingBase)
                    // par1Entity);
                    knockback += EnchantmentHelper.func_77501_a(this);
                }

                if (attackDamage > 0.0F) {
                    int fireAspect = EnchantmentHelper.func_90036_a(this);

                    if (par1Entity instanceof EntityLivingBase && fireAspect > 0 && !par1Entity.func_70027_ad()) {
                        par1Entity.func_70015_d(fireAspect * 4);
                    }

                    if (par1Entity.func_70097_a(new EntityDamageSource("robot", this), attackDamage)) {
                        this.setLastAttacker(par1Entity);

                        if (knockback > 0) {
                            par1Entity.func_70024_g((double) (-MathHelper.func_76126_a(this.rotationYaw * (float) Math.PI / 180.0F) * (float) knockback * 0.5F),
                                    0.1D, (double) (MathHelper.func_76134_b(this.rotationYaw * (float) Math.PI / 180.0F) * (float) knockback * 0.5F));
                            this.motionX *= 0.6D;
                            this.motionZ *= 0.6D;
                            this.setSprinting(false);
                        }
                        // FIXME: This was probably meant to do something at some point...

                        // if (par1Entity instanceof EntityLivingBase) {
                        // EnchantmentHelper.((EntityLivingBase) par1Entity, this);
                        // }
                        //
                        // EnchantmentHelper.func_151385_b(this, par1Entity);

                        ItemStack itemstack = itemInUse;

                        if (itemstack != null && par1Entity instanceof EntityLivingBase) {
                            itemstack.func_77973_b().func_77644_a(itemstack, (EntityLivingBase) par1Entity, this);
                        }

                        if (itemInUse.field_77994_a == 0) {
                            setItemInUse(null);
                        }
                    }
                }
            }
        }
    }

    @Override
    public IZone getZoneToWork() {
        return getZone(ActionRobotWorkInArea.AreaType.WORK);
    }

    @Override
    public IZone getZoneToLoadUnload() {
        IZone zone = getZone(ActionRobotWorkInArea.AreaType.LOAD_UNLOAD);
        if (zone == null) {
            zone = getZoneToWork();
        }
        return zone;
    }

    private IZone getZone(AreaType areaType) {
        if (linkedDockingStation != null) {
            for (StatementSlot s : linkedDockingStation.getActiveActions()) {
                if (s.statement instanceof ActionRobotWorkInArea && ((ActionRobotWorkInArea) s.statement).getAreaType() == areaType) {
                    IZone zone = ActionRobotWorkInArea.getArea(s);

                    if (zone != null) {
                        return zone;
                    }
                }
            }
        }

        return null;
    }

    @Override
    public boolean containsItems() {
        for (ItemStack element : inv) {
            if (element != null) {
                return true;
            }
        }

        return false;
    }

    @Override
    public boolean hasFreeSlot() {
        for (ItemStack element : inv) {
            if (element == null) {
                return true;
            }
        }

        return false;
    }

    @Override
    public void unreachableEntityDetected(Entity entity) {
        unreachableEntities.put(entity, worldObj.getTotalWorldTime() + 1200);
    }

    @Override
    public boolean isKnownUnreachable(Entity entity) {
        if (unreachableEntities.containsKey(entity)) {
            if (unreachableEntities.get(entity) >= worldObj.getTotalWorldTime()) {
                return true;
            } else {
                unreachableEntities.remove(entity);
                return false;
            }
        } else {
            return false;
        }
    }

    protected void onRobotHit(boolean attacked) {
        if (!worldObj.isRemote) {
            if (attacked) {
                convertToItems();
            } else {
                if (wearables.size() > 0) {
                    entityDropItem(wearables.remove(wearables.size() - 1), 0);
                    syncWearablesToClient();
                } else if (itemInUse != null) {
                    entityDropItem(itemInUse, 0);
                    setItemInUse(null);
                } else {
                    convertToItems();
                }
            }
        }
    }

    @Override
    protected boolean interact(EntityPlayer player) {
        ItemStack stack = player.func_71045_bC();
        if (stack == null || stack.func_77973_b() == null) {
            return false;
        }

        RobotEvent.Interact robotInteractEvent = new RobotEvent.Interact(this, player, stack);
        MinecraftForge.EVENT_BUS.post(robotInteractEvent);
        if (robotInteractEvent.isCanceled()) {
            return false;
        }

        if (player.func_70093_af() && stack.func_77973_b() == BuildCraftCore.wrenchItem) {
            RobotEvent.Dismantle robotDismantleEvent = new RobotEvent.Dismantle(this, player);
            MinecraftForge.EVENT_BUS.post(robotDismantleEvent);
            if (robotDismantleEvent.isCanceled()) {
                return false;
            }

            onRobotHit(false);

            if (worldObj.isRemote) {
                ((ItemWrench) stack.func_77973_b()).wrenchUsed(player, this);
            }
            return true;
        } else if (wearables.size() < MAX_WEARABLES && stack.func_77973_b().isValidArmor(stack, 0, this)) {
            if (!worldObj.isRemote) {
                wearables.add(stack.func_77979_a(1));
                syncWearablesToClient();
            } else {
                player.func_71038_i();
            }
            return true;
        } else if (wearables.size() < MAX_WEARABLES && stack.func_77973_b() instanceof IRobotOverlayItem && ((IRobotOverlayItem) stack.func_77973_b())
                .isValidRobotOverlay(stack)) {
            if (!worldObj.isRemote) {
                wearables.add(stack.func_77979_a(1));
                syncWearablesToClient();
            } else {
                player.func_71038_i();
            }
            return true;
        } else if (wearables.size() < MAX_WEARABLES && stack.func_77973_b() instanceof ItemSkull) {
            if (!worldObj.isRemote) {
                ItemStack skullStack = stack.func_77979_a(1);
                initSkullItem(skullStack);
                wearables.add(skullStack);
                syncWearablesToClient();
            } else {
                player.func_71038_i();
            }
            return true;
        } else {
            return super.interact(player);
        }
    }

    private void initSkullItem(ItemStack skullStack) {
        if (skullStack.func_77942_o()) {
            NBTTagCompound nbttagcompound = skullStack.func_77978_p();
            GameProfile gameProfile = null;

            if (nbttagcompound.func_150297_b("SkullOwner", NBT.TAG_COMPOUND)) {
                gameProfile = NBTUtil.func_152459_a(nbttagcompound.func_74775_l("SkullOwner"));
            } else if (nbttagcompound.func_150297_b("SkullOwner", NBT.TAG_STRING) && !StringUtils.func_151246_b(nbttagcompound.func_74779_i("SkullOwner"))) {
                gameProfile = new GameProfile(null, nbttagcompound.func_74779_i("SkullOwner"));
            }
            if (gameProfile != null && !StringUtils.func_151246_b(gameProfile.getName())) {
                if (!gameProfile.isComplete() || !gameProfile.getProperties().containsKey("textures")) {
                    // TODO: FIND OUT HOW SKULLS LOAD GAME PROFILES
                    // gameProfile = MinecraftServer.getServer().getGameProfileRepository().(gameProfile.getName());

                    if (gameProfile != null) {
                        Property property = (Property) Iterables.getFirst(gameProfile.getProperties().get("textures"), (Object) null);

                        if (property == null) {
                            // gameProfile =
                            // MinecraftServer.getServer().func_147130_as().fillProfileProperties(gameProfile, true);
                        }
                    }
                }
            }
            if (gameProfile != null && gameProfile.isComplete() && gameProfile.getProperties().containsKey("textures")) {
                NBTTagCompound profileNBT = new NBTTagCompound();
                NBTUtil.func_180708_a(profileNBT, gameProfile);
                nbttagcompound.func_74782_a("SkullOwner", profileNBT);
            } else {
                nbttagcompound.func_82580_o("SkullOwner");
            }
        }
    }

    private void syncWearablesToClient() {
        BuildCraftCore.instance.sendToEntity(new PacketCommand(this, "syncWearables", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                data.writeByte(wearables.size());
                for (ItemStack s : wearables) {
                    NetworkUtils.writeStack(data, s);
                }
            }
        }), this);
    }

    private List<ItemStack> getDrops() {
        List<ItemStack> drops = new ArrayList<>();
        drops.add(ItemRobot.createRobotStack(board.getNBTHandler(), battery.getEnergyStored()));
        if (itemInUse != null) {
            drops.add(itemInUse);
        }
        for (ItemStack element : inv) {
            if (element != null) {
                drops.add(element);
            }
        }
        drops.addAll(wearables);
        return drops;
    }

    private void convertToItems() {
        if (!worldObj.isRemote && !isDead) {
            if (mainAI != null) {
                mainAI.abort();
            }
            List<ItemStack> drops = getDrops();
            for (ItemStack stack : drops) {
                entityDropItem(stack, 0);
            }
            isDead = true;
        }

        getRegistry().killRobot(this);
    }

    @Override
    public void setDead() {
        if (worldObj.isRemote) {
            super.setDead();
        }
    }

    @Override
    public void onChunkUnload() {
        getRegistry().unloadRobot(this);
    }

    @Override
    public boolean canBePushed() {
        return false;
    }

    @Override
    protected void collideWithEntity(Entity par1Entity) {

    }

    @Override
    public void applyEntityCollision(Entity par1Entity) {

    }

    public void setUniqueRobotId(long iRobotId) {
        robotId = iRobotId;
    }

    @Override
    public long getRobotId() {
        return robotId;
    }

    @Override
    public RobotRegistry getRegistry() {
        return (RobotRegistry) RobotManager.registryProvider.getRegistry(worldObj);
    }

    @Override
    public void releaseResources() {
        getRegistry().releaseResources(this);
    }

    /** Tries to receive items in parameters, return items that are left after the operation. */
    @Override
    public ItemStack receiveItem(TileEntity tile, ItemStack stack) {
        if (currentDockingStation != null && currentDockingStation.index().subtract(tile.func_174877_v()).distanceSq(BlockPos.field_177992_a) == 1
            && mainAI != null) {

            return mainAI.getActiveAI().receiveItem(stack);
        } else {
            return stack;
        }
    }

    @Override
    public int fill(EnumFacing from, FluidStack resource, boolean doFill) {
        int result = 0;

        if (tank != null && !tank.isFluidEqual(resource)) {
            return 0;
        }

        if (tank == null) {
            tank = new FluidStack(resource.getFluid(), 0);
        }

        if (tank.amount + resource.amount <= maxFluid) {
            result = resource.amount;

            if (doFill) {
                tank.amount += resource.amount;
            }
        } else {
            result = maxFluid - tank.amount;

            if (doFill) {
                tank.amount = maxFluid;
            }
        }

        if (tank != null && tank.amount == 0) {
            tank = null;
        }

        return result;
    }

    @Override
    public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain) {
        if (tank != null && tank.isFluidEqual(resource)) {
            return drain(from, resource.amount, doDrain);
        } else {
            return null;
        }
    }

    @Override
    public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain) {
        FluidStack result = null;

        if (tank == null) {
            result = null;
        } else if (tank.amount <= maxDrain) {
            result = tank.copy();

            if (doDrain) {
                tank = null;
            }
        } else {
            result = tank.copy();
            result.amount = maxDrain;

            if (doDrain) {
                tank.amount -= maxDrain;
            }
        }

        if (tank != null && tank.amount == 0) {
            tank = null;
        }

        return result;
    }

    @Override
    public boolean canFill(EnumFacing from, Fluid fluid) {
        return tank == null || tank.amount == 0 || (tank.amount < maxFluid && tank.getFluid().getID() == fluid.getID());
    }

    @Override
    public boolean canDrain(EnumFacing from, Fluid fluid) {
        return tank != null && tank.amount != 0 && tank.getFluid().getID() == fluid.getID();
    }

    @Override
    public FluidTankInfo[] getTankInfo(EnumFacing from) {
        return new FluidTankInfo[] { new FluidTankInfo(tank, maxFluid) };
    }

    @Override
    public void getDebugInfo(List<String> left, List<String> right, EnumFacing side) {
        left.add("Robot " + board.getNBTHandler().getID() + " (" + getBattery().getEnergyStored() + "/" + getBattery().getMaxEnergyStored() + " RF)");
        left.add(String.format("Position: %.2f, %.2f, %.2f", posX, posY, posZ));
        left.add("AI tree:");
        AIRobot aiRobot = mainAI;
        while (aiRobot != null) {
            left.add("- " + RobotManager.getAIRobotName(aiRobot.getClass()) + " (" + aiRobot.getEnergyCost() + " RF/t)");
            if (aiRobot instanceof IDebuggable) {
                ((IDebuggable) aiRobot).getDebugInfo(left, right, side);
            }
            aiRobot = aiRobot.getDelegateAI();
        }
    }

    public int receiveEnergy(int maxReceive, boolean simulate) {
        int energyReceived = getBattery().receiveEnergy(maxReceive, simulate);

        // 5 RF/t is set as the "sleep threshold" for detecting charging.
        if (!simulate && energyReceived > 5 && ticksCharging <= 25) {
            ticksCharging += 5;
        }

        return energyReceived;
    }

    public List<ItemStack> getWearables() {
        return wearables;
    }

    // Something to do with IInventory

    @Override
    public int func_174887_a_(int id) {
        return 0;
    }

    @Override
    public void func_174885_b(int id, int value) {}

    @Override
    public int func_174890_g() {
        return 0;
    }

    @Override
    public void func_174888_l() {}

    private void updateItem(ItemStack stack, int i, boolean held) {
        if (stack != null && stack.func_77973_b() != null) {
            int id = Item.func_150891_b(stack.func_77973_b());
            // did this item not throw an exception before?
            if (!blacklistedItemsForUpdate.contains(id)) {
                try {
                    stack.func_77973_b().func_77663_a(stack, worldObj, this, i, held);
                } catch (Exception e) {
                    // the item threw an exception, print it and do not let it update once more
                    e.printStackTrace();
                    blacklistedItemsForUpdate.add(id);
                }
            }
        }
    }
}
