/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * The BuildCraft API is distributed under the terms of the MIT License. Please check the contents of the license, which
 * should be located as "LICENSE.API" in the BuildCraft source code distribution. */
package buildcraft.robotics;

import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.World;

import buildcraft.api.robots.*;
import buildcraft.core.lib.utils.NBTUtils;

public class StackRequest {
    private IRequestProvider requester;

    private int slot;

    private ItemStack stack;

    private DockingStation station;
    private BlockPos stationIndex;
    private EnumFacing stationSide;

    public StackRequest(IRequestProvider requester, int slot, ItemStack stack) {
        this.requester = requester;
        this.slot = slot;
        this.stack = stack;
        this.station = null;
    }

    private StackRequest(int slot, ItemStack stack, BlockPos stationIndex, EnumFacing stationSide) {
        requester = null;
        this.slot = slot;
        this.stack = stack;
        station = null;
        this.stationIndex = stationIndex;
        this.stationSide = stationSide;
    }

    public IRequestProvider getRequester(World world) {
        if (requester == null) {
            DockingStation dockingStation = getStation(world);
            if (dockingStation != null) {
                requester = dockingStation.getRequestProvider();
            }
        }
        return requester;
    }

    public int getSlot() {
        return slot;
    }

    public ItemStack getStack() {
        return stack;
    }

    public DockingStation getStation(World world) {
        if (station == null) {
            IRobotRegistry robotRegistry = RobotManager.registryProvider.getRegistry(world);
            station = robotRegistry.getStation(stationIndex, stationSide);
        }
        return station;
    }

    public void setStation(DockingStation station) {
        this.station = station;
        this.stationIndex = station.index();
        this.stationSide = station.side();
    }

    public void writeToNBT(NBTTagCompound nbt) {
        nbt.func_74768_a("slot", slot);

        NBTTagCompound stackNBT = new NBTTagCompound();
        stack.func_77955_b(stackNBT);
        nbt.func_74782_a("stack", stackNBT);

        if (station != null) {
            nbt.func_74782_a("stationIndex", NBTUtils.writeBlockPos(stationIndex));
            nbt.func_74774_a("stationSide", (byte) station.side().ordinal());
        }
    }

    public static StackRequest loadFromNBT(NBTTagCompound nbt) {
        if (nbt.func_74764_b("stationIndex")) {
            int slot = nbt.func_74762_e("slot");

            ItemStack stack = ItemStack.func_77949_a(nbt.func_74775_l("stack"));

            BlockPos stationIndex = NBTUtils.readBlockPos(nbt.func_74781_a("stationIndex"));
            EnumFacing stationSide = EnumFacing.values()[nbt.func_74771_c("stationSide")];

            return new StackRequest(slot, stack, stationIndex, stationSide);
        } else {
            return null;
        }
    }

    public ResourceId getResourceId(World world) {
        return getStation(world) != null ? new ResourceIdRequest(getStation(world), slot) : null;
    }
}
