/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.robotics;

import java.util.Arrays;

import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.server.MinecraftServer;
import net.minecraft.util.IChatComponent;

import buildcraft.BuildCraftCore;
import buildcraft.BuildCraftRobotics;
import buildcraft.api.core.IZone;
import buildcraft.api.core.SafeTimeTracker;
import buildcraft.api.items.IMapLocation;
import buildcraft.api.items.INamedItem;
import buildcraft.core.ItemMapLocation;
import buildcraft.core.lib.block.TileBuildCraft;
import buildcraft.core.lib.inventory.SimpleInventory;
import buildcraft.core.lib.network.base.Packet;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.robotics.gui.ContainerZonePlan;
import buildcraft.robotics.map.MapWorld;

import io.netty.buffer.ByteBuf;

public class TileZonePlan extends TileBuildCraft implements IInventory {

    public static final int RESOLUTION = 2048;
    public static final int CRAFT_TIME = 120;
    private static final int PREVIEW_BLOCKS_PER_PIXEL = 10;
    private static int RESOLUTION_CHUNKS = RESOLUTION >> 4;

    public int chunkStartX, chunkStartZ;
    public short progress = 0;
    public String mapName = "";

    private final byte[] previewColors = new byte[80];
    private final SimpleInventory inv = new SimpleInventory(3, "inv", 64);
    private final SafeTimeTracker previewRecalcTimer = new SafeTimeTracker(100);

    private boolean previewColorsPushed = false;
    private ZonePlan[] selectedAreas = new ZonePlan[16];
    private int currentSelectedArea = 0;

    public byte[] getPreviewTexture(boolean force) {
        if (!previewColorsPushed || force) {
            previewColorsPushed = true;
            return previewColors;
        }
        return null;
    }

    @Override
    public void initialize() {
        super.initialize();

        int chunkX = field_174879_c.func_177958_n() >> 4;
        int chunkZ = field_174879_c.func_177952_p() >> 4;

        chunkStartX = chunkX - RESOLUTION_CHUNKS / 2;
        chunkStartZ = chunkZ - RESOLUTION_CHUNKS / 2;
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (previewRecalcTimer.markTimeIfDelay(field_145850_b)) {
            recalculatePreview();
        }

        if (inv.func_70301_a(0) != null && inv.func_70301_a(1) == null && inv.func_70301_a(0).func_77973_b() instanceof ItemMapLocation) {

            if (progress < CRAFT_TIME) {
                progress++;

                if (field_145850_b.func_82737_E() % 5 == 0) {
                    sendNetworkUpdate();
                }
            } else {
                ItemStack stack = inv.func_70298_a(0, 1);

                if (selectedAreas[currentSelectedArea] != null) {
                    ItemMapLocation.setZone(stack, selectedAreas[currentSelectedArea]);
                    ((INamedItem) stack.func_77973_b()).setName(stack, mapName);
                }

                inv.func_70299_a(1, stack);
            }
        } else if (progress != 0) {
            progress = 0;
            sendNetworkUpdate();
        }
    }

    private void recalculatePreview() {
        byte[] newPreviewColors = new byte[80];
        MapWorld mw = BuildCraftRobotics.manager.getWorld(field_145850_b);

        for (int y = 0; y < 8; y++) {
            for (int x = 0; x < 10; x++) {
                int tx = (x * PREVIEW_BLOCKS_PER_PIXEL) - (5 * PREVIEW_BLOCKS_PER_PIXEL) + (PREVIEW_BLOCKS_PER_PIXEL / 2);
                int ty = (y * PREVIEW_BLOCKS_PER_PIXEL) - (4 * PREVIEW_BLOCKS_PER_PIXEL) + (PREVIEW_BLOCKS_PER_PIXEL / 2);
                newPreviewColors[y * 10 + x] = (byte) mw.getColor(func_174877_v().func_177958_n() - (func_174877_v().func_177958_n() % PREVIEW_BLOCKS_PER_PIXEL) + tx, func_174877_v().func_177952_p()
                    - (func_174877_v().func_177952_p() % PREVIEW_BLOCKS_PER_PIXEL) + ty);
            }
        }

        if (!Arrays.equals(previewColors, newPreviewColors)) {
            System.arraycopy(newPreviewColors, 0, previewColors, 0, 80);
            sendNetworkUpdate();
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);
        nbt.func_74778_a("name", mapName);

        NBTTagCompound invNBT = new NBTTagCompound();
        inv.writeToNBT(invNBT);
        nbt.func_74782_a("inv", invNBT);

        for (int i = 0; i < selectedAreas.length; ++i) {
            if (selectedAreas[i] != null) {
                NBTTagCompound subNBT = new NBTTagCompound();
                selectedAreas[i].writeToNBT(subNBT);
                nbt.func_74782_a("selectedArea[" + i + "]", subNBT);
            }
        }
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);

        mapName = nbt.func_74779_i("name");

        if (mapName == null) {
            mapName = "";
        }

        inv.readFromNBT(nbt.func_74775_l("inv"));

        for (int i = 0; i < selectedAreas.length; ++i) {
            if (nbt.func_74764_b("selectedArea[" + i + "]")) {
                selectedAreas[i] = new ZonePlan();
                selectedAreas[i].readFromNBT(nbt.func_74775_l("selectedArea[" + i + "]"));
            }
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        stream.writeShort(progress);
        NetworkUtils.writeUTF(stream, mapName);
        stream.writeBytes(previewColors, 0, 80);
    }

    @Override
    public void readData(ByteBuf stream) {
        progress = stream.readShort();
        mapName = NetworkUtils.readUTF(stream);
        stream.readBytes(previewColors, 0, 80);
        previewColorsPushed = false;
    }

    private void importMap(ItemStack stack) {
        if (stack != null && stack.func_77973_b() instanceof IMapLocation) {
            final IZone zone = ((IMapLocation) stack.func_77973_b()).getZone(stack);
            if (zone != null && zone instanceof ZonePlan) {
                selectedAreas[currentSelectedArea] = (ZonePlan) zone;

                for (EntityPlayerMP e : MinecraftServer.func_71276_C().func_71203_ab().field_72404_b) {
                    if (e.field_71070_bA != null && e.field_71070_bA instanceof ContainerZonePlan && ((ContainerZonePlan) e.field_71070_bA)
                            .getTile() == this) {
                        Packet p = new PacketCommand(e.field_71070_bA, "areaLoaded", new CommandWriter() {
                            @Override
                            public void write(ByteBuf data) {
                                ((ZonePlan) zone).writeData(data);
                            }
                        });

                        BuildCraftCore.instance.sendToPlayer(e, p);
                    }
                }
            }
        }
    }

    public ZonePlan selectArea(int index) {
        if (selectedAreas[index] == null) {
            selectedAreas[index] = new ZonePlan();
        }

        currentSelectedArea = index;

        return selectedAreas[index];
    }

    public void setArea(int index, ZonePlan area) {
        selectedAreas[index] = area;
    }

    @Override
    public int func_70302_i_() {
        return inv.func_70302_i_();
    }

    @Override
    public ItemStack func_70301_a(int slotId) {
        return inv.func_70301_a(slotId);
    }

    @Override
    public ItemStack func_70298_a(int slotId, int count) {
        return inv.func_70298_a(slotId, count);
    }

    @Override
    public ItemStack func_70304_b(int slotId) {
        return inv.func_70304_b(slotId);
    }

    @Override
    public void func_70299_a(int slotId, ItemStack itemstack) {
        inv.func_70299_a(slotId, itemstack);

        if (!field_145850_b.field_72995_K && slotId == 2) {
            importMap(itemstack);
        }
    }

    @Override
    public IChatComponent getDisplayName() {
        return inv.func_145748_c_();
    }

    @Override
    public boolean hasCustomName() {
        return inv.func_145818_k_();
    }

    @Override
    public int func_70297_j_() {
        return inv.func_70297_j_();
    }

    @Override
    public boolean func_70300_a(EntityPlayer entityplayer) {
        return inv.func_70300_a(entityplayer);
    }

    @Override
    public void func_174889_b(EntityPlayer player) {
        inv.func_174889_b(player);
    }

    @Override
    public void func_174886_c(EntityPlayer player) {
        inv.func_174886_c(player);
    }

    @Override
    public boolean func_94041_b(int i, ItemStack itemstack) {
        return inv.func_94041_b(i, itemstack);
    }
}
