/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.robotics;

import java.util.HashMap;
import java.util.Map;
import java.util.Random;

import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.BlockPos;
import net.minecraft.util.Vec3;
import net.minecraft.world.ChunkCoordIntPair;

import net.minecraftforge.common.util.Constants;

import buildcraft.api.core.ISerializable;
import buildcraft.api.core.IZone;

import io.netty.buffer.ByteBuf;

public class ZonePlan implements IZone, ISerializable {
    private final HashMap<ChunkCoordIntPair, ZoneChunk> chunkMapping = new HashMap<>();

    public boolean get(int x, int z) {
        int xChunk = x >> 4;
        int zChunk = z >> 4;
        ChunkCoordIntPair chunkId = new ChunkCoordIntPair(xChunk, zChunk);
        ZoneChunk property;

        if (!chunkMapping.containsKey(chunkId)) {
            return false;
        } else {
            property = chunkMapping.get(chunkId);
            return property.get(x & 0xF, z & 0xF);
        }
    }

    public void set(int x, int z, boolean val) {
        int xChunk = x >> 4;
        int zChunk = z >> 4;
        ChunkCoordIntPair chunkId = new ChunkCoordIntPair(xChunk, zChunk);
        ZoneChunk property;

        if (!chunkMapping.containsKey(chunkId)) {
            if (val) {
                property = new ZoneChunk();
                chunkMapping.put(chunkId, property);
            } else {
                return;
            }
        } else {
            property = chunkMapping.get(chunkId);
        }

        property.set(x & 0xF, z & 0xF, val);

        if (property.isEmpty()) {
            chunkMapping.remove(chunkId);
        }
    }

    public void writeToNBT(NBTTagCompound nbt) {
        NBTTagList list = new NBTTagList();

        for (Map.Entry<ChunkCoordIntPair, ZoneChunk> e : chunkMapping.entrySet()) {
            NBTTagCompound subNBT = new NBTTagCompound();
            subNBT.func_74768_a("chunkX", e.getKey().field_77276_a);
            subNBT.func_74768_a("chunkZ", e.getKey().field_77275_b);
            e.getValue().writeToNBT(subNBT);
            list.func_74742_a(subNBT);
        }

        nbt.func_74782_a("chunkMapping", list);
    }

    public void readFromNBT(NBTTagCompound nbt) {
        NBTTagList list = nbt.func_150295_c("chunkMapping", Constants.NBT.TAG_COMPOUND);

        for (int i = 0; i < list.func_74745_c(); ++i) {
            NBTTagCompound subNBT = list.func_150305_b(i);

            ChunkCoordIntPair id = new ChunkCoordIntPair(subNBT.func_74762_e("chunkX"), subNBT.func_74762_e("chunkZ"));

            ZoneChunk chunk = new ZoneChunk();
            chunk.readFromNBT(subNBT);

            chunkMapping.put(id, chunk);
        }
    }

    @Override
    public double distanceTo(BlockPos index) {
        return Math.sqrt(distanceToSquared(index));
    }

    @Override
    public double distanceToSquared(BlockPos index) {
        double maxSqrDistance = Double.MAX_VALUE;

        for (Map.Entry<ChunkCoordIntPair, ZoneChunk> e : chunkMapping.entrySet()) {
            double dx = (e.getKey().field_77276_a << 4 + 8) - index.func_177958_n();
            double dz = (e.getKey().field_77275_b << 4 + 8) - index.func_177952_p();

            double sqrDistance = dx * dx + dz * dz;

            if (sqrDistance < maxSqrDistance) {
                maxSqrDistance = sqrDistance;
            }
        }

        return maxSqrDistance;
    }

    @Override
    public boolean contains(Vec3 point) {
        int xBlock = (int) Math.floor(point.field_72450_a);
        int zBlock = (int) Math.floor(point.field_72449_c);

        return get(xBlock, zBlock);
    }

    @Override
    public BlockPos getRandomBlockPos(Random rand) {
        if (chunkMapping.size() == 0) {
            return null;
        }

        int chunkId = rand.nextInt(chunkMapping.size());

        for (Map.Entry<ChunkCoordIntPair, ZoneChunk> e : chunkMapping.entrySet()) {
            if (chunkId == 0) {
                BlockPos i = e.getValue().getRandomBlockPos(rand);
                int x = (e.getKey().field_77276_a << 4) + i.func_177958_n();
                int z = (e.getKey().field_77275_b << 4) + i.func_177952_p();

                return new BlockPos(x, i.func_177956_o(), z);
            }

            chunkId--;
        }

        return null;
    }

    @Override
    public void readData(ByteBuf stream) {
        chunkMapping.clear();
        int size = stream.readInt();
        for (int i = 0; i < size; i++) {
            ChunkCoordIntPair key = new ChunkCoordIntPair(stream.readInt(), stream.readInt());
            ZoneChunk value = new ZoneChunk();
            value.readData(stream);
            chunkMapping.put(key, value);
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        stream.writeInt(chunkMapping.size());
        for (Map.Entry<ChunkCoordIntPair, ZoneChunk> e : chunkMapping.entrySet()) {
            stream.writeInt(e.getKey().field_77276_a);
            stream.writeInt(e.getKey().field_77275_b);
            e.getValue().writeData(stream);
        }
    }
}
