/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.robotics.ai;

import buildcraft.api.core.BuildCraftAPI;
import buildcraft.api.robots.EntityRobotBase;
import buildcraft.core.lib.utils.IterableAlgorithmRunner;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.PathFinding;
import buildcraft.core.lib.utils.Utils;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.BlockPos;
import net.minecraftforge.common.util.Constants;

import java.util.LinkedList;

public class AIRobotGotoBlock extends AIRobotGoto {

    private PathFinding pathSearch;
    private IterableAlgorithmRunner pathSearchJob;
    private LinkedList<BlockPos> path;
    private double prevDistance = Double.MAX_VALUE;
    private BlockPos finalPos;
    private double maxDistance = 0;
    private BlockPos lastBlockInPath;
    private boolean loadedFromNBT;

    public AIRobotGotoBlock(EntityRobotBase iRobot) {
        super(iRobot);
    }

    public AIRobotGotoBlock(EntityRobotBase robot, BlockPos pos) {
        this(robot);
        finalPos = pos;
    }

    public AIRobotGotoBlock(EntityRobotBase robot, BlockPos pos, double iMaxDistance) {
        this(robot, pos);
        maxDistance = iMaxDistance;
    }

    public AIRobotGotoBlock(EntityRobotBase robot, LinkedList<BlockPos> iPath) {
        this(robot);
        path = iPath;
        finalPos = path.getLast();
        setNextInPath();
    }

    @Override
    public void start() {
        robot.undock();
    }

    @Override
    public void update() {
        if (loadedFromNBT) {
            // Prevent a race condition with terminate() being called in
            // setNextInPath.
            setNextInPath();
            loadedFromNBT = false;
        }

        if (path == null && pathSearch == null) {
            pathSearch = new PathFinding(robot.worldObj, new BlockPos((int) Math.floor(robot.posX), (int) Math.floor(robot.posY), (int) Math.floor(
                    robot.posZ)), finalPos, maxDistance);

            pathSearchJob = new IterableAlgorithmRunner(pathSearch, 50);
            pathSearchJob.start();
        } else if (path != null && next != null) {
            double distance = robot.getDistance(next.field_72450_a, next.field_72448_b, next.field_72449_c);

            if (!robot.isMoving() || distance > prevDistance) {
                if (path.size() > 0) {
                    path.removeFirst();
                }

                setNextInPath();
            } else {
                prevDistance = robot.getDistance(next.field_72450_a, next.field_72448_b, next.field_72449_c);
            }
        } else {
            if (pathSearchJob.isDone()) {
                path = pathSearch.getResult();

                if (path.size() == 0) {
                    setSuccess(false);
                    terminate();
                    return;
                }

                lastBlockInPath = path.getLast();

                setNextInPath();
            }
        }

        if (path != null && path.size() == 0) {
            robot.motionX = 0;
            robot.motionY = 0;
            robot.motionZ = 0;

            if (lastBlockInPath != null) {
                robot.posX = lastBlockInPath.func_177958_n() + 0.5F;
                robot.posY = lastBlockInPath.func_177956_o() + 0.5F;
                robot.posZ = lastBlockInPath.func_177952_p() + 0.5F;
            }
            terminate();
        }
    }

    private void setNextInPath() {
        if (path != null && path.size() > 0) {

            boolean isFirst = prevDistance == Double.MAX_VALUE;

            BlockPos next = path.getFirst();
            if (isFirst || BuildCraftAPI.isSoftBlock(robot.worldObj, next)) {
                setDestination(robot, Utils.convertMiddle(next));
                prevDistance = Double.MAX_VALUE;
                robot.aimItemAt(next);
            } else {
                // Path invalid!
                path = null;

                if (pathSearchJob != null) {
                    pathSearchJob.terminate();
                    robot.motionX = 0;
                    robot.motionY = 0;
                    robot.motionZ = 0;
                }
            }
        }
    }

    @Override
    public void end() {
        if (pathSearchJob != null) {
            pathSearchJob.terminate();
            robot.motionX = 0;
            robot.motionY = 0;
            robot.motionZ = 0;
        }
    }

    @Override
    public boolean canLoadFromNBT() {
        return true;
    }

    @Override
    public void writeSelfToNBT(NBTTagCompound nbt) {
        super.writeSelfToNBT(nbt);

        nbt.func_74782_a("finalPos", NBTUtils.writeBlockPos(finalPos));
        nbt.func_74780_a("maxDistance", maxDistance);

        if (path != null) {
            NBTTagList pathList = new NBTTagList();

            for (BlockPos i : path) {
                pathList.func_74742_a(NBTUtils.writeBlockPos(i));
            }

            nbt.func_74782_a("path", pathList);
        }
    }

    @Override
    public void loadSelfFromNBT(NBTTagCompound nbt) {
        super.loadSelfFromNBT(nbt);

        finalPos = NBTUtils.readBlockPos(nbt.func_74781_a("finalPos"));
        maxDistance = nbt.func_74769_h("maxDistance");

        if (nbt.func_74764_b("path")) {
            NBTTagList pathList = nbt.func_150295_c("path", Constants.NBT.TAG_COMPOUND);

            path = new LinkedList<>();

            for (int i = 0; i < pathList.func_74745_c(); ++i) {
                path.add(NBTUtils.readBlockPos(pathList.func_179238_g(i)));
            }
        }

        loadedFromNBT = true;
    }
}
