package buildcraft.robotics.map;

import java.io.File;
import java.util.Date;

import com.google.common.collect.HashBiMap;

import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.chunk.Chunk;
import net.minecraft.world.chunk.IChunkProvider;
import net.minecraft.world.gen.ChunkProviderServer;

import net.minecraftforge.common.DimensionManager;
import net.minecraftforge.event.world.BlockEvent;
import net.minecraftforge.event.world.ChunkEvent;
import net.minecraftforge.event.world.WorldEvent;
import net.minecraftforge.fml.common.eventhandler.SubscribeEvent;
import net.minecraftforge.fml.common.gameevent.TickEvent;
import net.minecraftforge.fml.relauncher.Side;

import buildcraft.core.lib.utils.Utils;

public class MapManager implements Runnable {
    private static final int UPDATE_DELAY = 60000;
    private final HashBiMap<World, MapWorld> worldMap = HashBiMap.create();
    private final File location;
    private boolean stop = false;
    private long lastSaveTime;

    public MapManager(File location) {
        this.location = location;
    }

    public void stop() {
        stop = true;
        saveAllWorlds();
    }

    public MapWorld getWorld(World world) {
        if (world == null || world.field_72995_K) {
            return null;
        }

        if (!worldMap.containsKey(world)) {
            synchronized (worldMap) {
                worldMap.put(world, new MapWorld(world, location));
            }
        }
        return worldMap.get(world);
    }

    private static boolean doUpdate(MapWorld world, Chunk chunk) {
        int x = chunk.field_76635_g;
        int z = chunk.field_76647_h;
        long updateTime = (new Date()).getTime() - UPDATE_DELAY;
        return world.getUpdateTime(x, z) < updateTime || !world.hasChunk(x, z);
    }

    private void updateChunk(World rworld, Chunk chunk, boolean force) {
        MapWorld world = getWorld(rworld);
        if (world != null && (force || doUpdate(world, chunk))) {
            world.updateChunk(chunk);
        }
    }

    private void updateChunkDelayed(World rworld, Chunk chunk, boolean force, byte time) {
        MapWorld world = getWorld(rworld);
        if (world != null && (force || doUpdate(world, chunk))) {
            world.updateChunkDelayed(chunk, time);
        }
    }

    @SubscribeEvent
    public void tickDelayedWorlds(TickEvent.WorldTickEvent event) {
        if (event.phase == TickEvent.Phase.END && event.side == Side.SERVER) {
            MapWorld w = worldMap.get(event.world);
            if (w != null) {
                w.tick();
            }
        }
    }

    @SubscribeEvent
    public void worldUnloaded(WorldEvent.Unload event) {
        if (worldMap.containsKey(event.world)) {
            worldMap.get(event.world).save();
            synchronized (worldMap) {
                worldMap.remove(event.world);
            }
        }
    }

    @SubscribeEvent
    public void chunkLoaded(ChunkEvent.Load event) {
        updateChunkDelayed(event.world, event.getChunk(), false, (byte) (40 + Utils.RANDOM.nextInt(20)));
    }

    @SubscribeEvent
    public void chunkUnloaded(ChunkEvent.Unload event) {
        updateChunk(event.world, event.getChunk(), false);
    }

    @SubscribeEvent
    public void blockPlaced(BlockEvent.PlaceEvent placeEvent) {
        Chunk chunk = placeEvent.world.func_175726_f(placeEvent.pos);
        MapWorld world = getWorld(placeEvent.world);
        if (world != null && doUpdate(world, chunk)) {
            int hv = placeEvent.world.func_175645_m(placeEvent.pos).func_177956_o();
            if (placeEvent.pos.func_177956_o() >= (hv - 3)) {
                world.updateChunk(chunk);
            }
        }
    }

    @SubscribeEvent
    public void blockBroken(BlockEvent.BreakEvent placeEvent) {
        Chunk chunk = placeEvent.world.func_175726_f(placeEvent.pos);
        MapWorld world = getWorld(placeEvent.world);
        if (world != null && doUpdate(world, chunk)) {
            int hv = placeEvent.world.func_175645_m(placeEvent.pos).func_177956_o();
            if (placeEvent.pos.func_177956_o() >= (hv - 3)) {
                world.updateChunk(chunk);
            }
        }
    }

    public void saveAllWorlds() {
        synchronized (worldMap) {
            for (MapWorld world : worldMap.values()) {
                world.save();
            }
        }
    }

    @Override
    public void run() {
        lastSaveTime = (new Date()).getTime();

        while (!stop) {
            long now = (new Date()).getTime();

            if (now - lastSaveTime > 120000) {
                saveAllWorlds();
                lastSaveTime = now;
            }

            try {
                Thread.sleep(4000);
            } catch (Exception e) {

            }
        }
    }

    public void initialize() {
        for (WorldServer ws : DimensionManager.getWorlds()) {
            MapWorld mw = getWorld(ws);
            IChunkProvider provider = ws.func_72863_F();
            if (provider instanceof ChunkProviderServer) {
                for (Object o : ((ChunkProviderServer) provider).func_152380_a()) {
                    if (o != null && o instanceof Chunk) {
                        Chunk c = (Chunk) o;
                        if (!mw.hasChunk(c.field_76635_g, c.field_76647_h)) {
                            mw.updateChunkDelayed(c, (byte) (40 + Utils.RANDOM.nextInt(20)));
                        }
                    }
                }
            }
        }
    }
}
