/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.silicon;

import buildcraft.BuildCraftSilicon;
import buildcraft.api.core.IInvSlot;
import buildcraft.api.core.IStackFilter;
import buildcraft.api.power.ILaserTarget;
import buildcraft.core.lib.inventory.*;
import buildcraft.core.lib.inventory.filters.CraftingFilter;
import buildcraft.core.lib.network.PacketSlotChange;
import buildcraft.core.lib.utils.BCStringUtils;
import buildcraft.core.lib.utils.CraftingUtils;
import buildcraft.core.lib.utils.Utils;
import buildcraft.core.proxy.CoreProxy;
import com.google.common.collect.Lists;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.inventory.*;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.world.WorldServer;
import net.minecraftforge.oredict.OreDictionary;

import java.lang.ref.WeakReference;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;

public class TileAdvancedCraftingTable extends TileLaserTableBase implements IInventory, ILaserTarget, ISidedInventory, PacketSlotChange.ITile {

    private static final int[] SLOTS = Utils.createSlotArray(0, 24);
    private static final EnumSet<EnumFacing> SEARCH_SIDES = EnumSet.of(EnumFacing.DOWN, EnumFacing.NORTH, EnumFacing.SOUTH, EnumFacing.EAST,
            EnumFacing.WEST);
    private static final int REQUIRED_POWER = 5000;
    private final CraftingGrid craftingSlots;
    private final InventoryMapper invInput;
    private final InventoryMapper invOutput;
    private SlotCrafting craftSlot;
    private boolean craftable;
    private boolean justCrafted;
    private IRecipe currentRecipe;
    private InventoryCraftResult craftResult;
    private InternalInventoryCrafting internalInventoryCrafting;

    private final class InternalInventoryCraftingContainer extends Container {

        @Override
        public boolean func_75145_c(EntityPlayer var1) {
            return false;
        }
    }

    private final class CraftingGrid extends SimpleInventory {

        public int[][] oreIDs = new int[9][];

        public CraftingGrid() {
            super(9, "CraftingSlots", 1);
            Arrays.fill(oreIDs, new int[0]);
        }

        @Override
        public void func_70299_a(int slotId, ItemStack itemstack) {
            super.func_70299_a(slotId, itemstack);

            if (TileAdvancedCraftingTable.this.func_145831_w() == null || !TileAdvancedCraftingTable.this.func_145831_w().field_72995_K) {
                int[] id = new int[0];
                if (itemstack != null) {
                    int[] ids = OreDictionary.getOreIDs(itemstack);
                    if (ids.length > 0) {
                        id = ids;
                    }
                }
                oreIDs[slotId] = id;
            }
        }
    }

    private final class InternalInventoryCrafting extends InventoryCrafting {

        public int[] hitCount;
        private int[] bindings = new int[9];
        private ItemStack[] tempStacks;
        private boolean useRecipeStack;

        private InternalInventoryCrafting() {
            super(new InternalInventoryCraftingContainer(), 3, 3);
        }

        @Override
        public ItemStack func_70301_a(int slot) {
            if (slot >= 0 && slot < 9) {
                if (useRecipeStack || tempStacks == null) {
                    return craftingSlots.func_70301_a(slot);
                } else {
                    if (bindings[slot] >= 0) {
                        return tempStacks[bindings[slot]];
                    }
                }
            }

            // vanilla returns null for out of bound stacks in InventoryCrafting as well
            return null;
        }

        @Override
        public void func_70299_a(int slot, ItemStack par2ItemStack) {
            if (tempStacks != null && slot >= 0 && slot < 9 && bindings[slot] >= 0) {
                tempStacks[bindings[slot]] = par2ItemStack;
            }
        }

        @Override
        public ItemStack func_70298_a(int slot, int amount) {
            if (tempStacks != null && slot >= 0 && slot < 9 && bindings[slot] >= 0) {
                if (tempStacks[bindings[slot]].field_77994_a <= amount) {
                    ItemStack result = tempStacks[bindings[slot]];
                    tempStacks[bindings[slot]] = null;
                    return result;
                } else {
                    ItemStack result = tempStacks[bindings[slot]].func_77979_a(amount);

                    if (tempStacks[bindings[slot]].field_77994_a <= 0) {
                        tempStacks[bindings[slot]] = null;
                    }

                    return result;
                }
            } else {
                return null;
            }
        }

        public void recipeUpdate(boolean flag) {
            useRecipeStack = flag;
        }
    }

    public TileAdvancedCraftingTable() {
        craftingSlots = new CraftingGrid();
        inv.addListener(this);
        invInput = new InventoryMapper(inv, 0, 15);
        invOutput = new InventoryMapper(inv, 15, 9);
        craftResult = new InventoryCraftResult();
    }

    public WeakReference<EntityPlayer> getInternalPlayer() {
        return CoreProxy.proxy.getBuildCraftPlayer((WorldServer) field_145850_b, func_174877_v().func_177984_a());
    }

    @Override
    public void func_145841_b(NBTTagCompound data) {
        super.func_145841_b(data);
        craftingSlots.writeToNBT(data, "craftingSlots");
    }

    @Override
    public void func_145839_a(NBTTagCompound data) {
        super.func_145839_a(data);
        if (data.func_74764_b("StorageSlots")) {
            inv.readFromNBT(data, "StorageSlots");
        }

        if (data.func_74764_b("items")) {
            craftingSlots.readFromNBT(data);
        } else {
            craftingSlots.readFromNBT(data, "craftingSlots");
        }
    }

    @Override
    public int func_70302_i_() {
        return 24;
    }

    @Override
    public String getInventoryName() {
        return BCStringUtils.localize("tile.assemblyWorkbenchBlock.name");
    }

    @Override
    public void func_70296_d() {
        super.func_70296_d();
        craftable = craftResult.func_70301_a(0) != null;
    }

    @Override
    public int getRequiredEnergy() {
        return craftResult.func_70301_a(0) != null ? REQUIRED_POWER : 0;
    }

    @Override
    public int getProgressScaled(int i) {
        return (getEnergy() * i) / REQUIRED_POWER;
    }

    @Override
    public void func_73660_a() {
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (internalInventoryCrafting == null) {
            internalInventoryCrafting = new InternalInventoryCrafting();
            craftSlot = new SlotCrafting(getInternalPlayer().get(), internalInventoryCrafting, craftResult, 0, 0, 0);
            updateRecipe();
        }
        if (field_145850_b.field_72995_K) {
            return;
        }
        updateRecipe();
        searchNeighborsForIngredients();
        locateAndBindIngredients();
        updateRecipeOutputDisplay();
        justCrafted = false;
        if (canCraftAndOutput()) {
            if (getEnergy() >= getRequiredEnergy()) {
                craftItem();
                justCrafted = true;
            }
        } else {
            craftable = false;
            internalInventoryCrafting.tempStacks = null;
            internalInventoryCrafting.hitCount = null;
            setEnergy(0);
        }
    }

    private boolean canCraftAndOutput() {
        if (!hasIngredients()) {
            return false;
        }
        ItemStack output = getRecipeOutput();
        if (output == null) {
            return false;
        }
        return InvUtils.isRoomForStack(output, EnumFacing.UP, invOutput);
    }

    private void locateAndBindIngredients() {
        internalInventoryCrafting.tempStacks = new InventoryCopy(inv).getItemStacks();
        internalInventoryCrafting.hitCount = new int[internalInventoryCrafting.tempStacks.length];
        ItemStack[] inputSlots = internalInventoryCrafting.tempStacks;

        for (int gridSlot = 0; gridSlot < craftingSlots.func_70302_i_(); gridSlot++) {
            internalInventoryCrafting.bindings[gridSlot] = -1;

            if (craftingSlots.func_70301_a(gridSlot) == null) {
                continue;
            }

            boolean foundMatch = false;

            for (int inputSlot = 0; inputSlot < inputSlots.length; inputSlot++) {
                if (!isMatchingIngredient(gridSlot, inputSlot)) {
                    continue;
                }

                if (internalInventoryCrafting.hitCount[inputSlot] < inputSlots[inputSlot].field_77994_a
                    && internalInventoryCrafting.hitCount[inputSlot] < inputSlots[inputSlot].func_77976_d()) {
                    internalInventoryCrafting.bindings[gridSlot] = inputSlot;
                    internalInventoryCrafting.hitCount[inputSlot]++;
                    foundMatch = true;
                    break;
                }
            }

            if (!foundMatch) {
                return;
            }
        }
    }

    private boolean isMatchingIngredient(int gridSlot, int inputSlot) {
        ItemStack inputStack = internalInventoryCrafting.tempStacks[inputSlot];

        if (inputStack == null) {
            return false;
        } else if (StackHelper.isMatchingItem(craftingSlots.func_70301_a(gridSlot), inputStack, true, false)) {
            return true;
        } else {
            return StackHelper.isCraftingEquivalent(craftingSlots.oreIDs[gridSlot], inputStack);
        }
    }

    private boolean hasIngredients() {
        return currentRecipe != null && currentRecipe.func_77569_a(internalInventoryCrafting, field_145850_b);
    }

    private void craftItem() {
        EntityPlayer internalPlayer = getInternalPlayer().get();
        ItemStack recipeOutput = getRecipeOutput();
        craftSlot.func_82870_a(internalPlayer, recipeOutput);
        ItemStack[] tempStorage = internalInventoryCrafting.tempStacks;

        for (int i = 0; i < tempStorage.length; i++) {
            if (tempStorage[i] != null && tempStorage[i].field_77994_a <= 0) {
                tempStorage[i] = null;
            }

            inv.getItemStacks()[i] = tempStorage[i];
        }

        subtractEnergy(getRequiredEnergy());
        List<ItemStack> outputs = Lists.newArrayList(recipeOutput.func_77946_l());

        for (int i = 0; i < internalPlayer.field_71071_by.field_70462_a.length; i++) {
            if (internalPlayer.field_71071_by.field_70462_a[i] != null) {
                outputs.add(internalPlayer.field_71071_by.field_70462_a[i]);
                internalPlayer.field_71071_by.field_70462_a[i] = null;
            }
        }

        for (ItemStack output : outputs) {
            output.field_77994_a -= Transactor.getTransactorFor(invOutput, EnumFacing.UP).add(output, true).field_77994_a;

            if (output.field_77994_a > 0) {
                output.field_77994_a -= Utils.addToRandomInventoryAround(field_145850_b, func_174877_v(), output);
            }

            if (output.field_77994_a > 0) {
                InvUtils.dropItems(field_145850_b, output, func_174877_v().func_177984_a());
            }
        }
    }

    private void searchNeighborsForIngredients() {
        for (IInvSlot slot : InventoryIterator.getIterable(craftingSlots, EnumFacing.UP)) {
            ItemStack ingred = slot.getStackInSlot();

            if (ingred == null) {
                continue;
            }

            IStackFilter filter = new CraftingFilter(ingred);

            if (InvUtils.countItems(invInput, EnumFacing.UP, filter) < InvUtils.countItems(craftingSlots, EnumFacing.UP, filter)) {
                for (EnumFacing side : SEARCH_SIDES) {
                    TileEntity tile = getTile(side);

                    if (tile instanceof IInventory) {
                        IInventory inv = InvUtils.getInventory((IInventory) tile);
                        ItemStack result = InvUtils.moveOneItem(inv, side.func_176734_d(), invInput, side, filter);

                        if (result != null) {
                            return;
                        }
                    }
                }
            }
        }
    }

    @Override
    public void updateCraftingMatrix(int slot, ItemStack stack) {
        craftingSlots.func_70299_a(slot, stack);
        updateRecipe();

        if (field_145850_b.field_72995_K) {
            PacketSlotChange packet = new PacketSlotChange(this, slot, stack);
            BuildCraftSilicon.instance.sendToServer(packet);
        }
    }

    private void updateRecipe() {
        if (internalInventoryCrafting == null) {
            return;
        }

        internalInventoryCrafting.recipeUpdate(true);

        if (this.currentRecipe == null || !this.currentRecipe.func_77569_a(internalInventoryCrafting, field_145850_b)) {
            currentRecipe = CraftingUtils.findMatchingRecipe(internalInventoryCrafting, field_145850_b);
        }

        internalInventoryCrafting.recipeUpdate(false);
        func_70296_d();
    }

    private void updateRecipeOutputDisplay() {
        if (internalInventoryCrafting == null || currentRecipe == null) {
            craftResult.func_70299_a(0, null);
            return;
        }

        ItemStack resultStack = getRecipeOutput();

        if (resultStack == null) {
            internalInventoryCrafting.recipeUpdate(true);
            resultStack = getRecipeOutput();
            internalInventoryCrafting.recipeUpdate(false);
        }

        craftResult.func_70299_a(0, resultStack);
        func_70296_d();
    }

    private ItemStack getRecipeOutput() {
        if (internalInventoryCrafting == null || currentRecipe == null) {
            return null;
        } else {
            return currentRecipe.func_77572_b(internalInventoryCrafting);
        }
    }

    public IInventory getCraftingSlots() {
        return craftingSlots;
    }

    public IInventory getOutputSlot() {
        return craftResult;
    }

    @Override
    public boolean canCraft() {
        return craftable && !justCrafted;
    }

    @Override
    public boolean hasWork() {
        return requiresLaserEnergy();
    }

    @Override
    public int[] func_180463_a(EnumFacing side) {
        return SLOTS;
    }

    @Override
    public boolean func_180462_a(int slot, ItemStack stack, EnumFacing side) {
        return func_94041_b(slot, stack);
    }

    @Override
    public boolean func_180461_b(int slot, ItemStack stack, EnumFacing side) {
        return slot >= 15;
    }

    @Override
    public boolean func_94041_b(int slot, ItemStack stack) {
        return slot < 15;
    }
}
