/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.silicon;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;

import net.minecraft.inventory.IInventory;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.nbt.NBTTagString;

import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fml.relauncher.Side;

import buildcraft.BuildCraftCore;
import buildcraft.api.recipes.CraftingResult;
import buildcraft.api.recipes.IFlexibleCrafter;
import buildcraft.api.recipes.IFlexibleRecipe;
import buildcraft.api.recipes.IFlexibleRecipeViewable;
import buildcraft.core.lib.network.command.CommandWriter;
import buildcraft.core.lib.network.command.ICommandReceiver;
import buildcraft.core.lib.network.command.PacketCommand;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.BCStringUtils;
import buildcraft.core.recipes.AssemblyRecipeManager;

import io.netty.buffer.ByteBuf;

public class TileAssemblyTable extends TileLaserTableBase implements IInventory, IFlexibleCrafter, ICommandReceiver {
    public String currentRecipeId = "";
    public IFlexibleRecipe<ItemStack> currentRecipe;
    public HashMap<String, CraftingResult<ItemStack>> plannedOutputIcons = new HashMap<>();
    private HashSet<String> plannedOutput = new HashSet<>();
    private boolean queuedNetworkUpdate = false;

    public List<CraftingResult<ItemStack>> getPotentialOutputs() {
        List<CraftingResult<ItemStack>> result = new LinkedList<>();

        for (IFlexibleRecipe<ItemStack> recipe : AssemblyRecipeManager.INSTANCE.getRecipes()) {
            CraftingResult<ItemStack> r = recipe.craft(this, true);

            if (r != null) {
                result.add(r);
            }
        }

        return result;
    }

    private void queueNetworkUpdate() {
        queuedNetworkUpdate = true;
    }

    @Override
    public void func_73660_a() { // WARNING: run only server-side, see canUpdate()
        super.func_73660_a();

        if (field_145850_b.field_72995_K) {
            return;
        }

        if (queuedNetworkUpdate) {
            sendNetworkUpdate();
            queuedNetworkUpdate = false;
        }

        if (currentRecipe == null) {
            return;
        }

        if (!currentRecipe.canBeCrafted(this)) {
            setNextCurrentRecipe();

            if (currentRecipe == null) {
                return;
            }
        }

        if (getEnergy() >= currentRecipe.craft(this, true).energyCost) {
            if (currentRecipe.canBeCrafted(this)) {
                CraftingResult<ItemStack> result = currentRecipe.craft(this, false);
                setEnergy(Math.max(0, getEnergy() - result.energyCost));
                outputStack(result.crafted.copy(), true);

                setNextCurrentRecipe();
            }
        }
    }

    /* IINVENTORY */
    @Override
    public int func_70302_i_() {
        return 12;
    }

    @Override
    public void func_70299_a(int slot, ItemStack stack) {
        super.func_70299_a(slot, stack);

        if (currentRecipe == null) {
            setNextCurrentRecipe();
        }
    }

    @Override
    public String getInventoryName() {
        return BCStringUtils.localize("tile.assemblyTableBlock.name");
    }

    @Override
    public void readData(ByteBuf stream) {
        super.readData(stream);
        currentRecipeId = NetworkUtils.readUTF(stream);
        plannedOutput.clear();
        int size = stream.readUnsignedByte();
        for (int i = 0; i < size; i++) {
            plannedOutput.add(NetworkUtils.readUTF(stream));
        }

        // Update plannedOutputIcons
        generatePlannedOutputIcons();

        currentRecipe = AssemblyRecipeManager.INSTANCE.getRecipe(currentRecipeId);
    }

    private void generatePlannedOutputIcons() {
        for (String s : plannedOutput) {
            IFlexibleRecipe<ItemStack> recipe = AssemblyRecipeManager.INSTANCE.getRecipe(s);
            if (recipe != null) {
                CraftingResult<ItemStack> result = recipe.craft(this, true);
                if (result != null && result.usedItems != null && result.usedItems.size() > 0) {
                    plannedOutputIcons.put(s, result);
                } else if (recipe instanceof IFlexibleRecipeViewable) {
                    // !! HACK !! TODO !! HACK !!
                    Object out = ((IFlexibleRecipeViewable) recipe).getOutput();
                    if (out instanceof ItemStack) {
                        result = new CraftingResult<>();
                        result.crafted = (ItemStack) out;
                        result.recipe = recipe;
                        plannedOutputIcons.put(s, result);
                    }
                }
            } else {
                plannedOutput.remove(s);
            }
        }

        for (String s : plannedOutputIcons.keySet().toArray(new String[plannedOutputIcons.size()])) {
            if (!(plannedOutput.contains(s))) {
                plannedOutputIcons.remove(s);
            }
        }
    }

    @Override
    public void writeData(ByteBuf stream) {
        super.writeData(stream);
        NetworkUtils.writeUTF(stream, currentRecipeId);
        stream.writeByte(plannedOutput.size());
        for (String s : plannedOutput) {
            NetworkUtils.writeUTF(stream, s);
        }
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);

        NBTTagList list = nbt.func_150295_c("plannedIds", Constants.NBT.TAG_STRING);

        for (int i = 0; i < list.func_74745_c(); ++i) {
            IFlexibleRecipe<ItemStack> recipe = AssemblyRecipeManager.INSTANCE.getRecipe(list.func_150307_f(i));

            if (recipe != null) {
                plannedOutput.add(recipe.getId());
            }
        }

        if (nbt.func_74764_b("recipeId")) {
            IFlexibleRecipe<ItemStack> recipe = AssemblyRecipeManager.INSTANCE.getRecipe(nbt.func_74779_i("recipeId"));

            if (recipe != null) {
                setCurrentRecipe(recipe);
            }
        }
    }

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);

        NBTTagList list = new NBTTagList();

        for (String recipe : plannedOutput) {
            list.func_74742_a(new NBTTagString(recipe));
        }

        nbt.func_74782_a("plannedIds", list);

        if (currentRecipe != null) {
            nbt.func_74778_a("recipeId", currentRecipe.getId());
        }
    }

    public boolean isPlanned(IFlexibleRecipe<ItemStack> recipe) {
        if (recipe == null) {
            return false;
        }

        return plannedOutput.contains(recipe.getId());
    }

    public boolean isAssembling(IFlexibleRecipe<ItemStack> recipe) {
        return recipe != null && recipe == currentRecipe;
    }

    private void setCurrentRecipe(IFlexibleRecipe<ItemStack> recipe) {
        currentRecipe = recipe;

        if (recipe != null) {
            currentRecipeId = recipe.getId();
        } else {
            currentRecipeId = "";
        }

        // Update plannedOutputIcons
        generatePlannedOutputIcons();

        if (field_145850_b != null && !field_145850_b.field_72995_K) {
            queueNetworkUpdate();
        }
    }

    @Override
    public int getRequiredEnergy() {
        if (currentRecipe != null) {
            CraftingResult<ItemStack> result = currentRecipe.craft(this, true);

            if (result != null) {
                return result.energyCost;
            } else {
                return 0;
            }
        } else {
            return 0;
        }
    }

    public void planOutput(IFlexibleRecipe<ItemStack> recipe) {
        if (recipe != null && !isPlanned(recipe)) {
            plannedOutput.add(recipe.getId());

            if (!isAssembling(currentRecipe) || !isPlanned(currentRecipe)) {
                setCurrentRecipe(recipe);
            }

            queueNetworkUpdate();
        }
    }

    public void cancelPlanOutput(IFlexibleRecipe<ItemStack> recipe) {
        if (isAssembling(recipe)) {
            setCurrentRecipe(null);
        }

        plannedOutput.remove(recipe.getId());

        if (!plannedOutput.isEmpty()) {
            setCurrentRecipe(AssemblyRecipeManager.INSTANCE.getRecipe(plannedOutput.iterator().next()));
        }

        queueNetworkUpdate();
    }

    public void setNextCurrentRecipe() {
        boolean takeNext = false;

        for (String recipeId : plannedOutput) {
            IFlexibleRecipe<ItemStack> recipe = AssemblyRecipeManager.INSTANCE.getRecipe(recipeId);

            if (recipe == null) {
                continue;
            }

            if (recipe == currentRecipe) {
                takeNext = true;
            } else if (takeNext && recipe.canBeCrafted(this)) {
                setCurrentRecipe(recipe);
                return;
            }
        }

        for (String recipeId : plannedOutput) {
            IFlexibleRecipe<ItemStack> recipe = AssemblyRecipeManager.INSTANCE.getRecipe(recipeId);

            if (recipe == null) {
                continue;
            }

            if (recipe.canBeCrafted(this)) {
                setCurrentRecipe(recipe);
                return;
            }
        }

        setCurrentRecipe(null);
    }

    public void rpcSelectRecipe(final String id, final boolean select) {
        BuildCraftCore.instance.sendToServer(new PacketCommand(this, "select", new CommandWriter() {
            @Override
            public void write(ByteBuf data) {
                NetworkUtils.writeUTF(data, id);
                data.writeBoolean(select);
            }
        }));
    }

    @Override
    public void receiveCommand(String command, Side side, Object sender, ByteBuf stream) {
        if (side.isServer() && "select".equals(command)) {
            String id = NetworkUtils.readUTF(stream);
            boolean select = stream.readBoolean();

            IFlexibleRecipe<ItemStack> recipe = AssemblyRecipeManager.INSTANCE.getRecipe(id);

            if (recipe != null) {
                if (select) {
                    planOutput(recipe);
                } else {
                    cancelPlanOutput(recipe);
                }
            }
        }
    }

    @Override
    public boolean hasWork() {
        return currentRecipe != null;
    }

    @Override
    public boolean canCraft() {
        return hasWork();
    }

    @Override
    public boolean func_94041_b(int slot, ItemStack stack) {
        return true;
    }

    @Override
    public int getCraftingItemStackSize() {
        return func_70302_i_();
    }

    @Override
    public ItemStack getCraftingItemStack(int slotid) {
        return func_70301_a(slotid);
    }

    @Override
    public ItemStack decrCraftingItemStack(int slotid, int val) {
        return func_70298_a(slotid, val);
    }

    @Override
    public FluidStack getCraftingFluidStack(int tankid) {
        return null;
    }

    @Override
    public FluidStack decrCraftingFluidStack(int tankid, int val) {
        return null;
    }

    @Override
    public int getCraftingFluidStackSize() {
        return 0;
    }
}
