/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.transport;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import com.google.common.base.Strings;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import net.minecraft.block.Block;
import net.minecraft.block.BlockQuartz;
import net.minecraft.block.BlockStainedGlass;
import net.minecraft.block.state.IBlockState;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.init.Blocks;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ResourceLocation;
import net.minecraft.world.World;
import net.minecraftforge.common.ForgeHooks;
import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.common.Loader;
import net.minecraftforge.fml.common.registry.GameRegistry;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import buildcraft.BuildCraftCore;
import buildcraft.BuildCraftTransport;
import buildcraft.api.core.BCLog;
import buildcraft.api.core.JavaTools;
import buildcraft.api.facades.FacadeType;
import buildcraft.api.facades.IFacadeItem;
import buildcraft.api.recipes.BuildcraftRecipeRegistry;
import buildcraft.api.transport.IPipe;
import buildcraft.api.transport.PipeWire;
import buildcraft.api.transport.pluggable.IPipePluggableItem;
import buildcraft.api.transport.pluggable.PipePluggable;
import buildcraft.core.BCCreativeTab;
import buildcraft.core.lib.items.ItemBuildCraft;
import buildcraft.core.lib.utils.BCStringUtils;
import buildcraft.core.lib.utils.Utils;
import buildcraft.core.proxy.CoreProxy;

public class ItemFacade extends ItemBuildCraft implements IFacadeItem, IPipePluggableItem {
    public static class FacadeState {
        public final IBlockState state;
        public final boolean transparent;
        public final boolean hollow;
        public final PipeWire wire;

        public FacadeState(IBlockState state, PipeWire wire) {
            this.state = state;
            this.wire = wire;
            this.transparent = false;
            this.hollow = false;
        }

        public FacadeState(IBlockState state, PipeWire wire, boolean hollow) {
            this.state = state;
            this.wire = wire;
            this.transparent = false;
            this.hollow = hollow;
        }

        public FacadeState(NBTTagCompound nbt) {
            String key = nbt.func_74779_i("block");
            Block block = (Block) Block.field_149771_c.func_82594_a(new ResourceLocation(key));
            if (block == null) throw new NullPointerException("Could not load a block from the key \"" + key + "\"");
            int metadata = nbt.func_74771_c("metadata");
            state = block.func_176203_a(metadata);
            this.wire = nbt.func_74764_b("wire") ? PipeWire.fromOrdinal(nbt.func_74771_c("wire")) : null;
            this.transparent = nbt.func_74764_b("transparent") && nbt.func_74767_n("transparent");
            this.hollow = nbt.func_74764_b("hollow") && nbt.func_74767_n("hollow");
        }

        private FacadeState(PipeWire wire) {
            state = Blocks.field_150357_h.func_176223_P();
            this.wire = wire;
            this.transparent = true;
            this.hollow = false;
        }

        public static FacadeState create(IBlockState state) {
            return create(state, null);
        }

        public static FacadeState create(IBlockState state, PipeWire wire) {
            return new FacadeState(state, wire);
        }

        public static FacadeState createTransparent(PipeWire wire) {
            return new FacadeState(wire);
        }

        public void writeToNBT(NBTTagCompound nbt) {
            if (state != null) {
                nbt.func_74778_a("block", Utils.getNameForBlock(state.func_177230_c()));
                nbt.func_74774_a("metadata", (byte) state.func_177230_c().func_176201_c(state));
            }
            if (wire != null) {
                nbt.func_74774_a("wire", (byte) wire.ordinal());
            }
            nbt.func_74757_a("transparent", transparent);
            nbt.func_74757_a("hollow", hollow);
        }

        public static NBTTagList writeArray(FacadeState[] states) {
            if (states == null) {
                return null;
            }
            NBTTagList list = new NBTTagList();
            for (FacadeState state : states) {
                NBTTagCompound stateNBT = new NBTTagCompound();
                state.writeToNBT(stateNBT);
                list.func_74742_a(stateNBT);
            }
            return list;
        }

        public static FacadeState[] readArray(NBTTagList list) {
            if (list == null) {
                return new FacadeState[0];
            }
            final int length = list.func_74745_c();
            FacadeState[] states = new FacadeState[length];
            for (int i = 0; i < length; i++) {
                states[i] = new FacadeState(list.func_150305_b(i));
            }
            return states;
        }

        @Override
        public int hashCode() {
            return new HashCodeBuilder().append(hollow).append(transparent).append(state).append(wire).build();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj) return true;
            if (obj == null) return false;
            if (getClass() != obj.getClass()) return false;
            FacadeState other = (FacadeState) obj;
            if (hollow != other.hollow) return false;
            if (state == null) {
                if (other.state != null) return false;
            } else if (!state.equals(other.state)) return false;
            if (transparent != other.transparent) return false;
            if (wire != other.wire) return false;
            return true;
        }
    }

    public static final ArrayList<ItemStack> allFacades = new ArrayList<>();
    public static final ArrayList<ItemStack> allHollowFacades = new ArrayList<>();
    public static final ArrayList<String> allFacadeIDs = new ArrayList<>();
    public static final ArrayList<String> blacklistedFacades = new ArrayList<>();

    private static final Block NULL_BLOCK = null;
    private static final ItemStack NO_MATCH = new ItemStack(NULL_BLOCK, 0, 0);

    private static final Block[] PREVIEW_FACADES = new Block[] { Blocks.field_150344_f, Blocks.field_150417_aV, Blocks.field_150359_w };
    private static int RANDOM_FACADE_ID = -1;

    public ItemFacade() {
        super(BuildCraftTransport.showAllFacadesCreative ? BCCreativeTab.get("facades") : BCCreativeTab.get("main"));

        func_77627_a(true);
        func_77656_e(0);
    }

    @Override
    public String func_77653_i(ItemStack itemstack) {
        switch (getFacadeType(itemstack)) {
            case Basic:
                FacadeState[] states = getFacadeStates(itemstack);
                String displayName = states.length > 0 ? getFacadeStateDisplayName(states[0]) : "CORRUPT";
                return super.func_77653_i(itemstack) + ": " + displayName;
            case Phased:
                return BCStringUtils.localize("item.FacadePhased.name");
            default:
                return "";
        }
    }

    @Override
    public String func_77667_c(ItemStack itemstack) {
        return "item.Facade";
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    public void func_77624_a(ItemStack stack, EntityPlayer player, List list, boolean debug) {
        for (FacadeState state : getFacadeStates(stack)) {
            if (state != null && !state.transparent && state.state != null && Item.func_150898_a(state.state.func_177230_c()) != null) {
                Item.func_150898_a(state.state.func_177230_c()).func_77624_a(new ItemStack(state.state.func_177230_c(), 1, state.state.func_177230_c()
                        .func_176201_c(state.state)), player, list, debug);
            }
        }
        if (getFacadeType(stack) == FacadeType.Phased) {
            String stateString = BCStringUtils.localize("item.FacadePhased.state");
            FacadeState defaultState = null;
            for (FacadeState state : getFacadeStates(stack)) {
                if (state.wire == null) {
                    defaultState = state;
                    continue;
                }
                list.add(String.format(stateString, state.wire.getColor(), getFacadeStateDisplayName(state)));
            }
            if (defaultState != null) {
                list.add(1, String.format(BCStringUtils.localize("item.FacadePhased.state_default"), getFacadeStateDisplayName(defaultState)));
            }
        }
    }

    public static String getFacadeStateDisplayName(FacadeState state) {
        if (state.state == null) {
            return BCStringUtils.localize("item.FacadePhased.state_transparent");
        }
        // if (state.state.getBlock().getRenderType() == 31) {
        // TODO: Find out what render type is 31... and what this now means
        // meta &= 0x3;
        // } else if (state.block.getRenderType() == 39 && meta > 2) {
        // meta = 2;
        // }
        String s = CoreProxy.proxy.getItemDisplayName(new ItemStack(state.state.func_177230_c(), 1, state.state.func_177230_c().func_176201_c(state.state)));
        if (state.hollow) {
            s += " (" + BCStringUtils.localize("item.Facade.state_hollow") + ")";
        }
        return s;
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    @SideOnly(Side.CLIENT)
    public void func_150895_a(Item item, CreativeTabs par2CreativeTabs, List itemList) {
        if (BuildCraftTransport.showAllFacadesCreative) {
            for (ItemStack stack : allFacades) {
                itemList.add(stack);
            }
            for (ItemStack stack : allHollowFacades) {
                itemList.add(stack);
            }
        } else {
            List<ItemStack> hollowFacades = new ArrayList<>();
            for (Block b : PREVIEW_FACADES) {
                if (isBlockValidForFacade(b.func_176203_a(0)) && !isBlockBlacklisted(b)) {
                    ItemStack facade = getFacadeForBlock(b.func_176203_a(0));
                    itemList.add(facade);
                    FacadeState state = getFacadeStates(facade)[0];
                    hollowFacades.add(getFacade(new FacadeState(state.state, state.wire, true)));
                }
            }
            if (RANDOM_FACADE_ID < 0) {
                RANDOM_FACADE_ID = BuildCraftCore.random.nextInt(allFacades.size());
            }
            itemList.add(allFacades.get(RANDOM_FACADE_ID));
            itemList.addAll(hollowFacades);
            itemList.add(allHollowFacades.get(RANDOM_FACADE_ID));
        }
    }

    public void initialize() {
        for (Object o : Block.field_149771_c) {
            Block b = (Block) o;

            Item item = Item.func_150898_a(b);

            if (item == null) {
                continue;
            }

            if (isBlockBlacklisted(b)) {
                continue;
            }

            registerValidFacades(b, item);
        }
    }

    private void registerValidFacades(Block block, Item item) {
        ArrayList<ItemStack> stacks = new ArrayList<>(16);
        HashSet<IBlockState> states = new HashSet<>();
        try {
            if (FMLCommonHandler.instance().getSide() == Side.CLIENT) {
                for (CreativeTabs ct : item.getCreativeTabs()) {
                    block.func_149666_a(item, ct, stacks);
                }
            } else {
                for (int i = 0; i < 16; i++) {
                    try {
                        IBlockState state = block.func_176203_a(i);
                        if (state != null && !states.contains(state)) {
                            states.add(state);
                            stacks.add(new ItemStack(item, 1, i));
                        }
                    } catch (Exception e) {

                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        for (ItemStack stack : stacks) {
            try {
                if (block.hasTileEntity(block.func_176223_P())) continue;

                // Check if all of these functions work correctly.
                // If an exception is fired, or null is returned, this generally means that
                // this block is invalid.
                try {
                    if (stack.func_82833_r() == null || Strings.isNullOrEmpty(stack.func_77977_a())) continue;
                } catch (Throwable t) {
                    continue;
                }

                addFacade(stack);
            } catch (IndexOutOfBoundsException e) {

            } catch (Throwable t) {
                t.printStackTrace();
            }
        }
    }

    private static boolean isBlockBlacklisted(Block block) {
        String blockName = Utils.getNameForBlock(block);

        if (blockName == null) return true;

        // Blocks blacklisted by mods should always be treated as blacklisted
        for (String blacklistedBlock : blacklistedFacades)
            if (blockName.equals(blacklistedBlock)) return true;

        // Blocks blacklisted by config should depend on the config settings
        for (String blacklistedBlock : BuildCraftTransport.facadeBlacklist) {
            if (blockName.equals(JavaTools.stripSurroundingQuotes(blacklistedBlock))) return true
                ^ BuildCraftTransport.facadeTreatBlacklistAsWhitelist;
        }

        return false ^ BuildCraftTransport.facadeTreatBlacklistAsWhitelist;
    }

    public static boolean isTransparentFacade(IBlockState state) {
        Block block = state.func_177230_c();
        return !block.func_176214_u() && !block.func_149662_c();
    }

    private static boolean isBlockValidForFacade(IBlockState state) {
        try {
            Block block = state.func_177230_c();

            // HACK: Improved facade validation will only come in 8.0, so for now we simply
            // whitelist stained glass to keep compatibility.
            if (block instanceof BlockStainedGlass) {
                return true;
            }

            if (!block.func_149730_j() || !block.func_149686_d() || block.hasTileEntity(state)) return false;
            if (block.func_149704_x() != 0.0 || block.func_149665_z() != 0.0 || block.func_149706_B() != 0.0) return false;
            if (block.func_149753_y() != 1.0 || block.func_149669_A() != 1.0 || block.func_149693_C() != 1.0) return false;

            return true;
        } catch (Throwable ignored) {
            return false;
        }
    }

    public static FacadeState[] getFacadeStates(ItemStack stack) {
        if (!stack.func_77942_o()) return new FacadeState[0];
        NBTTagCompound nbt = stack.func_77978_p();
        nbt = migrate(stack, nbt);
        if (!nbt.func_74764_b("states")) return new FacadeState[0];
        return FacadeState.readArray(nbt.func_150295_c("states", Constants.NBT.TAG_COMPOUND));
    }

    private static NBTTagCompound migrate(ItemStack stack, NBTTagCompound nbt) {
        Block block = null, blockAlt = null;
        int metadata = 0, metadataAlt;
        PipeWire wire = null;
        if (nbt.func_74764_b("id")) block = Block.field_149771_c.func_148754_a(nbt.func_74762_e("id"));
        else if (nbt.func_74764_b("name")) block = Block.field_149771_c.func_82594_a(new ResourceLocation(nbt.func_74779_i("name")));
        if (nbt.func_74764_b("name_alt")) blockAlt = Block.field_149771_c.func_82594_a(new ResourceLocation(nbt.func_74779_i("name_alt")));
        if (nbt.func_74764_b("meta")) metadata = nbt.func_74762_e("meta");
        if (nbt.func_74764_b("meta_alt")) metadataAlt = nbt.func_74762_e("meta_alt");
        else metadataAlt = stack.func_77952_i() & 0x0000F;
        if (nbt.func_74764_b("wire")) wire = PipeWire.fromOrdinal(nbt.func_74762_e("wire"));
        if (block != null) {
            FacadeState[] states;
            FacadeState mainState = FacadeState.create(block.func_176203_a(metadata));
            if (blockAlt != null && wire != null) {
                FacadeState altState = FacadeState.create(blockAlt.func_176203_a(metadataAlt), wire);
                states = new FacadeState[] { mainState, altState };
            } else {
                states = new FacadeState[] { mainState };
            }
            NBTTagCompound newNbt = getFacade(states).func_77978_p();
            stack.func_77982_d(newNbt);
            return newNbt;
        }
        return nbt;
    }

    @Override
    public IBlockState[] getBlockStatesForFacade(ItemStack stack) {
        FacadeState[] states = getFacadeStates(stack);
        IBlockState[] blocks = new IBlockState[states.length];
        for (int i = 0; i < states.length; i++) {
            blocks[i] = states[i].state;
        }
        return blocks;
    }

    // GETTERS FOR FACADE DATA
    @Override
    public FacadeType getFacadeType(ItemStack stack) {
        if (!stack.func_77942_o()) {
            return FacadeType.Basic;
        }
        NBTTagCompound nbt = stack.func_77978_p();
        if (!nbt.func_74764_b("type")) {
            return FacadeType.Basic;
        }
        return FacadeType.fromOrdinal(nbt.func_74762_e("type"));
    }

    @Override
    public boolean doesSneakBypassUse(World world, BlockPos pos, EntityPlayer player) {
        // Simply send shift click to the pipe / mod block.
        return true;
    }

    public void addFacade(ItemStack itemStack) {
        if (itemStack.field_77994_a == 0) itemStack.field_77994_a = 1;

        Block block = Block.func_149634_a(itemStack.func_77973_b());
        if (block == null) return;

        IBlockState bstate = block.func_176203_a(itemStack.func_77952_i());
        if (!isBlockValidForFacade(bstate)) {
            return;
        }

        String recipeId = "buildcraft:facade{" + Utils.getNameForBlock(block) + "#" + itemStack.func_77952_i() + "}";

        ItemStack facade = getFacadeForBlock(bstate);
        ItemStack realStack = itemStack.func_77946_l();
        realStack.func_77964_b(block.func_180651_a(bstate));

        if (!allFacadeIDs.contains(recipeId)) {
            allFacadeIDs.add(recipeId);
            allFacades.add(facade);

            ItemStack facade6 = facade.func_77946_l();
            facade6.field_77994_a = 6;

            FacadeState state = getFacadeStates(facade6)[0];
            ItemStack facadeHollow = getFacade(new FacadeState(state.state, state.wire, true));

            allHollowFacades.add(facadeHollow);

            ItemStack facade6Hollow = facadeHollow.func_77946_l();
            facade6Hollow.field_77994_a = 6;

            // 3 Structurepipes + this block makes 6 facades
            if (Loader.isModLoaded("BuildCraft|Silicon") && !BuildCraftTransport.facadeForceNonLaserRecipe) {
                BuildcraftRecipeRegistry.assemblyTable.addRecipe(recipeId, 8000, facade6, new ItemStack(BuildCraftTransport.pipeStructureCobblestone,
                        3), itemStack);

                BuildcraftRecipeRegistry.assemblyTable.addRecipe(recipeId + ":hollow", 8000, facade6Hollow, new ItemStack(
                        BuildCraftTransport.pipeStructureCobblestone, 3), itemStack);

                BuildcraftRecipeRegistry.assemblyTable.addRecipe(recipeId + ":toHollow", 160, facadeHollow, facade);
                BuildcraftRecipeRegistry.assemblyTable.addRecipe(recipeId + ":fromHollow", 160, facade, facadeHollow);
            } else {
                GameRegistry.addShapedRecipe(facade6, "t ", "ts", "t ", 't', realStack, 's', BuildCraftTransport.pipeStructureCobblestone);
                GameRegistry.addShapedRecipe(facade6Hollow, "t ", " s", "t ", 't', realStack, 's', BuildCraftTransport.pipeStructureCobblestone);
            }
        }
    }

    public static void blacklistFacade(String blockName) {
        if (!blacklistedFacades.contains(blockName)) blacklistedFacades.add(blockName);
    }

    public class FacadeRecipe implements IRecipe {

        @Override
        public boolean func_77569_a(InventoryCrafting inventorycrafting, World world) {
            Object[] facade = getFacadeBlockFromCraftingGrid(inventorycrafting);

            return facade != null && facade[0] != null && ((IBlockState[]) facade[0]).length == 1;
        }

        @Override
        public ItemStack func_77572_b(InventoryCrafting inventorycrafting) {
            Object[] facade = getFacadeBlockFromCraftingGrid(inventorycrafting);
            if (facade == null || ((IBlockState[]) facade[0]).length != 1) return null;

            IBlockState block = ((IBlockState[]) facade[0])[0];
            ItemStack originalFacade = (ItemStack) facade[1];

            if (block == null) return null;

            return getNextFacadeItemStack(block, originalFacade);
        }

        private Object[] getFacadeBlockFromCraftingGrid(InventoryCrafting inventorycrafting) {
            ItemStack slotmatch = null;
            int countOfItems = 0;
            for (int i = 0; i < inventorycrafting.func_70302_i_(); i++) {
                ItemStack slot = inventorycrafting.func_70301_a(i);

                if (slot != null && slot.func_77973_b() == ItemFacade.this && slotmatch == null) {
                    slotmatch = slot;
                    countOfItems++;
                } else if (slot != null) slotmatch = NO_MATCH;

                if (countOfItems > 1) return null;
            }

            if (slotmatch != null && slotmatch != NO_MATCH) return new Object[] { getBlockStatesForFacade(slotmatch), slotmatch };

            return null;
        }

        private ItemStack getNextFacadeItemStack(IBlockState state, ItemStack originalFacade) {
            // TODO: Add an API for me!
            IBlockState newState = state;
            if (newState.func_177227_a().contains(BlockQuartz.field_176335_a)) {
                BlockQuartz.EnumType type = newState.func_177229_b(BlockQuartz.field_176335_a);
                if ("lines".equals(type.toString())) {
                    newState = newState.func_177226_a(BlockQuartz.field_176335_a, BlockQuartz.EnumType.func_176794_a(((type.func_176796_a() - 1) % 3) + 2));
                }
            } else {
                for (net.minecraft.block.properties.IProperty<?> prop : state.func_177228_b().keySet()) {
                    if (prop.func_177701_a().equals("axis")) {
                        newState = newState.func_177231_a(prop);
                    }
                }
            }

            return getFacadeForBlock(newState);
        }

        @Override
        public int func_77570_a() {
            return 1;
        }

        @Override
        public ItemStack func_77571_b() {
            return null;
        }

        @Override
        public ItemStack[] func_179532_b(InventoryCrafting inv) {
            ItemStack[] itemStack = new ItemStack[inv.func_70302_i_()];

            for (int i = 0; i < itemStack.length; ++i) {
                ItemStack itemstack = inv.func_70301_a(i);
                itemStack[i] = ForgeHooks.getContainerItem(itemstack);
            }

            return itemStack;
        }
    }

    @Override
    public ItemStack getFacadeForBlock(IBlockState state) {
        return getFacade(FacadeState.create(state));
    }

    public static ItemStack getAdvancedFacade(PipeWire wire, IBlockState state, IBlockState stateAlt) {
        return getFacade(FacadeState.create(state), FacadeState.create(stateAlt, wire));
    }

    public static ItemStack getFacade(FacadeState... states) {
        if (states == null || states.length == 0) return null;
        final boolean basic = states.length == 1 && states[0].wire == null;

        ItemStack stack = new ItemStack(BuildCraftTransport.facadeItem, 1, 0);

        NBTTagCompound nbt = new NBTTagCompound();
        nbt.func_74774_a("type", (byte) (basic ? FacadeType.Basic : FacadeType.Phased).ordinal());
        nbt.func_74782_a("states", FacadeState.writeArray(states));

        stack.func_77982_d(nbt);
        return stack;
    }

    @Override
    public PipePluggable createPipePluggable(IPipe pipe, EnumFacing side, ItemStack stack) {
        return new FacadePluggable(getFacadeStates(stack));
    }
}
