/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.transport;

import java.util.List;

import com.google.common.base.Throwables;

import org.apache.logging.log4j.Level;

import net.minecraft.block.Block;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.EnumDyeColor;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.network.NetworkManager;
import net.minecraft.network.play.server.S35PacketUpdateTileEntity;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.*;
import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.world.World;

import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fluids.Fluid;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidTankInfo;
import net.minecraftforge.fluids.IFluidHandler;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import cofh.api.energy.IEnergyHandler;
import cofh.api.energy.IEnergyProvider;
import cofh.api.energy.IEnergyReceiver;

import buildcraft.BuildCraftCore;
import buildcraft.BuildCraftTransport;
import buildcraft.api.core.BCLog;
import buildcraft.api.core.EnumPipePart;
import buildcraft.api.core.IIconProvider;
import buildcraft.api.core.ISerializable;
import buildcraft.api.gates.IGateExpansion;
import buildcraft.api.power.IRedstoneEngineReceiver;
import buildcraft.api.tiles.IDebuggable;
import buildcraft.api.transport.*;
import buildcraft.api.transport.pluggable.IFacadePluggable;
import buildcraft.api.transport.pluggable.PipePluggable;
import buildcraft.core.DefaultProps;
import buildcraft.core.internal.IDropControlInventory;
import buildcraft.core.lib.ITileBufferHolder;
import buildcraft.core.lib.TileBuffer;
import buildcraft.core.lib.network.IGuiReturnHandler;
import buildcraft.core.lib.network.ISyncedTile;
import buildcraft.core.lib.network.PacketTileState;
import buildcraft.core.lib.network.base.Packet;
import buildcraft.core.lib.utils.NetworkUtils;
import buildcraft.core.lib.utils.Utils;
import buildcraft.transport.ItemFacade.FacadeState;
import buildcraft.transport.gates.GateFactory;
import buildcraft.transport.gates.GatePluggable;
import buildcraft.transport.pluggable.PlugPluggable;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;

public class TileGenericPipe extends TileEntity implements IFluidHandler, IPipeTile, ITileBufferHolder, IDropControlInventory, ISyncedTile, ISolidSideTile, IGuiReturnHandler, IRedstoneEngineReceiver, IDebuggable, IPipeConnection, ITickable,
        IEnergyProvider {

    public boolean initialized = false;
    public final PipeRenderState renderState = new PipeRenderState();
    public final PipePluggableState pluggableState = new PipePluggableState();
    public final CoreState coreState = new CoreState();
    public boolean[] pipeConnectionsBuffer = new boolean[6];

    public Pipe<?> pipe;
    public int redstoneInput;
    public int[] redstoneInputSide = new int[EnumFacing.field_82609_l.length];

    protected boolean deletePipe = false;
    protected boolean sendClientUpdate = false;
    protected boolean blockNeighborChange = false;
    protected int blockNeighborChangedSides = 0;
    protected boolean refreshRenderState = false;
    protected boolean pipeBound = false;
    protected boolean resyncGateExpansions = false;
    protected boolean attachPluggables = false;
    protected SideProperties sideProperties = new SideProperties();

    private TileBuffer[] tileBuffer;
    private int glassColor = -1;

    public static class CoreState implements ISerializable, Comparable<CoreState> {
        public String pipeId = null;

        @Override
        public void writeData(ByteBuf data) {
            NetworkUtils.writeUTF(data, pipeId);
        }

        @Override
        public void readData(ByteBuf data) {
            pipeId = NetworkUtils.readUTF(data);
        }

        @Override
        public int compareTo(CoreState o) {
            return 0;
        }
    }

    public static class SideProperties {
        PipePluggable[] pluggables = new PipePluggable[EnumFacing.field_82609_l.length];

        public void writeToNBT(NBTTagCompound nbt) {
            for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
                PipePluggable pluggable = pluggables[i];
                final String key = "pluggable[" + i + "]";
                if (pluggable == null) {
                    nbt.func_82580_o(key);
                } else {
                    NBTTagCompound pluggableData = new NBTTagCompound();
                    pluggableData.func_74778_a("pluggableName", PipeManager.getPluggableName(pluggable.getClass()));
                    pluggable.writeToNBT(pluggableData);
                    nbt.func_74782_a(key, pluggableData);
                }
            }
        }

        public void readFromNBT(NBTTagCompound nbt) {
            for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
                final String key = "pluggable[" + i + "]";
                if (!nbt.func_74764_b(key)) {
                    continue;
                }
                try {
                    NBTTagCompound pluggableData = nbt.func_74775_l(key);
                    Class<?> pluggableClass = PipeManager.getPluggableByName(pluggableData.func_74779_i("pluggableName"));
                    if (pluggableClass != null) {
                        if (!PipePluggable.class.isAssignableFrom(pluggableClass)) {
                            BCLog.logger.warn("Wrong pluggable class: " + pluggableClass);
                            continue;
                        }
                        PipePluggable pluggable = (PipePluggable) pluggableClass.newInstance();
                        pluggable.readFromNBT(pluggableData);
                        pluggables[i] = pluggable;
                    }
                } catch (Exception e) {
                    BCLog.logger.warn("Failed to load side state");
                    e.printStackTrace();
                }
            }

            // Migration code
            for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
                PipePluggable pluggable = null;
                if (nbt.func_74764_b("facadeState[" + i + "]")) {
                    pluggable = new FacadePluggable(FacadeState.readArray(nbt.func_150295_c("facadeState[" + i + "]", Constants.NBT.TAG_COMPOUND)));
                } else {
                    // Migration support for 5.0.x and 6.0.x // no longer required
                    // if (nbt.hasKey("facadeBlocks[" + i + "]")) {
                    // // 5.0.x
                    // Block block = (Block) Block.blockRegistry.getObjectById(nbt.getInteger("facadeBlocks[" + i +
                    // "]"));
                    // int blockId = nbt.getInteger("facadeBlocks[" + i + "]");
                    //
                    // if (blockId != 0) {
                    // int metadata = nbt.getInteger("facadeMeta[" + i + "]");
                    // pluggable = new FacadePluggable(new FacadeState[] { FacadeState.create(state) });
                    // }
                    // } else if (nbt.hasKey("facadeBlocksStr[" + i + "][0]")) {
                    // // 6.0.x
                    // FacadeState mainState = FacadeState.create((Block)
                    // Block.blockRegistry.getObject(nbt.getString("facadeBlocksStr[" + i
                    // + "][0]")), nbt.getInteger("facadeMeta[" + i + "][0]"));
                    // if (nbt.hasKey("facadeBlocksStr[" + i + "][1]")) {
                    // FacadeState phasedState = FacadeState.create((Block)
                    // Block.blockRegistry.getObject(nbt.getString("facadeBlocksStr[" + i
                    // + "][1]")), nbt.getInteger("facadeMeta[" + i + "][1]"),
                    // PipeWire.fromOrdinal(nbt.getInteger("facadeWires[" + i
                    // + "]")));
                    // pluggable = new FacadePluggable(new FacadeState[] { mainState, phasedState });
                    // } else {
                    // pluggable = new FacadePluggable(new FacadeState[] { mainState });
                    // }
                    // }
                }

                if (nbt.func_74767_n("plug[" + i + "]")) {
                    pluggable = new PlugPluggable();
                }

                if (pluggable != null) {
                    pluggables[i] = pluggable;
                }
            }
        }

        public void rotateLeft() {
            PipePluggable[] newPluggables = new PipePluggable[EnumFacing.field_82609_l.length];
            for (EnumFacing dir : EnumFacing.field_82609_l) {
                EnumFacing rotated = dir.func_176740_k() == Axis.Y ? dir : dir.func_176746_e();
                newPluggables[rotated.ordinal()] = pluggables[dir.ordinal()];
            }
            pluggables = newPluggables;
        }

        public boolean dropItem(TileGenericPipe pipe, EnumFacing direction, EntityPlayer player) {
            boolean result = false;
            PipePluggable pluggable = pluggables[direction.ordinal()];
            if (pluggable != null) {
                pluggable.onDetachedPipe(pipe, direction);
                if (!pipe.func_145831_w().field_72995_K) {
                    ItemStack[] stacks = pluggable.getDropItems(pipe);
                    if (stacks != null) {
                        for (ItemStack stack : stacks) {
                            Utils.dropTryIntoPlayerInventory(pipe.field_145850_b, pipe.field_174879_c, stack, player);
                        }
                    }
                }
                result = true;
            }
            return result;
        }

        public void invalidate() {
            for (PipePluggable p : pluggables) {
                if (p != null) {
                    p.invalidate();
                }
            }
        }

        public void validate(TileGenericPipe pipe) {
            for (EnumFacing d : EnumFacing.field_82609_l) {
                PipePluggable p = pluggables[d.ordinal()];

                if (p != null) {
                    p.validate(pipe, d);
                }
            }
        }
    }

    public TileGenericPipe() {}

    @Override
    public void func_145841_b(NBTTagCompound nbt) {
        super.func_145841_b(nbt);

        if (glassColor >= 0) {
            nbt.func_74774_a("stainedColor", (byte) glassColor);
        }
        for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
            final String key = "redstoneInputSide[" + i + "]";
            nbt.func_74774_a(key, (byte) redstoneInputSide[i]);
        }

        if (coreState.pipeId != null) {
            nbt.func_74778_a("pipeId", coreState.pipeId);
        } else {
            ResourceLocation loc = pipe != null ? Item.field_150901_e.func_177774_c(pipe.item) : null;
            String errData = "data=[";
            errData += "pipe=" + (pipe == null ? "null" : pipe.getClass());
            if (pipe != null) {
                errData += "item=" + pipe.item;
                errData += "class=" + (pipe.item == null ? "null" : pipe.item.getClass());
                errData += ", id=" + Item.field_150901_e.func_148757_b(pipe.item);
            }
            errData += ", loc=" + loc;
            errData += "]";
            if (loc == null) {
                String errLine = "A BuildCraft pipe @ " + field_174879_c.toString() + " could not save pipe ID! Please report to developers!";
                BCLog.logger.error(errLine + errData);
            } else {
                String errLine = "A BuildCraft pipe @ " + field_174879_c.toString() + " did not have pipe ID, but did have a valid item. Not a fatal error, but please report nonetheless.";
                BCLog.logger.warn(errLine + errData);
                nbt.func_74778_a("pipeId", loc.toString());
            }
        }

        if (pipe != null) {
            pipe.writeToNBT(nbt);
        }

        sideProperties.writeToNBT(nbt);
    }

    @Override
    public void func_145839_a(NBTTagCompound nbt) {
        super.func_145839_a(nbt);

        glassColor = nbt.func_74764_b("stainedColor") ? nbt.func_74771_c("stainedColor") : -1;

        redstoneInput = 0;

        for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
            final String key = "redstoneInputSide[" + i + "]";
            if (nbt.func_74764_b(key)) {
                redstoneInputSide[i] = nbt.func_74771_c(key);

                if (redstoneInputSide[i] > redstoneInput) {
                    redstoneInput = redstoneInputSide[i];
                }
            } else {
                redstoneInputSide[i] = 0;
            }
        }

        if (nbt.func_150297_b("pipeId", Constants.NBT.TAG_STRING)) {
            coreState.pipeId = nbt.func_74779_i("pipeId");
        }

        if (nbt.func_150297_b("pipeId", Constants.NBT.TAG_ANY_NUMERIC)) {
            int id = nbt.func_74762_e("pipeId");
            Item item = Item.field_150901_e.func_148754_a(id);
            ResourceLocation loc = Item.field_150901_e.func_177774_c(item);
            if (loc == null) coreState.pipeId = "";
            else coreState.pipeId = loc.toString();
        }

        if (!StringUtils.func_151246_b(coreState.pipeId)) {
            Item item = Item.field_150901_e.func_82594_a(new ResourceLocation(coreState.pipeId));
            if (item instanceof ItemPipe) {
                pipe = BlockGenericPipe.createPipe((ItemPipe) item);
            } else {
                BCLog.logger.warn(item + " was not an instanceof ItemPipe!" + coreState.pipeId);
                pipe = null;
            }
        }

        bindPipe();
        if (pipe != null) {
            pipe.readFromNBT(nbt);
        } else {
            BCLog.logger.log(Level.WARN, "Pipe failed to load from NBT at " + func_174877_v());
            deletePipe = true;
        }

        sideProperties.readFromNBT(nbt);
        attachPluggables = true;
    }

    @Override
    public void func_145843_s() {
        initialized = false;
        tileBuffer = null;

        if (pipe != null) {
            pipe.invalidate();
        }

        sideProperties.invalidate();

        super.func_145843_s();
    }

    @Override
    public void func_145829_t() {
        super.func_145829_t();
        initialized = false;
        tileBuffer = null;
        bindPipe();

        if (pipe != null) {
            pipe.validate();
        }

        sideProperties.validate(this);
    }

    protected void notifyBlockChanged() {
        field_145850_b.func_180496_d(func_174877_v(), getBlock());
        scheduleRenderUpdate();
        sendNetworkUpdate();
        if (pipe != null) {
            pipe.scheduleWireUpdate();
        }
    }

    @Override
    public void func_73660_a() {
        try {
            if (!field_145850_b.field_72995_K) {
                if (deletePipe) {
                    field_145850_b.func_175698_g(func_174877_v());
                }

                if (pipe == null || coreState.pipeId == null) {
                    return;
                }

                if (!initialized) {
                    initialize(pipe);
                }
            }

            if (attachPluggables) {
                attachPluggables = false;
                // Attach callback
                for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
                    if (sideProperties.pluggables[i] != null) {
                        pipe.eventBus.registerHandler(sideProperties.pluggables[i]);
                        sideProperties.pluggables[i].onAttachedPipe(this, EnumFacing.func_82600_a(i));
                    }
                }
                notifyBlockChanged();
            }

            if (!BlockGenericPipe.isValid(pipe)) {
                return;
            }

            pipe.updateEntity();

            for (EnumFacing direction : EnumFacing.field_82609_l) {
                PipePluggable p = getPipePluggable(direction);
                if (p != null) {
                    p.update(this, direction);
                }
            }

            if (field_145850_b.field_72995_K) {
                if (resyncGateExpansions) {
                    syncGateExpansions();
                }

                return;
            }

            if (blockNeighborChange) {
                for (int i = 0; i < 6; i++) {
                    if ((blockNeighborChangedSides & (1 << i)) != 0) {
                        blockNeighborChangedSides ^= 1 << i;
                        computeConnection(EnumFacing.func_82600_a(i));
                    }
                }
                pipe.onNeighborBlockChange(0);
                blockNeighborChange = false;
                refreshRenderState = true;
            }

            if (refreshRenderState) {
                refreshRenderState();
                refreshRenderState = false;
            }

            if (sendClientUpdate) {
                sendClientUpdate = false;

                if (!field_145850_b.field_72995_K) {
                    Packet updatePacket = getBCDescriptionPacket();
                    BuildCraftCore.instance.sendToPlayersNear(updatePacket, this);
                }
            }
        } catch (Throwable t) {
            BCLog.logger.warn("CRASH! OH NO!", t);
            Throwables.propagate(t);
        }
    }

    public void initializeFromItemMetadata(int i) {
        if (i >= 1 && i <= 16) {
            setPipeColor((i - 1) & 15);
        } else {
            setPipeColor(-1);
        }
    }

    public int getItemMetadata() {
        return getPipeColor() >= 0 ? (1 + getPipeColor()) : 0;
    }

    @Override
    public int getPipeColor() {
        return field_145850_b.field_72995_K ? renderState.getGlassColor() : this.glassColor;
    }

    public boolean setPipeColor(int color) {
        if (!field_145850_b.field_72995_K && color >= -1 && color < 16 && glassColor != color) {
            glassColor = color;
            notifyBlockChanged();
            field_145850_b.func_175722_b(func_174877_v(), getBlock());
            return true;
        }
        return false;
    }

    /** PRECONDITION: worldObj must not be null
     * 
     * @return <code>True</code> if any part of the render state changed */
    protected boolean refreshRenderState() {
        renderState.setGlassColor((byte) glassColor);

        // Pipe connections;
        for (EnumFacing o : EnumFacing.field_82609_l) {
            renderState.pipeConnectionMatrix.setConnected(o, this.pipeConnectionsBuffer[o.ordinal()]);
            if (pipeConnectionsBuffer[o.ordinal()]) {
                BlockPos connected = func_174877_v().func_177972_a(o);
                IBlockState state = field_145850_b.func_180495_p(connected);
                Block block = state.func_177230_c();
                ICustomPipeConnection connection = PipeConnectionAPI.getCustomConnection(block);
                if (connection == null) {
                    connection = DefaultPipeConnection.INSTANCE;
                }
                renderState.setExtension(o, connection.getExtension(field_145850_b, connected, o, state));
            }
        }

        // Pipe Textures
        renderState.textureMatrix.setIconIndex(null, pipe.getIconIndex(null));
        for (EnumFacing o : EnumFacing.values()) {
            renderState.textureMatrix.setIconIndex(o, pipe.getIconIndex(o));
        }

        // WireState
        for (PipeWire color : PipeWire.values()) {
            renderState.wireMatrix.setWire(color, pipe.wireSet[color.ordinal()]);

            for (EnumFacing direction : EnumFacing.field_82609_l) {
                renderState.wireMatrix.setWireConnected(color, direction, pipe.isWireConnectedTo(this.getTile(direction), color, direction));
            }

            boolean lit = pipe.signalStrength[color.ordinal()] > 0;
            renderState.wireMatrix.setWireLit(color, lit);
        }

        // Facades
        for (EnumFacing direction : EnumFacing.field_82609_l) {
            PipePluggable pluggable = sideProperties.pluggables[direction.ordinal()];
            if (!(pluggable instanceof FacadePluggable)) {
                continue;
            }

            FacadeState[] states = ((FacadePluggable) pluggable).states;
            // Iterate over all states and activate first proper
            int defaultState = -1;
            int activeState = -1;
            for (int i = 0; i < states.length; i++) {
                FacadeState state = states[i];
                if (state.wire == null) {
                    defaultState = i;
                    continue;
                }
                if (pipe != null && pipe.isWireActive(state.wire)) {
                    activeState = i;
                    break;
                }
            }
            if (activeState < 0) {
                activeState = defaultState;
            }
            ((FacadePluggable) pluggable).setActiveState(activeState);
        }

        pluggableState.setPluggables(sideProperties.pluggables);

        boolean isDirty = renderState.isDirty() || pluggableState.isDirty();

        if (isDirty) {
            sendNetworkUpdate();
            renderState.clean();
        }

        return isDirty;

    }

    public void initialize(Pipe<?> pipe) {
        initialized = false;

        this.field_145854_h = func_145838_q();

        if (pipe == null) {
            BCLog.logger.log(Level.WARN, "Pipe failed to initialize at {0}, deleting", func_174877_v());
            field_145850_b.func_175698_g(func_174877_v());
            return;
        }

        this.pipe = pipe;

        for (EnumFacing o : EnumFacing.field_82609_l) {
            TileEntity tile = getTile(o);

            if (tile instanceof ITileBufferHolder) {
                ((ITileBufferHolder) tile).blockCreated(o, BuildCraftTransport.genericPipeBlock, this);
            }
            if (tile instanceof IPipeTile) {
                ((IPipeTile) tile).scheduleNeighborChange();
            }
        }

        bindPipe();

        if (coreState.pipeId == null || coreState.pipeId.length() <= 0) {
            throw new IllegalStateException("Tried to create a pipe without a pipeID from item " + pipe.item);
        }

        computeConnections();
        scheduleNeighborChange();
        scheduleRenderUpdate();

        if (!pipe.isInitialized()) {
            pipe.initialize();
        }

        initialized = true;
    }

    private void bindPipe() {
        if (!pipeBound && pipe != null) {
            pipe.setTile(this);
            coreState.pipeId = Item.field_150901_e.func_177774_c(pipe.item).toString();
            pipeBound = true;
        }
    }

    public boolean isInitialized() {
        return initialized;
    }

    @Override
    public void scheduleNeighborChange() {
        blockNeighborChange = true;
        blockNeighborChangedSides = 0x3F;
    }

    public void scheduleNeighborChange(EnumPipePart part) {
        blockNeighborChange = true;
        blockNeighborChangedSides |= part == EnumPipePart.CENTER ? 0x3F : (1 << part.ordinal());
    }

    @Override
    public boolean canInjectItems(EnumFacing from) {
        if (getPipeType() != IPipeTile.PipeType.ITEM) {
            return false;
        }
        return isPipeConnected(from);
    }

    @Override
    public int injectItem(ItemStack payload, boolean doAdd, EnumFacing from, EnumDyeColor color) {
        if (BlockGenericPipe.isValid(pipe) && pipe.transport instanceof PipeTransportItems && isPipeConnected(from) && pipe.inputOpen(from)) {

            if (doAdd) {
                Vec3 itemPos = Utils.convertMiddle(func_174877_v()).func_178787_e(Utils.convert(from, 0.4));

                TravelingItem pipedItem = TravelingItem.make(itemPos, payload);
                if (pipedItem.isCorrupted()) {
                    return 0;
                }

                pipedItem.color = color;
                ((PipeTransportItems) pipe.transport).injectItem(pipedItem, from.func_176734_d());
            }
            return payload.field_77994_a;
        }

        return 0;
    }

    @Override
    public PipeType getPipeType() {
        if (BlockGenericPipe.isValid(pipe)) {
            return pipe.transport.getPipeType();
        }
        return null;
    }

    /* SMP */

    public Packet getBCDescriptionPacket() {
        bindPipe();
        updateCoreState();

        PacketTileState packet = new PacketTileState(this);

        if (pipe != null && pipe.transport != null) {
            pipe.transport.sendDescriptionPacket();
        }

        packet.addStateForSerialization((byte) 0, coreState);
        packet.addStateForSerialization((byte) 1, renderState);
        packet.addStateForSerialization((byte) 2, pluggableState);

        if (pipe instanceof ISerializable) {
            packet.addStateForSerialization((byte) 3, (ISerializable) pipe);
        }

        return packet;
    }

    @Override
    public S35PacketUpdateTileEntity func_145844_m() {
        NBTTagCompound nbt = new NBTTagCompound();
        nbt.func_74778_a("net-type", "desc-packet");
        Packet p = getBCDescriptionPacket();
        ByteBuf buf = Unpooled.buffer();
        p.writeData(buf);
        byte[] bytes = new byte[buf.readableBytes()];
        buf.readBytes(bytes);
        nbt.func_74773_a("net-data", bytes);
        S35PacketUpdateTileEntity tileUpdate = new S35PacketUpdateTileEntity(func_174877_v(), 0, nbt);
        return tileUpdate;
    }

    @Override
    public void onDataPacket(NetworkManager net, S35PacketUpdateTileEntity pkt) {
        if (!field_145850_b.field_72995_K) return;
        if (pkt.func_148857_g() == null) throw new RuntimeException("No NBTTag compound! This is a bug!");
        NBTTagCompound nbt = pkt.func_148857_g();
        try {
            if ("desc-packet".equals(nbt.func_74779_i("net-type"))) {
                byte[] bytes = nbt.func_74770_j("net-data");
                ByteBuf data = Unpooled.wrappedBuffer(bytes);
                PacketTileState p = new PacketTileState();
                p.readData(data);
                // The player is not used so its fine to pass null
                p.applyData(field_145850_b, null);
            } else {
                BCLog.logger.warn("Recieved a packet with a different type that expected (" + nbt.func_74779_i("net-type") + ")");
            }
        } catch (Throwable t) {
            throw new RuntimeException("Failed to read a packet! (net-type=\"" + nbt.func_74781_a("net-type") + "\", net-data=\"" + nbt.func_74781_a("net-data") + "\")", t);
        }
    }

    public void sendNetworkUpdate() {
        sendClientUpdate = true;
    }

    @Override
    public void blockRemoved(EnumFacing from) {

    }

    public TileBuffer[] getTileCache() {
        if (tileBuffer == null && pipe != null) {
            tileBuffer = TileBuffer.makeBuffer(field_145850_b, func_174877_v(), pipe.transport.delveIntoUnloadedChunks());
        }
        return tileBuffer;
    }

    @Override
    public void blockCreated(EnumFacing from, Block block, TileEntity tile) {
        TileBuffer[] cache = getTileCache();
        if (cache != null) {
            cache[from.func_176734_d().ordinal()].set(block.func_176223_P(), tile);
        }
    }

    @Override
    public Block getBlock(EnumFacing to) {
        TileBuffer[] cache = getTileCache();
        if (cache != null) {
            return cache[to.ordinal()].getBlockState().func_177230_c();
        } else {
            return null;
        }
    }

    @Override
    public TileEntity getTile(EnumFacing to) {
        return getTile(to, false);
    }

    public TileEntity getTile(EnumFacing to, boolean forceUpdate) {
        TileBuffer[] cache = getTileCache();
        if (cache != null) {
            return cache[to.ordinal()].getTile(forceUpdate);
        } else {
            return null;
        }
    }

    protected boolean canPipeConnect_internal(TileEntity with, EnumFacing side) {
        if (!(pipe instanceof IPipeConnectionForced) || !((IPipeConnectionForced) pipe).ignoreConnectionOverrides(side)) {
            if (with instanceof IPipeConnection) {
                IPipeConnection.ConnectOverride override = ((IPipeConnection) with).overridePipeConnection(pipe.transport.getPipeType(), side.func_176734_d());
                if (override != IPipeConnection.ConnectOverride.DEFAULT) {
                    return override == IPipeConnection.ConnectOverride.CONNECT;
                }
            }
        }

        if (with instanceof IPipeTile) {
            IPipeTile other = (IPipeTile) with;

            if (other.hasBlockingPluggable(side.func_176734_d())) {
                return false;
            }

            if (other.getPipeColor() >= 0 && glassColor >= 0 && other.getPipeColor() != glassColor) {
                return false;
            }

            Pipe<?> otherPipe = (Pipe<?>) other.getPipe();

            if (!BlockGenericPipe.isValid(otherPipe)) {
                return false;
            }

            if (!otherPipe.canPipeConnect(this, side.func_176734_d())) {
                return false;
            }
        }

        return pipe.canPipeConnect(with, side);
    }

    /** Checks if this tile can connect to another tile
     *
     * @param with - The other Tile
     * @param side - The orientation to get to the other tile ('with')
     * @return true if pipes are considered connected */
    protected boolean canPipeConnect(TileEntity with, EnumFacing side) {
        if (with == null) {
            return false;
        }

        if (hasBlockingPluggable(side)) {
            return false;
        }

        if (!BlockGenericPipe.isValid(pipe)) {
            return false;
        }

        // This is called when the other pipe may not have a world yet, and so it crashes
        if (!with.func_145830_o() && func_145830_o()) {
            with.func_145834_a(field_145850_b);
        }

        return canPipeConnect_internal(with, side);
    }

    @Override
    public boolean hasBlockingPluggable(EnumFacing side) {
        PipePluggable pluggable = getPipePluggable(side);
        if (pluggable == null) {
            return false;
        }

        if (pluggable instanceof IPipeConnection) {
            IPipe neighborPipe = getNeighborPipe(side);
            if (neighborPipe != null) {
                IPipeConnection.ConnectOverride override = ((IPipeConnection) pluggable).overridePipeConnection(neighborPipe.getTile().getPipeType(), side);
                if (override == IPipeConnection.ConnectOverride.CONNECT) {
                    return true;
                } else if (override == IPipeConnection.ConnectOverride.DISCONNECT) {
                    return false;
                }
            }
        }
        return pluggable.isBlocking(this, side);
    }

    protected void computeConnections() {
        for (EnumFacing face : EnumFacing.values()) {
            computeConnection(face);
        }
    }

    protected void computeConnection(EnumFacing side) {
        TileBuffer[] cache = getTileCache();
        if (cache == null) {
            return;
        }

        // TileBuffer t = cache[side.ordinal()];
        // For blocks which are not loaded, keep the old connection value.
        // if (t.exists() || !initialized) {
        // t.refresh();

        pipeConnectionsBuffer[side.ordinal()] = canPipeConnect(field_145850_b.func_175625_s(field_174879_c.func_177972_a(side))/* t.getTile(
                                                                                                        * ) */, side);
        // }
    }

    @Override
    public boolean isPipeConnected(EnumFacing with) {
        if (with == null) return false;
        if (field_145850_b.field_72995_K) {
            return renderState.pipeConnectionMatrix.isConnected(with);
        } else {
            return pipeConnectionsBuffer[with.ordinal()];
        }
    }

    @Override
    public boolean doDrop() {
        if (BlockGenericPipe.isValid(pipe)) {
            return pipe.doDrop();
        } else {
            return false;
        }
    }

    @Override
    public void onChunkUnload() {
        if (pipe != null) {
            pipe.onChunkUnload();
        }
    }

    /** ITankContainer implementation * */
    @Override
    public int fill(EnumFacing from, FluidStack resource, boolean doFill) {
        if (BlockGenericPipe.isValid(pipe) && pipe.transport instanceof IFluidHandler && !hasBlockingPluggable(from)) {
            return ((IFluidHandler) pipe.transport).fill(from, resource, doFill);
        } else {
            return 0;
        }
    }

    @Override
    public FluidStack drain(EnumFacing from, int maxDrain, boolean doDrain) {
        if (BlockGenericPipe.isValid(pipe) && pipe.transport instanceof IFluidHandler && !hasBlockingPluggable(from)) {
            return ((IFluidHandler) pipe.transport).drain(from, maxDrain, doDrain);
        } else {
            return null;
        }
    }

    @Override
    public FluidStack drain(EnumFacing from, FluidStack resource, boolean doDrain) {
        if (BlockGenericPipe.isValid(pipe) && pipe.transport instanceof IFluidHandler && !hasBlockingPluggable(from)) {
            return ((IFluidHandler) pipe.transport).drain(from, resource, doDrain);
        } else {
            return null;
        }
    }

    @Override
    public boolean canFill(EnumFacing from, Fluid fluid) {
        if (BlockGenericPipe.isValid(pipe) && pipe.transport instanceof IFluidHandler && !hasBlockingPluggable(from)) {
            return ((IFluidHandler) pipe.transport).canFill(from, fluid);
        } else {
            return false;
        }
    }

    @Override
    public boolean canDrain(EnumFacing from, Fluid fluid) {
        if (BlockGenericPipe.isValid(pipe) && pipe.transport instanceof IFluidHandler && !hasBlockingPluggable(from)) {
            return ((IFluidHandler) pipe.transport).canDrain(from, fluid);
        } else {
            return false;
        }
    }

    @Override
    public FluidTankInfo[] getTankInfo(EnumFacing from) {
        return null;
    }

    @Override
    public void scheduleRenderUpdate() {
        refreshRenderState = true;
    }

    public boolean hasFacade(EnumFacing direction) {
        if (direction == null) {
            return false;
        } else {
            return sideProperties.pluggables[direction.ordinal()] instanceof IFacadePluggable;
        }
    }

    public boolean hasGate(EnumFacing direction) {
        if (direction == null) {
            return false;
        } else {
            return sideProperties.pluggables[direction.ordinal()] instanceof GatePluggable;
        }
    }

    public boolean setPluggable(EnumFacing direction, PipePluggable pluggable) {
        return setPluggable(direction, pluggable, null);
    }

    public boolean setPluggable(EnumFacing direction, PipePluggable pluggable, EntityPlayer player) {
        if (field_145850_b != null && field_145850_b.field_72995_K) {
            return false;
        }

        if (direction == null) {
            return false;
        }

        // Remove old pluggable
        if (sideProperties.pluggables[direction.ordinal()] != null) {
            sideProperties.dropItem(this, direction, player);
            pipe.eventBus.unregisterHandler(sideProperties.pluggables[direction.ordinal()]);
        }

        sideProperties.pluggables[direction.ordinal()] = pluggable;
        if (pluggable != null) {
            pipe.eventBus.registerHandler(pluggable);
            pluggable.onAttachedPipe(this, direction);
        }

        notifyBlockChanged();
        field_145850_b.func_175722_b(func_174877_v(), getBlock());
        return true;
    }

    protected void updateCoreState() {}

    public boolean hasEnabledFacade(EnumFacing direction) {
        return hasFacade(direction) && !((FacadePluggable) getPipePluggable(direction)).isTransparent();
    }

    // Legacy
    public void setGate(Gate gate, int direction) {
        if (sideProperties.pluggables[direction] == null) {
            gate.setDirection(EnumFacing.func_82600_a(direction));
            pipe.gates[direction] = gate;
            sideProperties.pluggables[direction] = new GatePluggable(gate);
        }
    }

    @SideOnly(Side.CLIENT)
    public IIconProvider getPipeIcons() {
        if (pipe == null) {
            return null;
        }
        return pipe.getIconProvider();
    }

    @Override
    public ISerializable getStateInstance(byte stateId) {
        switch (stateId) {
            case 0:
                return coreState;
            case 1:
                return renderState;
            case 2:
                return pluggableState;
            case 3:
                return (ISerializable) pipe;
        }
        throw new RuntimeException("Unknown state requested: " + stateId + " this is a bug!");
    }

    @Override
    public void afterStateUpdated(byte stateId) {
        if (!field_145850_b.field_72995_K) {
            return;
        }

        switch (stateId) {
            case 0:
                if (pipe != null) {
                    break;
                }

                if (pipe == null && coreState.pipeId != null) {
                    initialize(BlockGenericPipe.createPipe((ItemPipe) Item.field_150901_e.func_82594_a(new ResourceLocation(coreState.pipeId))));
                }

                if (pipe == null) {
                    break;
                }

                field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
                break;

            case 1: {
                if (renderState.needsRenderUpdate()) {
                    field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
                    renderState.clean();
                }
                break;
            }
            case 2: {
                PipePluggable[] newPluggables = pluggableState.getPluggables();

                // mark for render update if necessary
                for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
                    PipePluggable old = sideProperties.pluggables[i];
                    PipePluggable newer = newPluggables[i];
                    if (old == null && newer == null) {
                        continue;
                    } else if (old != null && newer != null && old.getClass() == newer.getClass()) {
                        if (newer.requiresRenderUpdate(old)) {
                            field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
                            break;
                        }
                    } else {
                        // one of them is null but not the other, so update
                        field_145850_b.func_175704_b(func_174877_v(), func_174877_v());
                        break;
                    }
                }
                sideProperties.pluggables = newPluggables.clone();

                for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
                    final PipePluggable pluggable = getPipePluggable(EnumFacing.func_82600_a(i));
                    if (pluggable != null && pluggable instanceof GatePluggable) {
                        final GatePluggable gatePluggable = (GatePluggable) pluggable;
                        Gate gate = pipe.gates[i];
                        if (gate == null || gate.logic != gatePluggable.getLogic() || gate.material != gatePluggable.getMaterial()) {
                            pipe.gates[i] = GateFactory.makeGate(pipe, gatePluggable.getMaterial(), gatePluggable.getLogic(), EnumFacing.func_82600_a(i));
                        }
                    } else {
                        pipe.gates[i] = null;
                    }
                }

                syncGateExpansions();
                break;
            }
        }
    }

    private void syncGateExpansions() {
        resyncGateExpansions = false;
        for (int i = 0; i < EnumFacing.field_82609_l.length; i++) {
            Gate gate = pipe.gates[i];
            if (gate == null) {
                continue;
            }
            GatePluggable gatePluggable = (GatePluggable) sideProperties.pluggables[i];
            if (gatePluggable.getExpansions().length > 0) {
                for (IGateExpansion expansion : gatePluggable.getExpansions()) {
                    if (expansion != null) {
                        if (!gate.expansions.containsKey(expansion)) {
                            gate.addGateExpansion(expansion);
                        }
                    } else {
                        resyncGateExpansions = true;
                    }
                }
            }
        }
    }

    @Override
    @SideOnly(Side.CLIENT)
    public double func_145833_n() {
        return DefaultProps.PIPE_CONTENTS_RENDER_DIST * DefaultProps.PIPE_CONTENTS_RENDER_DIST;
    }

    @Override
    public boolean shouldRefresh(World world, BlockPos pos, IBlockState oldState, IBlockState newState) {
        return oldState.func_177230_c() != newState.func_177230_c();
    }

    @Override
    public boolean isSolidOnSide(EnumFacing side) {
        if (hasPipePluggable(side) && getPipePluggable(side).isSolidOnSide(this, side)) {
            return true;
        }

        if (BlockGenericPipe.isValid(pipe) && pipe instanceof ISolidSideTile) {
            if (((ISolidSideTile) pipe).isSolidOnSide(side)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public PipePluggable getPipePluggable(EnumFacing side) {
        if (side == null) {
            return null;
        }

        return sideProperties.pluggables[side.ordinal()];
    }

    @Override
    public boolean hasPipePluggable(EnumFacing side) {
        if (side == null) {
            return false;
        }

        return sideProperties.pluggables[side.ordinal()] != null;
    }

    public Block getBlock() {
        return func_145838_q();
    }

    public boolean isUseableByPlayer(EntityPlayer player) {
        return field_145850_b.func_175625_s(func_174877_v()) == this;
    }

    @Override
    public void writeGuiData(ByteBuf data) {
        if (BlockGenericPipe.isValid(pipe) && pipe instanceof IGuiReturnHandler) {
            ((IGuiReturnHandler) pipe).writeGuiData(data);
        }
    }

    @Override
    public void readGuiData(ByteBuf data, EntityPlayer sender) {
        if (BlockGenericPipe.isValid(pipe) && pipe instanceof IGuiReturnHandler) {
            ((IGuiReturnHandler) pipe).readGuiData(data, sender);
        }
    }

    private IEnergyHandler internalGetEnergyHandler(EnumFacing side) {
        if (hasPipePluggable(side)) {
            PipePluggable pluggable = getPipePluggable(side);
            if (pluggable instanceof IEnergyHandler) {
                return (IEnergyHandler) pluggable;
            } else if (pluggable.isBlocking(this, side)) {
                return null;
            }
        }
        if (pipe instanceof IEnergyHandler) {
            return (IEnergyHandler) pipe;
        }
        return null;
    }

    @Override
    public boolean canConnectEnergy(EnumFacing from) {
        IEnergyHandler handler = internalGetEnergyHandler(from);
        if (handler != null) {
            return handler.canConnectEnergy(from);
        } else {
            return false;
        }
    }

    @Override
    public int receiveEnergy(EnumFacing from, int maxReceive, boolean simulate) {
        IEnergyHandler handler = internalGetEnergyHandler(from);
        if (handler instanceof IEnergyReceiver) {
            return ((IEnergyReceiver) handler).receiveEnergy(from, maxReceive, simulate);
        } else {
            return 0;
        }
    }

    @Override
    public int extractEnergy(EnumFacing from, int maxExtract, boolean simulate) {
        IEnergyHandler handler = internalGetEnergyHandler(from);
        if (handler instanceof IEnergyProvider) {
            return ((IEnergyProvider) handler).extractEnergy(from, maxExtract, simulate);
        } else {
            return 0;
        }
    }

    @Override
    public int getEnergyStored(EnumFacing from) {
        IEnergyHandler handler = internalGetEnergyHandler(from);
        if (handler != null) {
            return handler.getEnergyStored(from);
        } else {
            return 0;
        }
    }

    @Override
    public int getMaxEnergyStored(EnumFacing from) {
        IEnergyHandler handler = internalGetEnergyHandler(from);
        if (handler != null) {
            return handler.getMaxEnergyStored(from);
        } else {
            return 0;
        }
    }

    @Override
    public Block getNeighborBlock(EnumFacing dir) {
        return getBlock(dir);
    }

    @Override
    public TileEntity getNeighborTile(EnumFacing dir) {
        return getTile(dir);
    }

    @Override
    public IPipe getNeighborPipe(EnumFacing dir) {
        TileEntity neighborTile = getTile(dir);
        if (neighborTile instanceof IPipeTile) {
            return ((IPipeTile) neighborTile).getPipe();
        } else {
            return null;
        }
    }

    @Override
    public IPipe getPipe() {
        return pipe;
    }

    @Override
    public boolean canConnectRedstoneEngine(EnumFacing side) {
        if (pipe instanceof IRedstoneEngineReceiver) {
            return ((IRedstoneEngineReceiver) pipe).canConnectRedstoneEngine(side);
        } else {
            return (getPipeType() != PipeType.POWER) && (getPipeType() != PipeType.STRUCTURE);
        }
    }

    @Override
    @SideOnly(Side.CLIENT)
    public void getDebugInfo(List<String> left, List<String> right, EnumFacing side) {
        int glassColor = getPipeColor();
        if (glassColor >= 0 && glassColor < 16) {
            left.add("");
            left.add("Colour = " + EnumDyeColor.values()[glassColor]);
        }

        if (pipe instanceof IDebuggable) {
            ((IDebuggable) pipe).getDebugInfo(left, right, side);
        }
        if (pipe.transport instanceof IDebuggable) {
            ((IDebuggable) pipe.transport).getDebugInfo(left, right, side);
        }
        if (getPipePluggable(side) != null && getPipePluggable(side) instanceof IDebuggable) {
            ((IDebuggable) getPipePluggable(side)).getDebugInfo(left, right, side);
        }
    }

    @Override
    public ConnectOverride overridePipeConnection(PipeType type, EnumFacing with) {
        if (type == PipeType.POWER && hasPipePluggable(with) && getPipePluggable(with) instanceof IEnergyHandler) {
            return ConnectOverride.CONNECT;
        }
        return ConnectOverride.DEFAULT;
    }

    @Override
    public World getWorldBC() {
        return func_145831_w();
    }

    @Override
    public BlockPos getPosBC() {
        return func_174877_v();
    }
}
