/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.transport;

import buildcraft.BuildCraftCore;
import buildcraft.core.lib.inventory.StackHelper;
import buildcraft.core.lib.utils.NBTUtils;
import buildcraft.core.lib.utils.Utils;
import com.google.common.collect.MapMaker;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.item.EnumDyeColor;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Vec3;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;

import java.util.EnumSet;
import java.util.Map;

public class TravelingItem {

    public static final TravelingItemCache serverCache = new TravelingItemCache();
    public static final TravelingItemCache clientCache = new TravelingItemCache();
    public static final InsertionHandler DEFAULT_INSERTION_HANDLER = new InsertionHandler();
    private static int maxId = 0;

    public final EnumSet<EnumFacing> blacklist = EnumSet.noneOf(EnumFacing.class);

    public Vec3 pos;
    public final int id;
    public boolean toCenter = true;
    public EnumDyeColor color;
    public EnumFacing input = null;
    public EnumFacing output = null;

    public int displayList;
    public boolean hasDisplayList;

    protected float speed = 0.01F;

    protected ItemStack itemStack;
    protected TileEntity container;
    protected NBTTagCompound extraData;
    protected InsertionHandler insertionHandler = DEFAULT_INSERTION_HANDLER;

    /* CONSTRUCTORS */
    protected TravelingItem(int id) {
        this.id = id;
    }

    public static TravelingItem make(int id) {
        TravelingItem item = new TravelingItem(id);
        getCache().cache(item);
        return item;
    }

    public static TravelingItem make() {
        return make(maxId < Short.MAX_VALUE ? ++maxId : (maxId = Short.MIN_VALUE));
    }

    public static TravelingItem make(Vec3 pos, ItemStack stack) {
        TravelingItem item = make();
        item.pos = pos;
        item.itemStack = stack.func_77946_l();
        return item;
    }

    public static TravelingItem make(NBTTagCompound nbt) {
        TravelingItem item = make();
        item.readFromNBT(nbt);
        return item;
    }

    public static TravelingItemCache getCache() {
        if (FMLCommonHandler.instance().getEffectiveSide() == Side.CLIENT) {
            return clientCache;
        }
        return serverCache;
    }

    public void movePosition(Vec3 toAdd) {
        pos = pos.func_178787_e(toAdd);
    }

    public float getSpeed() {
        return speed;
    }

    public void setSpeed(float speed) {
        this.speed = speed;
    }

    public ItemStack getItemStack() {
        return itemStack;
    }

    public void setItemStack(ItemStack item) {
        this.itemStack = item;
    }

    public TileEntity getContainer() {
        return container;
    }

    public void setContainer(TileEntity container) {
        this.container = container;
    }

    public NBTTagCompound getExtraData() {
        if (extraData == null) {
            extraData = new NBTTagCompound();
        }
        return extraData;
    }

    public boolean hasExtraData() {
        return extraData != null;
    }

    public void setInsertionHandler(InsertionHandler handler) {
        if (handler == null) {
            return;
        }
        this.insertionHandler = handler;
    }

    public InsertionHandler getInsertionHandler() {
        return insertionHandler;
    }

    public void reset() {
        toCenter = true;
        blacklist.clear();
        input = null;
        output = null;
    }

    /* SAVING & LOADING */
    public void readFromNBT(NBTTagCompound data) {
        pos = new Vec3(data.func_74769_h("x"), data.func_74769_h("y"), data.func_74769_h("z"));

        setSpeed(data.func_74760_g("speed"));
        setItemStack(ItemStack.func_77949_a(data.func_74775_l("Item")));

        toCenter = data.func_74767_n("toCenter");
        input = NBTUtils.readEnum(data.func_74781_a("input"), EnumFacing.class);
        output = NBTUtils.readEnum(data.func_74781_a("output"), EnumFacing.class);

        byte c = data.func_74771_c("color");
        if (c != -1) {
            color = EnumDyeColor.func_176764_b(c);
        }

        if (data.func_74764_b("extraData")) {
            extraData = data.func_74775_l("extraData");
        }
    }

    public void writeToNBT(NBTTagCompound data) {
        data.func_74780_a("x", pos.field_72450_a);
        data.func_74780_a("y", pos.field_72448_b);
        data.func_74780_a("z", pos.field_72449_c);
        data.func_74776_a("speed", getSpeed());
        NBTTagCompound itemStackTag = new NBTTagCompound();
        getItemStack().func_77955_b(itemStackTag);
        data.func_74782_a("Item", itemStackTag);

        data.func_74757_a("toCenter", toCenter);
        data.func_74782_a("input", NBTUtils.writeEnum(input));
        data.func_74782_a("output", NBTUtils.writeEnum(output));

        data.func_74774_a("color", color != null ? (byte) color.func_176765_a() : -1);

        if (extraData != null) {
            data.func_74782_a("extraData", extraData);
        }
    }

    public EntityItem toEntityItem() {
        if (container != null && !container.func_145831_w().field_72995_K) {
            if (getItemStack().field_77994_a <= 0) {
                return null;
            }

            Vec3 motion = Utils.convert(output, 0.1 + getSpeed() * 2D);

            EntityItem entity = new EntityItem(container.func_145831_w(), pos.field_72450_a, pos.field_72448_b, pos.field_72449_c, getItemStack());
            entity.lifespan = BuildCraftCore.itemLifespan * 20;
            entity.func_174869_p();

            float f3 = 0.00F + container.func_145831_w().field_73012_v.nextFloat() * 0.04F - 0.02F;
            entity.field_70159_w = (float) container.func_145831_w().field_73012_v.nextGaussian() * f3 + motion.field_72450_a;
            entity.field_70181_x = (float) container.func_145831_w().field_73012_v.nextGaussian() * f3 + motion.field_72448_b;
            entity.field_70179_y = (float) container.func_145831_w().field_73012_v.nextGaussian() * f3 + +motion.field_72449_c;
            return entity;
        }
        return null;
    }

    public float getEntityBrightness(float f) {
        // int i = MathHelper.floor_double(xCoord);
        // int j = MathHelper.floor_double(zCoord);

        // Ok... is this a nether checking thing?
        // And why would you want this?
        // Being removed unless testing requires it
        // if (container != null && !container.getWorld().isAirBlock(new BlockPos(i, 64, j))) {

        double d = 2 / 3D;
        // int k = MathHelper.floor_double(pos.yCoord + d);
        return container.func_145831_w().func_175724_o(Utils.convertFloor(pos.func_72441_c(0, d, 0)));
        // } else {
        // return 0.0F;
        // }
    }

    public boolean isCorrupted() {
        return itemStack == null || itemStack.field_77994_a <= 0 || itemStack.func_77973_b() == null;
    }

    public boolean canBeGroupedWith(TravelingItem otherItem) {
        if (otherItem == this) {
            return false;
        }
        if (toCenter != otherItem.toCenter) {
            return false;
        }
        if (output != otherItem.output) {
            return false;
        }
        if (color != otherItem.color) {
            return false;
        }
        if (hasExtraData() || otherItem.hasExtraData()) {
            return false;
        }
        if (insertionHandler != DEFAULT_INSERTION_HANDLER) {
            return false;
        }
        if (!blacklist.equals(otherItem.blacklist)) {
            return false;
        }
        if (otherItem.isCorrupted()) {
            return false;
        }
        return StackHelper.canStacksMerge(itemStack, otherItem.itemStack);
    }

    public boolean tryMergeInto(TravelingItem otherItem) {
        if (!canBeGroupedWith(otherItem)) {
            return false;
        }
        if (StackHelper.mergeStacks(itemStack, otherItem.itemStack, false) == itemStack.field_77994_a) {
            StackHelper.mergeStacks(itemStack, otherItem.itemStack, true);
            itemStack.field_77994_a = 0;
            return true;
        }
        return false;
    }

    public boolean ignoreWeight() {
        return false;
    }

    @Override
    public int hashCode() {
        return this.id;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final TravelingItem other = (TravelingItem) obj;
        if (this.id != other.id) {
            return false;
        }
        return true;
    }

    public void cleanup() {
        if (hasDisplayList) {
            TransportProxy.proxy.clearDisplayList(displayList);
            hasDisplayList = false;
        }
    }

    @Override
    public String toString() {
        return "TravelingItem: " + id;
    }

    public static class InsertionHandler {
        public boolean canInsertItem(TravelingItem item, Object target) {
            return true;
        }
    }

    public static class TravelingItemCache {
        private final Map<Integer, TravelingItem> itemCache = new MapMaker().weakValues().makeMap();

        public void cache(TravelingItem item) {
            itemCache.put(item.id, item);
        }

        public TravelingItem get(int id) {
            return itemCache.get(id);
        }
    }
}
