/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.transport.gates;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.nbt.NBTTagString;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.StatCollector;

import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import buildcraft.BuildCraftTransport;
import buildcraft.api.gates.GateExpansions;
import buildcraft.api.gates.IGateExpansion;
import buildcraft.api.transport.IPipe;
import buildcraft.api.transport.pluggable.IPipePluggableItem;
import buildcraft.api.transport.pluggable.PipePluggable;
import buildcraft.core.BCCreativeTab;
import buildcraft.core.lib.inventory.InvUtils;
import buildcraft.core.lib.items.ItemBuildCraft;
import buildcraft.core.lib.utils.BCStringUtils;
import buildcraft.transport.Gate;
import buildcraft.transport.Pipe;
import buildcraft.transport.gates.GateDefinition.GateLogic;
import buildcraft.transport.gates.GateDefinition.GateMaterial;

public class ItemGate extends ItemBuildCraft implements IPipePluggableItem {

    protected static final String NBT_TAG_MAT = "mat";
    protected static final String NBT_TAG_LOGIC = "logic";
    protected static final String NBT_TAG_EX = "ex";
    private static ArrayList<ItemStack> allGates;

    public ItemGate() {
        super();
        func_77627_a(false);
        func_77656_e(0);
        setPassSneakClick(true);
        func_77637_a(BCCreativeTab.get("gates"));
    }

    private static NBTTagCompound getNBT(ItemStack stack) {
        if (stack == null || !(stack.func_77973_b() instanceof ItemGate)) {
            return null;
        } else {
            return InvUtils.getItemData(stack);
        }
    }

    public static void setMaterial(ItemStack stack, GateMaterial material) {
        NBTTagCompound nbt = InvUtils.getItemData(stack);
        nbt.func_74774_a(NBT_TAG_MAT, (byte) material.ordinal());
    }

    public static GateMaterial getMaterial(ItemStack stack) {
        NBTTagCompound nbt = getNBT(stack);

        if (nbt == null) {
            return GateMaterial.REDSTONE;
        } else {
            return GateMaterial.fromOrdinal(nbt.func_74771_c(NBT_TAG_MAT));
        }
    }

    public static GateLogic getLogic(ItemStack stack) {
        NBTTagCompound nbt = getNBT(stack);

        if (nbt == null) {
            return GateLogic.AND;
        } else {
            return GateLogic.fromOrdinal(nbt.func_74771_c(NBT_TAG_LOGIC));
        }
    }

    public static void setLogic(ItemStack stack, GateLogic logic) {
        NBTTagCompound nbt = InvUtils.getItemData(stack);
        nbt.func_74774_a(NBT_TAG_LOGIC, (byte) logic.ordinal());
    }

    public static void addGateExpansion(ItemStack stack, IGateExpansion expansion) {
        NBTTagCompound nbt = getNBT(stack);

        if (nbt == null) {
            return;
        }

        NBTTagList expansionList = nbt.func_150295_c(NBT_TAG_EX, Constants.NBT.TAG_STRING);
        expansionList.func_74742_a(new NBTTagString(expansion.getUniqueIdentifier()));
        nbt.func_74782_a(NBT_TAG_EX, expansionList);
    }

    public static boolean hasGateExpansion(ItemStack stack, IGateExpansion expansion) {
        NBTTagCompound nbt = getNBT(stack);

        if (nbt == null) {
            return false;
        }

        try {
            NBTTagList expansionList = nbt.func_150295_c(NBT_TAG_EX, Constants.NBT.TAG_STRING);

            for (int i = 0; i < expansionList.func_74745_c(); i++) {
                String ex = expansionList.func_150307_f(i);

                if (ex.equals(expansion.getUniqueIdentifier())) {
                    return true;
                }
            }
        } catch (RuntimeException error) {}

        return false;
    }

    public static Set<IGateExpansion> getInstalledExpansions(ItemStack stack) {
        Set<IGateExpansion> expansions = new HashSet<>();
        NBTTagCompound nbt = getNBT(stack);

        if (nbt == null) {
            return expansions;
        }

        try {
            NBTTagList expansionList = nbt.func_150295_c(NBT_TAG_EX, Constants.NBT.TAG_STRING);
            for (int i = 0; i < expansionList.func_74745_c(); i++) {
                String exTag = expansionList.func_150307_f(i);
                IGateExpansion ex = GateExpansions.getExpansion(exTag);
                if (ex != null) {
                    expansions.add(ex);
                }
            }
        } catch (RuntimeException error) {}

        return expansions;
    }

    public static ItemStack makeGateItem(GateMaterial material, GateLogic logic) {
        ItemStack stack = new ItemStack(BuildCraftTransport.pipeGate);
        NBTTagCompound nbt = InvUtils.getItemData(stack);
        nbt.func_74774_a(NBT_TAG_MAT, (byte) material.ordinal());
        nbt.func_74774_a(NBT_TAG_LOGIC, (byte) logic.ordinal());

        return stack;
    }

    public static ItemStack makeGateItem(Gate gate) {
        ItemStack stack = new ItemStack(BuildCraftTransport.pipeGate);
        NBTTagCompound nbt = InvUtils.getItemData(stack);
        nbt.func_74774_a(NBT_TAG_MAT, (byte) gate.material.ordinal());
        nbt.func_74774_a(NBT_TAG_LOGIC, (byte) gate.logic.ordinal());

        for (IGateExpansion expansion : gate.expansions.keySet()) {
            addGateExpansion(stack, expansion);
        }

        return stack;
    }

    @Override
    public String func_77667_c(ItemStack stack) {
        return GateDefinition.getLocalizedName(getMaterial(stack), getLogic(stack));
    }

    @Override
    public String func_77653_i(ItemStack stack) {
        return ("" + StatCollector.func_74838_a(this.func_77657_g(stack))).trim();
    }

    @SuppressWarnings({ "unchecked", "rawtypes" })
    @Override
    @SideOnly(Side.CLIENT)
    public void func_150895_a(Item item, CreativeTabs tab, List itemList) {
        for (GateMaterial material : GateMaterial.VALUES) {
            for (GateLogic logic : GateLogic.VALUES) {
                if (material == GateMaterial.REDSTONE && logic == GateLogic.OR) {
                    continue;
                }

                itemList.add(makeGateItem(material, logic));

                for (IGateExpansion exp : GateExpansions.getExpansions()) {
                    ItemStack stackExpansion = makeGateItem(material, logic);
                    addGateExpansion(stackExpansion, exp);
                    itemList.add(stackExpansion);
                }
            }
        }
    }

    @Override
    public void func_77624_a(ItemStack stack, EntityPlayer player, List<String> list, boolean adv) {
        super.func_77624_a(stack, player, list, adv);

        list.add(BCStringUtils.localize("tip.gate.wires"));
        list.add(BCStringUtils.localize("tip.gate.wires." + getMaterial(stack).getTag()));
        Set<IGateExpansion> expansions = getInstalledExpansions(stack);

        if (!expansions.isEmpty()) {
            list.add(BCStringUtils.localize("tip.gate.expansions"));

            for (IGateExpansion expansion : expansions) {
                list.add(expansion.getDisplayName());
            }
        }
    }

    @Override
    public PipePluggable createPipePluggable(IPipe pipe, EnumFacing side, ItemStack stack) {
        Pipe<?> realPipe = (Pipe<?>) pipe;

        return new GatePluggable(GateFactory.makeGate(realPipe, stack, side));
    }

    public static ArrayList<ItemStack> getAllGates() {
        if (allGates == null) {
            allGates = new ArrayList<>();
            for (GateDefinition.GateMaterial m : GateDefinition.GateMaterial.VALUES) {
                for (GateDefinition.GateLogic l : GateDefinition.GateLogic.VALUES) {
                    if (m == GateMaterial.REDSTONE && l == GateLogic.OR) {
                        continue;
                    }

                    allGates.add(ItemGate.makeGateItem(m, l));
                }
            }
        }
        return allGates;
    }
}
