/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.transport.pipes;

import buildcraft.BuildCraftTransport;
import buildcraft.api.core.IIconProvider;
import buildcraft.core.lib.RFBattery;
import buildcraft.core.lib.inventory.ITransactor;
import buildcraft.core.lib.inventory.Transactor;
import buildcraft.core.lib.inventory.filters.StackFilter;
import buildcraft.core.lib.utils.Utils;
import buildcraft.core.proxy.CoreProxy;
import buildcraft.transport.*;
import buildcraft.transport.pipes.events.PipeEventItem;
import buildcraft.transport.utils.TransportUtils;
import cofh.api.energy.IEnergyReceiver;
import net.minecraft.entity.Entity;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.item.EntityMinecart;
import net.minecraft.entity.projectile.EntityArrow;
import net.minecraft.init.Items;
import net.minecraft.inventory.IInventory;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.util.AxisAlignedBB;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.Vec3;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import java.util.List;
import java.util.WeakHashMap;

public class PipeItemsObsidian extends Pipe<PipeTransportItems> implements IEnergyReceiver {
    private final RFBattery battery = new RFBattery(2560, 640, 0);
    private final WeakHashMap<Entity, Long> entityDropTime = new WeakHashMap<>();

    public PipeItemsObsidian(Item item) {
        super(new PipeTransportItems(), item);
    }

    @Override
    @SideOnly(Side.CLIENT)
    public IIconProvider getIconProvider() {
        return BuildCraftTransport.instance.pipeIconProvider;
    }

    @Override
    public int getIconIndex(EnumFacing direction) {
        return PipeIconProvider.TYPE.PipeItemsObsidian.ordinal();
    }

    @Override
    public void onEntityCollidedWithBlock(Entity entity) {
        super.onEntityCollidedWithBlock(entity);

        if (entity.field_70128_L) {
            return;
        }

        if (canSuck(entity, 0)) {
            pullItemIntoPipe(entity, 0);
        }
    }

    private AxisAlignedBB getSuckingBox(EnumFacing orientation, int distance) {
        if (orientation == null) {
            return null;
        }
        Vec3 p1 = Utils.convert(container.func_174877_v()).func_178787_e(Utils.VEC_HALF);
        Vec3 p2 = p1;

        switch (orientation) {
            case EAST:
                p1 = p1.func_72441_c(distance, 0, 0);
                p2 = p2.func_72441_c(distance + 1, 0, 0);
                break;
            case WEST:
                p1 = p1.func_72441_c(-distance - 1, 0, 0);
                p2 = p2.func_72441_c(-distance, 0, 0);
                break;
            case UP:
            case DOWN:
                p1 = p1.func_72441_c(distance + 1, 0, distance + 1);
                p2 = p2.func_72441_c(-distance, 0, -distance);
                break;
            case SOUTH:
                p1 = p1.func_72441_c(0, 0, distance);
                p2 = p2.func_72441_c(0, 0, distance + 1);
                break;
            case NORTH:
            default:
                p1 = p1.func_72441_c(0, 0, -distance - 1);
                p2 = p2.func_72441_c(0, 0, -distance);
                break;
        }

        switch (orientation) {
            case EAST:
            case WEST:
                p1 = p1.func_72441_c(0, distance + 1, distance + 1);
                p2 = p2.func_72441_c(0, -distance, -distance);
                break;
            case UP:
                p1 = p1.func_72441_c(0, distance + 1, 0);
                p2 = p2.func_72441_c(0, distance, 0);
                break;
            case DOWN:
                p1 = p1.func_72441_c(0, -distance - 1, 0);
                p2 = p2.func_72441_c(0, -distance, 0);
                break;
            case SOUTH:
            case NORTH:
            default:
                p1 = p1.func_72441_c(distance + 1, distance + 1, 0);
                p2 = p2.func_72441_c(-distance, -distance, 0);
                break;
        }

        Vec3 min = Utils.min(p1, p2);
        Vec3 max = Utils.max(p1, p2);

        return new AxisAlignedBB(min.field_72450_a, min.field_72448_b, min.field_72449_c, max.field_72450_a, max.field_72448_b, max.field_72449_c);
    }

    @Override
    public void updateEntity() {
        super.updateEntity();

        if (battery.getEnergyStored() > 0) {
            for (int j = 1; j < 5; ++j) {
                if (suckItem(j)) {
                    return;
                }
            }

            battery.useEnergy(0, 5, false);
        }
    }

    private boolean suckItem(int distance) {
        EnumFacing openOrientation = getOpenOrientation();
        if (openOrientation == null) {
            return false;
        }
        AxisAlignedBB box = getSuckingBox(openOrientation, distance);

        if (box == null) {
            return false;
        }

        List<Entity> discoveredEntities = container.func_145831_w().func_72872_a(Entity.class, box);

        for (Entity entity : discoveredEntities) {
            if (canSuck(entity, distance)) {
                pullItemIntoPipe(entity, distance);
                return true;
            }

            if (distance == 1 && entity instanceof EntityMinecart && entity instanceof IInventory) {
                EntityMinecart cart = (EntityMinecart) entity;
                if (!cart.field_70128_L) {
                    ITransactor trans = Transactor.getTransactorFor(cart, openOrientation);
                    ItemStack stack = trans.remove(StackFilter.ALL, false);

                    if (stack != null && battery.useEnergy(10, 10, false) > 0) {
                        stack = trans.remove(StackFilter.ALL, true);
                        if (stack != null) {
                            Vec3 pos = Utils.convertMiddle(container.func_174877_v());
                            TravelingItem item = TravelingItem.make(pos, stack);
                            transport.injectItem(item, openOrientation.func_176734_d());
                        }
                        return true;
                    }
                }
            }
        }

        return false;
    }

    public void pullItemIntoPipe(Entity entity, int distance) {
        if (container.func_145831_w().field_72995_K) {
            return;
        }

        EnumFacing orientation = getOpenOrientation();
        if (orientation != null) {
            orientation = orientation.func_176734_d();
            container.func_145831_w().func_72956_a(entity, "random.pop", 0.2F, ((container.func_145831_w().field_73012_v.nextFloat() - container.func_145831_w().field_73012_v
                    .nextFloat()) * 0.7F + 1.0F) * 2.0F);

            ItemStack stack;

            double speed = 0.01F;

            if (entity instanceof EntityItem) {
                EntityItem item = (EntityItem) entity;
                ItemStack contained = item.func_92059_d();

                if (contained == null) {
                    return;
                }

                TransportProxy.proxy.obsidianPipePickup(container.func_145831_w(), item, this.container);

                int energyUsed = Math.min(10 * contained.field_77994_a * distance, battery.getEnergyStored());

                if (distance == 0 || energyUsed / distance / 10 == contained.field_77994_a) {
                    stack = contained;
                    CoreProxy.proxy.removeEntity(entity);
                } else {
                    stack = contained.func_77979_a(energyUsed / distance / 10);
                }

                battery.useEnergy(energyUsed, energyUsed, false);

                speed = Math.sqrt(item.field_70159_w * item.field_70159_w + item.field_70181_x * item.field_70181_x + item.field_70179_y * item.field_70179_y);
                speed = speed / 2F - 0.05;

                if (speed < 0.01) {
                    speed = 0.01;
                }
            } else if (entity instanceof EntityArrow && battery.useEnergy(distance * 10, distance * 10, false) > 0) {
                stack = new ItemStack(Items.field_151032_g, 1);
                CoreProxy.proxy.removeEntity(entity);
            } else {
                return;
            }

            if (stack == null) {
                return;
            }

            TravelingItem item = TravelingItem.make(Utils.convert(container.func_174877_v()).func_72441_c(0.5, TransportUtils.getPipeFloorOf(stack), 0.5),
                    stack);

            item.setSpeed((float) speed);

            transport.injectItem(item, orientation);
        }
    }

    public void eventHandler(PipeEventItem.DropItem event) {
        entityDropTime.put(event.entity, event.entity.field_70170_p.func_82737_E() + 200);
    }

    public boolean canSuck(Entity entity, int distance) {
        if (!entity.func_70089_S()) {
            return false;
        }
        if (entity instanceof EntityItem) {
            EntityItem item = (EntityItem) entity;

            if (item.func_92059_d().field_77994_a <= 0) {
                return false;
            }

            long wt = entity.field_70170_p.func_82737_E();
            if (entityDropTime.containsKey(entity) && entityDropTime.get(entity) >= wt) {
                return false;
            }

            return battery.getEnergyStored() >= distance * 10;
        } else if (entity instanceof EntityArrow) {
            EntityArrow arrow = (EntityArrow) entity;
            return arrow.field_70251_a == 1 && battery.getEnergyStored() >= distance * 10;
        }
        return false;
    }

    @Override
    public boolean canConnectEnergy(EnumFacing from) {
        return true;
    }

    @Override
    public int receiveEnergy(EnumFacing from, int maxReceive, boolean simulate) {
        return battery.receiveEnergy(maxReceive, simulate);
    }

    @Override
    public int getEnergyStored(EnumFacing from) {
        return battery.getEnergyStored();
    }

    @Override
    public int getMaxEnergyStored(EnumFacing from) {
        return battery.getMaxEnergyStored();
    }
}
