package buildcraft.transport.pipes.bc8;

import java.util.Collection;
import java.util.UUID;

import net.minecraft.entity.Entity;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.world.World;

import buildcraft.api.core.BCLog;
import buildcraft.api.transport.pipe_bc8.IConnection_BC8;
import buildcraft.api.transport.pipe_bc8.IExtractionManager.IExtractable_BC8;
import buildcraft.api.transport.pipe_bc8.IInsertionManager.IInsertable_BC8;
import buildcraft.api.transport.pipe_bc8.IPipeHolder_BC8;
import buildcraft.api.transport.pipe_bc8.IPipe_BC8;
import buildcraft.api.transport.pipe_bc8.PipeAPI_BC8;
import buildcraft.core.lib.utils.NBTUtils;

public abstract class PipeConnection implements IConnection_BC8 {
    private final double length;
    private final IExtractable_BC8 extractable;
    private final IInsertable_BC8 insertable;

    public PipeConnection(double length, Object obj) {
        this.length = length;
        extractable = PipeAPI_BC8.EXTRACTION_MANAGER.getExtractableFor(obj);
        insertable = PipeAPI_BC8.INSERTION_MANAGER.getInsertableFor(obj);
    }

    @Override
    public double getLength() {
        return length;
    }

    @Override
    public IExtractable_BC8 getExtractor() {
        return extractable;
    }

    @Override
    public IInsertable_BC8 getInserter() {
        return insertable;
    }

    abstract NBTTagCompound saveConnection();

    private static Entity getEntityBuUUID(World world, UUID uuid) {
        for (Entity ent : (Collection<Entity>) world.field_72996_f) {
            if (ent.getPersistentID().equals(uuid)) return ent;
        }
        return null;
    }

    public static PipeConnection loadConnection(NBTTagCompound nbt, World world) {
        PipeConnectionType type = NBTUtils.readEnum(nbt.func_74781_a("type"), PipeConnectionType.class);
        if (type == PipeConnectionType.TILE) {
            BlockPos pos = NBTUtils.readBlockPos(nbt.func_74781_a("pos"));
            TileEntity tile = world.func_175625_s(pos);
            double length = nbt.func_74769_h("length");
            if (tile == null) return null;
            return new Tile(length, tile);
        } else if (type == PipeConnectionType.PIPE_TILE) {
            BlockPos pos = NBTUtils.readBlockPos(nbt.func_74781_a("pos"));
            TileEntity tile = world.func_175625_s(pos);
            if (tile == null) return null;
            if (!(tile instanceof IPipeHolder_BC8)) return null;
            double length = nbt.func_74769_h("length");
            return new Pipe(length, ((IPipeHolder_BC8) tile).getPipe());
        } else if (type == PipeConnectionType.PIPE_ENTITY) {
            UUID id = UUID.fromString(nbt.func_74779_i("uuid"));
            Entity ent = getEntityBuUUID(world, id);
            if (ent == null) return null;
            if (!(ent instanceof IPipeHolder_BC8)) return null;
            double length = nbt.func_74769_h("length");
            return new Pipe(length, ((IPipeHolder_BC8) ent).getPipe());
        } else if (type == PipeConnectionType.ENTITY) {
            UUID id = UUID.fromString(nbt.func_74779_i("uuid"));
            Entity ent = getEntityBuUUID(world, id);
            if (ent == null) return null;
            double length = nbt.func_74769_h("length");
            return new MovableEntity(length, ent);
        }

        BCLog.logger.warn("Tried to load a connection with an unknown type! " + type);
        return null;
    }

    private enum PipeConnectionType {
        TILE,
        PIPE_TILE,
        PIPE_ENTITY,
        ENTITY
    }

    public static class Tile extends PipeConnection implements IConnection_BC8.Tile {
        private final TileEntity tile;

        public Tile(double length, TileEntity tile) {
            super(length, tile);
            this.tile = tile;
        }

        @Override
        public TileEntity getOther() {
            return tile;
        }

        @Override
        NBTTagCompound saveConnection() {
            NBTTagCompound tag = new NBTTagCompound();
            tag.func_74782_a("type", NBTUtils.writeEnum(PipeConnectionType.TILE));
            tag.func_74782_a("pos", NBTUtils.writeBlockPos(tile.func_174877_v()));
            tag.func_74780_a("length", getLength());
            return tag;
        }
    }

    public static class Pipe extends PipeConnection implements IConnection_BC8.Pipe {
        private final IPipe_BC8 pipe;

        public Pipe(double length, IPipe_BC8 pipe) {
            super(length, pipe);
            this.pipe = pipe;
        }

        @Override
        public IPipe_BC8 getOther() {
            return pipe;
        }

        @Override
        NBTTagCompound saveConnection() {
            NBTTagCompound tag = new NBTTagCompound();
            tag.func_74780_a("length", getLength());
            if (pipe.getHolder() instanceof TileEntity) {
                tag.func_74782_a("type", NBTUtils.writeEnum(PipeConnectionType.PIPE_TILE));
                tag.func_74782_a("pos", NBTUtils.writeBlockPos(((TileEntity) pipe.getHolder()).func_174877_v()));
            } else if (pipe.getHolder() instanceof Entity) {
                tag.func_74782_a("type", NBTUtils.writeEnum(PipeConnectionType.PIPE_ENTITY));
                UUID uuid = ((Entity) pipe.getHolder()).getPersistentID();
                tag.func_74778_a("uuid", uuid.toString());
            } else {
                throw new IllegalStateException("Held a pipe object that had an unknow holder! Make a request if this is needed for " + pipe
                        .getHolder().getClass());
            }
            return tag;
        }
    }

    public static class MovableEntity extends PipeConnection implements IConnection_BC8.MovableEntity {
        private final Entity entity;

        public MovableEntity(double length, Entity entity) {
            super(length, entity);
            this.entity = entity;
        }

        @Override
        public Entity getOther() {
            return entity;
        }

        @Override
        NBTTagCompound saveConnection() {
            NBTTagCompound tag = new NBTTagCompound();
            tag.func_74780_a("length", getLength());
            tag.func_74782_a("type", NBTUtils.writeEnum(PipeConnectionType.ENTITY));
            UUID uuid = entity.getPersistentID();
            tag.func_74778_a("uuid", uuid.toString());
            return tag;
        }
    }
}
