/** Copyright (c) 2011-2015, SpaceToad and the BuildCraft Team http://www.mod-buildcraft.com
 * <p/>
 * BuildCraft is distributed under the terms of the Minecraft Mod Public License 1.0, or MMPL. Please check the contents
 * of the license located in http://www.mod-buildcraft.com/MMPL-1.0.txt */
package buildcraft.transport.schematics;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.BlockPos;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumFacing.Axis;

import net.minecraftforge.common.util.Constants;

import buildcraft.api.blueprints.*;
import buildcraft.api.core.BCLog;
import buildcraft.api.statements.IStatement;
import buildcraft.api.statements.IStatementParameter;
import buildcraft.api.statements.StatementManager;
import buildcraft.transport.BlockGenericPipe;
import buildcraft.transport.Gate;
import buildcraft.transport.Pipe;
import buildcraft.transport.TileGenericPipe.SideProperties;

import net.minecraft.util.EnumFacing.Axis;
public class SchematicPipe extends SchematicTile {

    private BuildingPermission permission = BuildingPermission.ALL;

    @Override
    public boolean isAlreadyBuilt(IBuilderContext context, BlockPos pos) {
        Pipe<?> pipe = BlockGenericPipe.getPipe(context.world(), pos);

        if (BlockGenericPipe.isValid(pipe)) {
            if (tileNBT.hasKey("pipeId", Constants.NBT.TAG_INT)) {
                return pipe.item == Item.func_150899_d(tileNBT.getInteger("pipeId"));
            } else {
                return pipe.item == Item.func_111206_d(tileNBT.getString("pipeId"));
            }
        } else {
            return false;
        }
    }

    @Override
    public void rotateLeft(IBuilderContext context) {
        SideProperties props = new SideProperties();

        props.readFromNBT(tileNBT);
        props.rotateLeft();
        props.writeToNBT(tileNBT);

        Item pipeItem;
        if (tileNBT.hasKey("pipeId", Constants.NBT.TAG_INT)) {
            pipeItem = Item.func_150899_d(tileNBT.getInteger("pipeId"));
        } else {
            pipeItem = Item.func_111206_d(tileNBT.getString("pipeId"));
        }

        if (BptPipeExtension.contains(pipeItem)) {
            BptPipeExtension.get(pipeItem).rotateLeft(this, context);
        }

        if (tileNBT.hasKey("Gate")) {
            // This code is handling specifically BuildCraft 6.0 gates. Sided
            // gates starting BuildCraft 6.1 work differently.

            NBTTagCompound gateNBT = tileNBT.getCompoundTag("Gate");
            rotateGateLeft(gateNBT);
        } else {
            // Post 6.1 treatment

            NBTTagCompound[] gatesNBT = new NBTTagCompound[6];

            for (int i = 0; i < 6; ++i) {
                if (tileNBT.hasKey("Gate[" + i + "]")) {
                    gatesNBT[i] = tileNBT.getCompoundTag("Gate[" + i + "]");
                }
            }

            for (int i = 0; i < 6; ++i) {
                EnumFacing face = EnumFacing.field_82609_l[i];
                if (face.func_176740_k() != Axis.Y) {
                    face = face.func_176746_e();
                }

                int newI = face.ordinal();

                if (gatesNBT[i] != null) {
                    rotateGateLeft(gatesNBT[i]);
                    tileNBT.setTag("Gate[" + newI + "]", gatesNBT[i]);
                } else {
                    tileNBT.removeTag("Gate[" + newI + "]");
                }
            }
        }
    }

    private void rotateGateLeft(NBTTagCompound gateNBT) {
        for (int i = 0; i < Gate.MAX_STATEMENTS; ++i) {
            if (gateNBT.func_74764_b("trigger[" + i + "]")) {
                IStatement t = StatementManager.statements.get(gateNBT.func_74779_i("trigger[" + i + "]"));
                t = t.rotateLeft();
                gateNBT.func_74778_a("trigger[" + i + "]", t.getUniqueTag());
            }

            if (gateNBT.func_74764_b("action[" + i + "]")) {
                IStatement a = StatementManager.statements.get(gateNBT.func_74779_i("action[" + i + "]"));
                a = a.rotateLeft();
                gateNBT.func_74778_a("action[" + i + "]", a.getUniqueTag());
            }

            for (int j = 0; j < Gate.MAX_PARAMETERS; ++j) {
                if (gateNBT.func_74764_b("triggerParameters[" + i + "][" + j + "]")) {
                    NBTTagCompound cpt = gateNBT.func_74775_l("triggerParameters[" + i + "][" + j + "]");
                    IStatementParameter parameter = StatementManager.createParameter(cpt.func_74779_i("kind"));
                    parameter.readFromNBT(cpt);

                    parameter = parameter.rotateLeft();

                    parameter.writeToNBT(cpt);
                    gateNBT.func_74782_a("triggerParameters[" + i + "][" + j + "]", cpt);
                }

                if (gateNBT.func_74764_b("actionParameters[" + i + "][" + j + "]")) {
                    NBTTagCompound cpt = gateNBT.func_74775_l("actionParameters[" + i + "][" + j + "]");
                    IStatementParameter parameter = StatementManager.createParameter(cpt.func_74779_i("kind"));
                    parameter.readFromNBT(cpt);

                    parameter = parameter.rotateLeft();

                    parameter.writeToNBT(cpt);
                    gateNBT.func_74782_a("actionParameters[" + i + "][" + j + "]", cpt);
                }
            }
        }

        if (gateNBT.func_74764_b("direction")) {
            EnumFacing face = EnumFacing.field_82609_l[gateNBT.func_74762_e("direction")];
            if (face.func_176740_k() != Axis.Y) {
                face = face.func_176746_e();
            }

            gateNBT.func_74768_a("direction", face.ordinal());
        }
    }

    @Override
    public void placeInWorld(IBuilderContext context, BlockPos pos, List<ItemStack> stacks) {
        tileNBT.setInteger("x", pos.func_177958_n());
        tileNBT.setInteger("y", pos.func_177956_o());
        tileNBT.setInteger("z", pos.func_177952_p());

        context.world().setBlockState(pos, state, 3);

        TileEntity tile = context.world().getTileEntity(pos);
        tile.func_145834_a(context.world());
        tile.func_145839_a(tileNBT);
    }

    @Override
    public void initializeFromObjectAt(IBuilderContext context, BlockPos pos) {
        TileEntity tile = context.world().getTileEntity(pos);
        Pipe<?> pipe = BlockGenericPipe.getPipe(context.world(), pos);

        if (BlockGenericPipe.isValid(pipe)) {
            tile.func_145841_b(tileNBT);

            // remove all pipe contents

            tileNBT.removeTag("travelingEntities");

            for (EnumFacing direction : EnumFacing.values()) {
                tileNBT.removeTag("tank[" + direction.ordinal() + "]");
                tileNBT.removeTag("transferState[" + direction.ordinal() + "]");
            }

            for (int i = 0; i < 6; ++i) {
                tileNBT.removeTag("powerQuery[" + i + "]");
                tileNBT.removeTag("nextPowerQuery[" + i + "]");
                tileNBT.removeTag("internalPower[" + i + "]");
                tileNBT.removeTag("internalNextPower[" + i + "]");
            }
        }
    }

    @Override
    public void storeRequirements(IBuilderContext context, BlockPos pos) {
        Pipe<?> pipe = BlockGenericPipe.getPipe(context.world(), pos);

        if (BlockGenericPipe.isValid(pipe)) {
            List<ItemStack> items = new ArrayList<>(pipe.computeItemDrop());
            items.add(new ItemStack(pipe.item, 1, pipe.container.getItemMetadata()));
            storedRequirements = items.toArray(new ItemStack[items.size()]);
            BCLog.logger.info("Stored the requirements" + Arrays.toString(storedRequirements).replace(",", ",\n") + "@" + pos);
        } else {
            BCLog.logger.info("Tried and failed to find a pipe at " + pos);
        }
    }

    @Override
    public void postProcessing(IBuilderContext context, BlockPos pos) {
        Item pipeItem = Item.func_150899_d(tileNBT.getInteger("pipeId"));

        if (BptPipeExtension.contains(pipeItem)) {
            BptPipeExtension.get(pipeItem).postProcessing(this, context);
        }
    }

    @Override
    public BuildingStage getBuildStage() {
        return BuildingStage.STANDALONE;
    }

    @Override
    public void idsToBlueprint(MappingRegistry registry) {
        super.idsToBlueprint(registry);

        if (tileNBT.hasKey("pipeId", Constants.NBT.TAG_INT)) {
            Item item = Item.func_150899_d(tileNBT.getInteger("pipeId"));

            tileNBT.setInteger("pipeId", registry.getIdForItem(item));
        } else if (tileNBT.hasKey("pipeId")) {
            Item item = Item.func_111206_d(tileNBT.getString("pipeId"));

            tileNBT.setInteger("pipeId", registry.getIdForItem(item));
        }
    }

    @Override
    public void idsToWorld(MappingRegistry registry) {
        super.idsToWorld(registry);

        if (tileNBT.hasKey("pipeId")) {
            try {
                Item item = registry.getItemForId(tileNBT.getInteger("pipeId"));

                tileNBT.setString("pipeId", Item.field_150901_e.func_177774_c(item).toString());
            } catch (MappingNotFoundException e) {
                tileNBT.removeTag("pipeId");
            }
        }
    }

    @Override
    public void writeSchematicToNBT(NBTTagCompound nbt, MappingRegistry registry) {
        super.writeSchematicToNBT(nbt, registry);
        nbt.func_74768_a("version", 2);
    }

    @Override
    public void readSchematicFromNBT(NBTTagCompound nbt, MappingRegistry registry) {
        super.readSchematicFromNBT(nbt, registry);

        if (!nbt.func_74764_b("version") || nbt.func_74762_e("version") < 2) {
            // Schematics previous to the fixes in version 2 had item id
            // translation badly broken. We need to flush out information that
            // would be otherwise corrupted - that is the inventory (with the
            // old formalism "items") and gate parameters.
            tileNBT.removeTag("items");

            if (tileNBT.hasKey("Gate")) {
                NBTTagCompound gateNBT = tileNBT.getCompoundTag("Gate");

                for (int i = 0; i < 8; ++i) {
                    if (gateNBT.func_74764_b("triggerParameters[" + i + "]")) {
                        NBTTagCompound parameterNBT = gateNBT.func_74775_l("triggerParameters[" + i + "]");

                        if (parameterNBT.func_74764_b("stack")) {
                            parameterNBT.func_82580_o("stack");
                        }
                    }
                }
            }
        }
    }

    @Override
    public BuildingPermission getBuildingPermission() {
        return permission;
    }
}
